% INNERPROD.M
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% THE SCOPE OF THIS CODE: 
% This Matlab(R) function computes the canonical inner product of two 
% tangent vectors  X and Y at an arbitrary point on the unitary group U(n).
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% USAGE and OPTIONS:
% ip=innerprod(X,Y)
% 
% INPUT:
% X, Y = square matrices of the same size representing tangent vectors
%
% OUTPUT:
% ip = the canonical inner product of two tangent vectors X, Y at a point W
% on the unitary Lie group U(n)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COPYRIGHT and TERMS OF USE:
%
% This work was done at Aalto University, Espoo, Finland, during 2005-2008 
% together with Jan Eriksson and Visa Koivunen who are greatly acknowledged.
% This code should ONLY be used for educational and scientific purposes 
% (e.g. to be compared to other algorithms), and in non-commercial scopes.
% 
% These codes comes for free as they are, and the author does not assume 
% any responsibility for their usage.
% The authors support reproducible research and open software and therefore,
% they require their credits to be given. 
% In case these codes are used, please cite the corresponding papers as
% follows.
%
% When using the Conjugate Gradient (CG) algorithm:
%
% [1] T. Abrudan, J. Eriksson, V. Koivunen,
% "Conjugate Gradient Algorithm for Optimization Under Unitary Matrix Constraint", 
% Signal Processing, vol. 89, no. 9, Sep. 2009, pp. 1704-1714.
% PDF: http://www.sciencedirect.com/science?_ob=MImg&_imagekey=B6V18-4VXB6YG-1-F&_cdi=5668&_user=952938&_orig=search&_coverDate=09%2F30%2F2009&_sk=999109990&view=c&wchp=dGLzVlz-zSkWA&md5=2c71f0f9a116588d1c9d224a899fecec&ie=/sdarticle.pdf
% CITATION: http://signal.hut.fi/~tabrudan/ATE_papers/AbrEriKoi09SP.txt
%
% When using the Steepest Descent/Ascent (SD/SA) algorithms:
%
% [2] T. Abrudan, J. Eriksson, V. Koivunen;
% "Steepest Descent Algorithm for Optimization under Unitary Matrix Constraint",
% IEEE Transactions on Signal Processing, vol. 56, no. 3, Mar. 2008, pp. 1134-1147. 
% PDF: http://ieeexplore.ieee.org/iel5/78/4451275/04436033.pdf?tp=&arnumber=4436033&isnumber=4451275
% CITATION: http://signal.hut.fi/~tabrudan/ATE_papers/AbrEriKoi08TSP.txt 
%
% When using the polynomial-based or the DFT-based line search methods, 
% please cite [2].
%
% The codes were write by Traian Abrudan (C) 2007 
% Comments, questions and suggestions may be sent to tabrudan@signal.hut.fi
%                                             (or to abrudant@gmail.com)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function ip=innerprod(X,Y)
if size(X)==size(Y) % check is matrix sizes are correct
    ip=0.5*real(trace(X'*Y));
else
    ip=[]; % return empty matrix
    error('ERROR (INNEPROD.M): input matrices X, Y must have the same size')
end
