function fout = progbar(x,whichbar,varargin)

%PROGBAR Display a Progress bar.
%   H = PROGBAR(X,'title', property, value, property, value, ...)
%   creates and displays a progbar of fractional length X.  The
%   handle to the progbar figure is returned in H.
%   X should be between 0 and 1.  Optional arguments property and
%   value allow to set corresponding progbar figure properties.
%   Property can also be an action keyword 'CreateCancelBtn', in
%   which case a cancel button will be added to the figure, and
%   the passed value string will be executed upon clicking on the
%   cancel button or the close figure button.
%
%   PROGBAR(X) will set the length of the bar in the most recently
%   created progbar window to the fractional length X.
%
%   PROGBAR(X,H) will set the length of the bar in progbar H
%   to the fractional length X.
%
%   PROGBAR(X,H,'updated title') will update the title text in
%   the progbar figure, in addition to setting the fractional
%   length to X.
%
%   PROGBAR is typically used inside a FOR loop that performs a
%   lengthy computation.  A sample usage is shown below:
%
%       h = progbar(0,'Please wait...','name','Progress Bar');
%       for i=1:100,
%           computation here %
%           progbar(i/100,h)
%       end
%       close(h)
%
%
%   This program is an adaptation of the file waitbar.m written by
%   Mathworks and released in 2004. 
%   It contains most of code originally wirtten by Mathworks. I have 
%   just adapted it to make it more aesthetically pleasing.
%


if nargin>=2
    if ischar(whichbar) || iscellstr(whichbar)
        type=2; %we are initializing
        name=whichbar;
    elseif isnumeric(whichbar)
        type=1; %we are updating, given a handle
        f=whichbar;
    else
        error('MATLAB:waitbar:InvalidInputs', ['Input arguments of type ' class(whichbar) ' not valid.'])
    end
elseif nargin==1
    f = findobj(allchild(0),'flat','Tag','TMWWaitbar');

    if isempty(f)
        type=2;
        name='Waitbar';
    else
        type=1;
        f=f(1);
    end
else
    error('MATLAB:waitbar:InvalidArguments', 'Input arguments not valid.');
end

x = max(0,min(100*x,100));

switch type
case 1,  % waitbar(x)    update
        p = findobj(f,'Tag','PROGPATCH');
        b = findobj(f,'Tag','PROGBAR');
        if isempty(f) || isempty(p) || isempty(b),
            error('MATLAB:waitbar:WaitbarHandlesNotFound', 'Couldn''t find waitbar handles.');
        end

        xpatch = [x 100 100 x];
        
        set(p,'XData',xpatch)
%        xline = get(l,'XData');
%        set(l,'XData',xline);

name = get(f,'Name');
updte = [num2str(round(x)) '% '];
if ~isempty(name)
    str_i=strfind(name,'%');
    if str_i
        name(1:str_i+1) = [];
    end
    name = [updte name];
else
    name = updte;
end

set(f,'Name',name);

        if nargin>2,
            % Update waitbar title:
            hAxes = findobj(f,'type','axes');
            hTitle = get(hAxes,'title');
            set(hTitle,'string',varargin{1});
        end
        
case 2,  % waitbar(x,name)  initialize
        vertMargin = 0;
        if nargin > 2,
            % we have optional arguments: property-value pairs
            if rem (nargin, 2 ) ~= 0
                error('MATLAB:waitbar:InvalidOptionalArgsPass',  'Optional initialization arguments must be passed in pairs');
            end
        end

        oldRootUnits = get(0,'Units');

        set(0, 'Units', 'points');
        screenSize = get(0,'ScreenSize');

        axFontSize=get(0,'FactoryAxesFontSize');

        pointsPerPixel = 72/get(0,'ScreenPixelsPerInch');

        width = 360 * pointsPerPixel;
        height = 75 * pointsPerPixel;
        pos = [screenSize(3)/2-width/2 screenSize(4)/2-height/2 width height];

        f = figure(...
            'Units', 'points', ...
            'BusyAction', 'queue', ...
            'Position', pos, ...
            'Resize','off', ...
            'CreateFcn','', ...
            'NumberTitle','off', ...
            'IntegerHandle','off', ...
            'MenuBar', 'none', ...
            'Tag','TSprogbar',...
            'Interruptible', 'off', ...
            'WindowStyle', 'normal', ...
            'DockControls', 'off', ...
            'Visible','off',...
            'Color',[1 1 1]);%[0.9294    0.9529    0.9961]);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % set figure properties as passed to the fcn
        % pay special attention to the 'cancel' request
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        visValue = 'on';
        if nargin > 2,
            propList = varargin(1:2:end);
            valueList = varargin(2:2:end);
            cancelBtnCreated = 0;

            visibleExist = strmatch('vis',lower(propList));
            if ~isempty(visibleExist)
                visValue = valueList{visibleExist};
            end

            for ii = 1:length( propList )
                try
                    if strcmpi(propList{ii}, 'createcancelbtn' ) && ~cancelBtnCreated
                        cancelBtnHeight = 23 * pointsPerPixel;
                        cancelBtnWidth = 60 * pointsPerPixel;
                        newPos = pos;
                        vertMargin = vertMargin + cancelBtnHeight;
                        newPos(4) = newPos(4)+vertMargin;
                        callbackFcn = [valueList{ii}];
                        set( f, 'Position', newPos, 'CloseRequestFcn', callbackFcn );
                        cancelButt = uicontrol('Parent',f, ...
                            'Units','points', ...
                            'Callback',callbackFcn, ...
                            'ButtonDownFcn', callbackFcn, ...
                            'Enable','on', ...
                            'Interruptible','off', ...
                            'Position', [pos(3)-cancelBtnWidth*1.4, 7,  ...
                            cancelBtnWidth, cancelBtnHeight], ...
                            'String','Cancel', ...
                            'Tag','TMWWaitbarCancelButton'); %#ok<NASGU>
                        cancelBtnCreated = 1;
                    else
                        % simply set the prop/value pair of the figure
                        set( f, propList{ii}, valueList{ii});
                    end
                catch
                    disp ( ['Warning: could not set property ''' propList{ii} ''' with value ''' num2str(valueList{ii}) '''' ] );
                end
            end
        end

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

        amount = 128;
        
        %map = winter(amount);
        map1 = [ones(10,1),ones(10,1),ones(10,1)];
        map2 = [linspace(0,1,118);linspace(0,1,118);linspace(1,1,118)]';
        map = [map2;map1];
       
        colormap(map(end:-1:1,:));     

        axNorm=[.05 .3 .9 .2];
        axPos=axNorm.*[pos(3:4),pos(3:4)] + [0 vertMargin 0 0];
        
        h = axes('XLim',[0 100],...
            'YLim',[0 1],...
            'Box','on', ...
            'Units','Points',...
            'FontSize', axFontSize,...
            'Position',axPos,...
            'XTickMode','manual',...
            'YTickMode','manual',...
            'XTick',[],...
            'YTick',[],...
            'XTickLabelMode','manual',...
            'XTickLabel',[],...
            'YTickLabelMode','manual',...
            'YTickLabel',[]);

        %tHandle=title(name);
        tHandle=get(h,'title');
        oldTitleUnits=get(tHandle,'Units');
        set(tHandle,...
            'Units',      'points',...
            'String',     name);

        tExtent=get(tHandle,'Extent');
        set(tHandle,'Units',oldTitleUnits);

        titleHeight=tExtent(4)+axPos(2)+axPos(4)+5;
        if titleHeight>pos(4)
            pos(4)=titleHeight;
            pos(2)=screenSize(4)/2-pos(4)/2;
            figPosDirty=true;
        else
            figPosDirty=false;
        end

        if tExtent(3)>pos(3)*1.10;
            pos(3)=min(tExtent(3)*1.10,screenSize(3));
            pos(1)=screenSize(3)/2-pos(3)/2;
            axPos([1,3])=axNorm([1,3])*pos(3);
            set(h,'Position',axPos);

            figPosDirty=true;
        end

        if figPosDirty
            set(f,'Position',pos);
        end
      
        %%% Plot the initial values        
        hold(h,'on');
        
        % plot Bar    
        
        b=barh(h,0,100,2,...
            'erasemode','none',...
            'edgecolor','none',...
            'Tag','PROGBAR',...
            'showbaseline','off');
        
        shading(h,'interp');
        
        ch = get(b,'Children');
        
        fvd = get(ch,'Faces');
        fvcd = get(ch,'FaceVertexCData');
        
        fvcd(fvd(1)) = 1;       % Color base vertices 1st index
        fvcd(fvd(4)) = 1;    
        fvcd(fvd(2)) = amount;   % Assign top vertices color
        fvcd(fvd(3)) = amount;
        
        set(ch,'FaceVertexCData',fvcd);
        
        % plot patch
        
        ypatch = [ 0 0  1  1];
        xpatch = [ x 100 100 x];
        
        ph=patch(xpatch,ypatch,[1 1 1],...
            'edgecolor',[1 1 1],...
            'Tag','PROGPATCH');                                                
        %
        
        hold(h,'off');
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        

        set(f,'HandleVisibility','callback','visible', visValue);

        set(0, 'Units', oldRootUnits);
end  % case
drawnow;

if nargout==1,
    fout = f;
end    

