% University of Alberta 
% Mahdi Shaghaghi
% 31 March 2014

% DOA_method options:
%      'root-MUSIC' 
%      'root-swap-root-MUSIC'
%      'unitary-root-MUSIC'; 
%      'root-swap-unitary-root-MUSIC'

% [ref1] Improved unitary root-MUSIC for DOA estimation based on pseudo-noise
% resampling, C. Qian, Feb 2014.

% Reliability test is done using conventional beamformer

% Updated: 20 Apr 2014: [U20Apr14]: gamma * ( U + U' ) is subtracted from
% the resampled covariance matrix.

function [fhat_rad] = pseudo_noise_resample(x0, R0, Q_M, K, d_lambda, ...
    p, q, ML_method, Ppnr, sigma2_n, f_CBF_left, f_CBF_right, DOA_method, gamma, U)

% x0 is a M*N data matrix measureed at N time instances by an array with M elements. 
% R0 is the initial sample covariance matrix used for the ML objective.
% Q_M is a matrix given in [ref1] used for unitary-root-MUSIC method.
% K is the number of sources.
% d_lambda is the interelement spacing over the wavelength.
% p, q, and ML_method are used in root-swap methods.
% Ppnr is the pseudo-noise resampling numbers.
% sigma2_n is used for pseudo-noise resampling.
% f_CBF_left and f_CBF_right determined the intervals for the reliability
% test.


[M, N] = size(x0);
sigma_n = sqrt(sigma2_n);

fhat_rad_bank = zeros(K, Ppnr);
mag_bank = zeros(K, Ppnr);

reliability = zeros(1, Ppnr);

for P_iter = 1:Ppnr
    
    pseudo_noise = sigma_n * randn(M,N); 
    x0_r = x0 + pseudo_noise; % Resampled data
    R_r = x0_r * (x0_r') / N; % Resampled covariance matrix
    
    R_r = R_r - gamma * ( U + U' ); % [U20Apr14]
    
    % Compute eigendecomposition and order by ascending eigenvalues
    
    [Q0, D0] = eig(R_r);
    [~, idx] = sort(abs(diag(D0))); % Ascending order: lambdahat1 <= lambdahat2 <= ...
    Q0 = Q0(:, idx);
    G = Q0(:, 1:(M-K));
    
    % root-MUSIC
        
    if (strcmp(DOA_method, 'root-MUSIC'))
        [fhat_rad_bank(:, P_iter), mag_bank(:, P_iter)] = root_music_doa(G, M, K, d_lambda); % The output is in [-pi/2,pi/2] range
    elseif (strcmp(DOA_method, 'root-swap-root-MUSIC'))
        [fhat_rad_bank(:, P_iter), mag_bank(:, P_iter)] = root_swap_root_music_doa(R0, G, M, K, d_lambda, p, q, ML_method); 
    elseif (strcmp(DOA_method, 'unitary-root-MUSIC'))
        [fhat_rad_bank(:, P_iter), ~, mag_bank(:, P_iter)] = unitary_root_music_doa(R_r, Q_M, M, K, d_lambda); 
    elseif (strcmp(DOA_method, 'root-swap-unitary-root-MUSIC'))
        [fhat_rad_bank(:, P_iter), ~, mag_bank(:, P_iter)] = root_swap_unitary_root_music_doa(R0, R_r, Q_M, M, K, d_lambda, p, q, ML_method); 
    end
    
    reliability(P_iter) = reliability_test_beamforming(fhat_rad_bank(:, P_iter), f_CBF_left, f_CBF_right);
    
end


if (sum(reliability) == 0)
    [~, idx] = max(mag_bank, [], 2); % Finding the index of the max element in each row.
    fhat_rad = zeros(K,1);
    for i = 1:K
        fhat_rad(i) = fhat_rad_bank(i, idx(i));
    end 
else
    fhat_rad = mean(fhat_rad_bank(:, reliability == 1), 2);
end




















