% University of Alberta 
% Mahdi Shaghaghi
% 15 January 2014

% Updated: 17 Mar 2014: [U17Mar14]: Stochastic ML objectice function is
% also considered. ref: eq (2.20), Stoica, 1990, Performance Study of 
% Conditional and Unconditional Direction-of-Arrival Estimation

% Updated: 3 Apr 2014 [U3Apr14]: magnitude of roots are also returned. 

function [fhatroot_Music_rad, maghatroot_Music_rad] = root_swap_root_music_doa(R0, G, M, K, d_lambda, p, q, ML_method)
% R0 is the initial sample covariance matrix used for the ML objective.
% G columns are the noise eigen vectors.
% M is the number of array elements.
% K is the number of sources
% d_lambda is the interelement spacing over the wavelength.

% We have M - 1 roots in the unit circle sorted in a way that the first
% root is the closest to the unit circle.

% We want to pick K numbber of the roots.
% The first p roots are picked (p < K).
% The last q roots are discarded.
% Now the problem is to choose K - p roots from roots p + 1 to M - 1 - q. 
% The set of the roots with the minimum ML objective is picked. 

Ar = zeros(2*M-1,1);
for i = 1:(M-K)
   Ar = Ar + conv(G(M:-1:1,i),conj(G(:,i))); 
end
fest = zeros(M - 1, 1);
mag = zeros(M - 1, 1); % [U3Apr14]
r_A = roots(Ar);

distinct_roots_num = 0;
unit_roots_indx = find(abs(r_A) == 1);
unit_roots_num = length(unit_roots_indx);
if (unit_roots_num~=0)
    unit_roots = r_A(unit_roots_indx);
    distinct_roots_indx = 1:2:unit_roots_num;
    distinct_roots_num = length(distinct_roots_indx);
    fest(1:distinct_roots_num) = angle(unit_roots(distinct_roots_indx));
    mag(1:distinct_roots_num) = 1; % [U3Apr14]
end
[i_min] = find(abs(r_A) < 1);
r_A_min = r_A(i_min);
freqsroot_MUSIC = angle(r_A_min);
[~,index] = sort(abs((abs(r_A_min) -1)));
fest((distinct_roots_num+1) : (M-1)) = freqsroot_MUSIC(index);  % Sorted angles based on closeness to unit circle
mag((distinct_roots_num+1) : (M-1)) = abs(r_A_min(index));  % [U3Apr14]

fest_rad = asin(fest/(2*pi*d_lambda));
idx_combinations = VChooseK((p + 1) : (M - 1 - q), K - p);

n = size(idx_combinations,1); % number of combinations.

idxR = (0 : (M - 1))';
Ahat = zeros(M, K);
ML_objective = zeros(n, 1);

for i = 1:n
    idx_tmp = [1:p, idx_combinations(i,:)];
    
    for k = 1:K
        Ahat(:,k) = exp(-1i*2*pi*d_lambda*sin(fest_rad(idx_tmp(k)))*idxR);
    end

    Ahat = Ahat/sqrt(M); % So that each column of Ahat has unit norm.
    PAhat = Ahat*((Ahat')*Ahat)^(-1)*(Ahat');
    PA_orthhat = eye(M,M) - PAhat;    
    
    if (strcmp(ML_method, 'Stochastic')) % [U17Mar14]
        ML_objective(i) = log( real( det( PAhat * R0 * PAhat + ...
            trace( PA_orthhat * R0 ) * PA_orthhat / (M-K) ) ) );
    elseif (strcmp(ML_method, 'Deterministic'))
        ML_objective(i) = real( -trace( PAhat * R0 ) );  
    end    
    
end

[~,idx] = min(ML_objective);
idx_tmp = [1:p, idx_combinations(idx,:)];

maghatroot_Music_rad = mag(idx_tmp); % [U3Apr14]

[fhatroot_Music_rad, idx] = sort(fest_rad(idx_tmp));

maghatroot_Music_rad = maghatroot_Music_rad(idx); % [U3Apr14]



