/**
 * 3 Band EQ based on Shannon interpolation and
 * continuous-time filtering concept.
 *
 * Copyright (C) 2010 Simo Srkk (STS)
 *
 * $Id: sts3bandeq_core.h,v 1.7 2010/01/09 21:33:06 ssarkka Exp $
 */

#ifndef __sts3bandeq_core__
#define __sts3bandeq_core__

#include <cstdio>
#include <cmath>

/**
 * Simple delay line (a kind of circular buffer) with COUNT elements
 */
template<int COUNT> class Sts3DelayLine {
public:
    /**
     * Constructor
     */
    Sts3DelayLine() : m_index(0) {
	for (int i = 0; i < COUNT; i++) {
	    m_elem[i] = 0.0;
	}
    }

    /**
     * Zero the delay line.
     */
    void reset() {
	m_index = 0;
	for (int i = 0; i < COUNT; i++) {
	    m_elem[i] = 0.0;
	}
    }

    /**
     * Put new sample in
     *
     * @param x Sample
     */
    void push(double x) {
	m_elem[m_index++] = x;
	if (m_index >= COUNT) m_index = 0;
    }

    /**
     * Return the sample delayed by n steps.
     * n must be on range [1,COUNT]. The last pushed
     * sample is assumed to have delay 1.
     *
     * @param n Number of delay steps.
     */
    double getDelayed(int n) const {
	int ind = m_index - n;
	if (ind < 0)
	    return m_elem[COUNT + ind];
	else
	    return m_elem[ind];
    }

    /**
     * Print all variables and elements to stdout.
     * Note that this is included into executable only
     * if symbol STS3BANDEQ_PRINT is defined.
     */
#ifdef STS3BANDEQ_PRINT
    void print(bool print_beginend = true) const {
	if (print_beginend) {
	    printf("BEGIN DELAYLINE(%d)\n",COUNT);
	}
	printf("  dl_index = %d\n", m_index);
	printf("  dl_elem = [");
	for (int i = 0; i < COUNT; i++) {
	    if (i == m_index) {
		printf("(%g) ", m_elem[i]);
	    }
	    else {
		printf("%g ", m_elem[i]);
	    }
	}
	printf("]\n");
	if (print_beginend) {
	    printf("END\n");
	}
    }
#endif

private:
    int    m_index;       // Index to next element to be filled
    double m_elem[COUNT]; // Delayed elements

};


/**
 * Filter section types
 */
enum {
    SECTYPE_UNKNOWN,
    SECTYPE_LOWSHELF,
    SECTYPE_PEAK,
    SECTYPE_HIGHSHELF
};

/**
 * Input reconstruction approximation
 * order parameter. I.e., 2*INPUT_COUNT+1
 * input samples are used for reconstructing
 * the input.
 */
#define INPUT_COUNT (10)

/**
 * Number of Simpson integration steps
 */
#define SIMPSON_STEPS (30)

/**
 * Masks for printing routine
 */
#define PRINTMASK_PARAM 0x0001
#define PRINTMASK_STATE 0x0002
#define PRINTMASK_INPUT 0x0004

#define PRINTMASK_ALL (-1)

/**
 * Single simple EQ section, which approximates
 * arbitrary second order analog filters. 
 */
class Sts2ndSection {
public:

    /**
     * Constructor. 
     */
    Sts2ndSection()
	: m_type(SECTYPE_UNKNOWN), m_T(1.0/44100.0), m_q(1), m_f(100), m_db(0),
  	  m_x1(0.0), m_x2(0.0) {}

    /**
     * Constructor. Computes the parameters also.
     *
     * @param fs Sampling frequency
     * @param q  Quality parameter
     * @param f  Frequency in Hz
     * @param db Level in decibels
     * @param type Type as constant SECTYPE_xxx
     */
    Sts2ndSection(double fs, double q, double f, double db, int type)
	: m_type(type), m_T(1/fs), m_q(q), m_f(f), m_db(db),
	m_x1(0.0), m_x2(0.0)
    {
	recalculate();
    }

    /**
     * Reset (zero) the filter section state.
     */
    void reset() {
	m_x1 = 0;
	m_x2 = 0;
	m_delayline.reset();
    }

    /**
     * Run the input through the filter and return the next filter
     * output. The calculation is done assuing that input has been
     * sample at time step k-1 and the output is for time step k.
     *
     * @param input Input sample
     * @return Output of filter
     */
    double process(double input) {
	double output;

	// Push input to delay line
	m_delayline.push(input);
	
	// Compute X = A*X
	double tmp1, tmp2;
	tmp1 = m_ta11 * m_x1 + m_ta12 * m_x2;
        tmp2 = m_ta21 * m_x1 + m_ta22 * m_x2;
	m_x1 = tmp1;
	m_x2 = tmp2;

	// Add terms B_j u[-j-1]
	for (int j = 0; j < 2*INPUT_COUNT+1; j++) {
	    double u = m_delayline.getDelayed(j+1);
	    m_x1 += m_bl1[j] * u;
	    m_x2 += m_bl2[j] * u;
	}

	// Output is y = x1 + c*u[-n]
	output = m_x1 + m_c * m_delayline.getDelayed(INPUT_COUNT);

	return output;
    }

    /**
     * Set sampling frequency (typically 44100 or 48000)
     *
     * @param fs Sampling frequency in Hz
     */
    void setSamplingFrequency(double fs) { m_T = 1/fs; recalculate(); }

    /**
     * Get sampling frequency
     *
     * @return Sampling frequency in Hz
     */
    double getSamplingFrequency() const { return 1/m_T; }

    /**
     * Set quality factor, which should be in range [0.1,10].
     *
     * @param q Quality factor
     */
    void setQuality(double q) { m_q  = q; recalculate(); }

    /**
     * Get quality factor
     *
     * @return Quality factor
     */
    double getQuality() const { return m_q; }

    /**
     * Set the frequency, which should be in range (0,fs/2).
     *
     * @param f Frequency in Hz
     */
    void setFrequency(double f) { m_f  = f; recalculate(); }

    /**
     * Get the frequency
     *
     * @return Frequency in Hz
     */
    double getFrequency() { return m_f; }

    /**
     * Set the level, which should be in range [-20,20].
     *
     * @param db Level in decibels
     */
    void setLevel(double db) { m_db = db; recalculate(); }

    /**
     * Get the level
     *
     * @return Level in decibels
     */
    double getLevel() { return m_db; }

    /**
     * Set the filter type
     *
     * @param type Type of the filter
     */
    void setType(int type) { m_type = type; recalculate(); }

    /**
     * Get the filter type
     *
     * @return Type of the filter
     */
    int getType() const { return m_type; }

    /**
     * Print all the filter parameters to stdout.
     * Note that this is included into executable only
     * if symbol STS3BANDEQ_PRINT is defined.
     */
#ifdef STS3BANDEQ_PRINT
    void print(int mask = PRINTMASK_ALL) const;
#endif

private:
    /**
     * Evaluate matrix exponential expm(t*w0*[-a1 1; -a2 0])
     * and store the results to m_ta11, m_ta12, m_ta21, m_ta22.
     * The other values than t are taken from instance variables.
     * 
     * @param t The time parameter
     */
    void eval_expm(double t);

    /**
     * Recalculate all parameters of the filter section.
     * This should be called always when parameters change.
     */
    void recalculate();

    int  m_type; // Type of the filter
    double m_T;  // Sampling time
    double m_q;  // Quality
    double m_f;  // Frequency
    double m_db; // Level in decibels

    double m_x1; // First state component
    double m_x2; // Second state component
    Sts3DelayLine<2*INPUT_COUNT+1> m_delayline; // Delay line of inputs

    double m_a1, m_a2, m_b1, m_b2; // Analog transfer function parameters
    double m_c; // Input feedforward coefficient

    double m_ta11, m_ta12,
	   m_ta21, m_ta22; // Transition matrix

    double m_bl1[2*INPUT_COUNT+1]; // Elements 1 in input coefficient matrices
    double m_bl2[2*INPUT_COUNT+1]; // Elements 2 in input coefficient matrices

};

#define DEFAULT_BURNIN    20
#define DEFAULT_NFADE     100
#define CHANGE_QUEUE_SIZE 100

enum {
    FSTATE_DONE,
    FSTATE_BURNIN,
    FSTATE_FADE
};

/**
 * 2nd section derivative, which is able to
 * fade between two filters at parameter change
 * to get rid of popping sounds. Note that in
 * proper object oriented style we should
 * declare an interface to the simple 2nd section
 * and implement here a generic fader. However we
 * would loose all the compiler inlining that way.
 */
class DepoppedSts2ndSection {
public:
    /**
     * Constructor
     *
     * @param fs     Sampling frequency
     * @param q      Quality parameter
     * @param f      Frequency in Hz
     * @param db     Level in decibels
     * @param type   Type as constant SECTYPE_xxx
     * @param burnin Number of burn-in samples
     * @param nfade  Number of fading samples
     */
    DepoppedSts2ndSection(double fs, double q, double f, double db, int type,
			  int burnin = DEFAULT_BURNIN,
			  int nfade  = DEFAULT_NFADE)
	: m_burnin(burnin), m_nfade(nfade),
	  m_active(0), m_state(FSTATE_DONE), m_fadecount(0),
	  m_last_q(q), m_last_f(f), m_last_db(db),
	  m_queue_head(0), m_queue_tail(0)
    {
	m_2ndsec[0] = Sts2ndSection(fs,q,f,db,type);
	m_2ndsec[1] = Sts2ndSection(fs,q,f,db,type);
    }

    /**
     * Reset (zero) the filter section state.
     */
    void reset() {
	m_active = 0;
	m_state = FSTATE_DONE;
	m_fadecount = 0;
	m_queue_head = 0;
	m_queue_tail = 0;
	m_2ndsec[0].reset();
	m_2ndsec[1].reset();
    }

    /**
     * Run the input through the filter and return the next filter
     * output. The calculation is done assuing that input has been
     * sample at time step k-1 and the output is for time step k.
     *
     * On parameter changes we run two parallel filters and fade
     * between them to eliminate popping sounds.
     *
     * @param input Input sample
     * @return Output of filter
     */
    double process(double input) {
	double output = 0;

	// Index of the non-active equalizer
	int other = (m_active == 0) ? 1 : 0;

	/*
	 * The normal state when there are
	 * no delays going on.
	 */
	if (m_state == FSTATE_DONE) {
	    //
	    // Check if parameter change is queued
	    //
	    if (m_queue_head != m_queue_tail) {
		// Something in the queue
		double q  = m_q_queue[m_queue_head];
		double f  = m_f_queue[m_queue_head];
		double db = m_db_queue[m_queue_head];
		m_queue_head = (m_queue_head + 1) % CHANGE_QUEUE_SIZE;

		// Initialize the secondary EQ
		double fs   = m_2ndsec[m_active].getSamplingFrequency();
		int type = m_2ndsec[m_active].getType();
		m_2ndsec[other] = Sts2ndSection(fs,q,f,db,type);

		m_state = FSTATE_BURNIN; // Start burnin
		m_fadecount = 0;
	    }

	    // Simply run the active equalizer
	    output = m_2ndsec[m_active].process(input);
	}

	/*
	 * We are on burnin period of fading
	 */
	else if (m_state == FSTATE_BURNIN) {
	    // Run both the equalizers
	    double output0 = m_2ndsec[m_active].process(input);
	    double output1 = m_2ndsec[other].process(input);

	    output1 = output1; // Get rid of compiler warning
	    output = output0; // Discard the second for now

	    if (++m_fadecount >= m_burnin) {
		m_fadecount = 0;
		m_state = FSTATE_FADE; // Start fading		
	    }
	}

	/*
	 * We are on actual period of fading
	 */
	else if (m_state == FSTATE_FADE) {
	    // Run both the equalizers
	    double output0 = m_2ndsec[m_active].process(input);
	    double output1 = m_2ndsec[other].process(input);

	    // Do simple cosine based interpolation
	    double alpha = 0.5 + 0.5 * cos(M_PI * m_fadecount / m_nfade);
	    output = alpha * output0 + (1 - alpha) * output1;

	    if (++m_fadecount >= m_nfade) {
		m_fadecount = 0;
		m_state = FSTATE_DONE; // Fading done
		m_active = other;      // Change the active equalizer
	    }
	}

	/*
	 * We should not come here
	 */
	else {
	    m_state = FSTATE_DONE;
	}

	return output;
    }

    /**
     * Set sampling frequency (typically 44100 or 48000)
     *
     * @param fs Sampling frequency in Hz
     */
    void setSamplingFrequency(double fs) {
	reset();
	m_2ndsec[m_active].setSamplingFrequency(fs);
    }

    /**
     * Get sampling frequency
     *
     * @return Sampling frequency in Hz
     */
    double getSamplingFrequency() const {
	return m_2ndsec[m_active].getSamplingFrequency();
    }

    /**
     * Set quality factor, which should be in range [0.1,10].
     *
     * @param q         Quality factor
     * @param no_fading Directly set the active filter (default false)
     */
    void setQuality(double q, bool no_fading = false) {
	m_last_q = q;

	if (no_fading) {
	    m_2ndsec[m_active].setQuality(q);
	}
	else {
	    // Push paramaters into queue
	    m_q_queue[m_queue_tail] = m_last_q;
	    m_f_queue[m_queue_tail] = m_last_f;
	    m_db_queue[m_queue_tail] = m_last_db;
	    m_queue_tail = (m_queue_tail + 1) % CHANGE_QUEUE_SIZE;
	}
    }

    /**
     * Get quality factor
     *
     * @return Quality factor
     */
    double getQuality() const { return m_last_q; }

    /**
     * Set the frequency, which should be in range (0,fs/2).
     *
     * @param f         Frequency in Hz
     * @param no_fading Directly set the active filter (default false)
     */
    void setFrequency(double f, bool no_fading = false) {
    	m_last_f = f;

	if (no_fading) {
	    m_2ndsec[m_active].setFrequency(f);
	}
	else {
	    // Push paramaters into queue
	    m_q_queue[m_queue_tail] = m_last_q;
	    m_f_queue[m_queue_tail] = m_last_f;
	    m_db_queue[m_queue_tail] = m_last_db;
	    m_queue_tail = (m_queue_tail + 1) % CHANGE_QUEUE_SIZE;
	}
    }

    /**
     * Get the frequency
     *
     * @return Frequency in Hz
     */
    double getFrequency() { return m_last_f; }

    /**
     * Set the level, which should be in range [-20,20].
     *
     * @param db        Level in decibels
     * @param no_fading Directly set the active filter (default false)
     */
    void setLevel(double db, bool no_fading = false) {
    	m_last_db = db;

	if (no_fading) {
	    m_2ndsec[m_active].setLevel(db);
	}
	else {
	    // Push paramaters into queue
	    m_q_queue[m_queue_tail] = m_last_q;
	    m_f_queue[m_queue_tail] = m_last_f;
	    m_db_queue[m_queue_tail] = m_last_db;
	    m_queue_tail = (m_queue_tail + 1) % CHANGE_QUEUE_SIZE;
	}
    }

    /**
     * Get the level
     *
     * @return Level in decibels
     */
    double getLevel() { return m_last_db; }

    /**
     * Set the filter type
     *
     * @param type Type of the filter
     */
    void setType(int type) {
	reset();
	m_2ndsec[m_active].setType(type);
    }

    /**
     * Get the filter type
     *
     * @return Type of the filter
     */
    int getType() const { return m_2ndsec[m_active].getType(); }

    /**
     * Print all the filter parameters to stdout.
     * Note that this is included into executable only
     * if symbol STS3BANDEQ_PRINT is defined.
     */
#ifdef STS3BANDEQ_PRINT
    void print(int mask = PRINTMASK_ALL) const {
	m_2ndsec[m_active].print(mask);
    }
#endif


private:
    Sts2ndSection m_2ndsec[2];
    int m_burnin;    // Burn-in samples
    int m_nfade;     // Fading samples
    int m_active;    // Active index
    int m_state;     // FSTATE_xxx
    int m_fadecount; // The current fade count

    // Last set parameters (we might still be fading to them)
    double m_last_q;
    double m_last_f;
    double m_last_db;
    
    // Queues
    int m_queue_head; // Next to come out
    int m_queue_tail; // Next empty place
    double m_q_queue[CHANGE_QUEUE_SIZE];
    double m_f_queue[CHANGE_QUEUE_SIZE];
    double m_db_queue[CHANGE_QUEUE_SIZE];
};

/**
 * Default EQ parameters (borrowed from Digirack EQ III)
 */
#define DEFAULT_LF_Q  1.0
#define DEFAULT_LF_F  100.0
#define DEFAULT_LF_DB 0.0

#define DEFAULT_LMF_Q  1.0
#define DEFAULT_LMF_F  200.0
#define DEFAULT_LMF_DB 0.0

#define DEFAULT_MF_Q  1.0
#define DEFAULT_MF_F  1000.0
#define DEFAULT_MF_DB 0.0

#define DEFAULT_HMF_Q  1.0
#define DEFAULT_HMF_F  2000.0
#define DEFAULT_HMF_DB 0.0

#define DEFAULT_HF_Q  1.0
#define DEFAULT_HF_F  6000.0
#define DEFAULT_HF_DB 0.0

class Sts3BandEQ {
public:
    /**
     * Constructor.
     *
     * @param fs Sampling frequency
     */
    Sts3BandEQ(double fs,
	       int burnin = DEFAULT_BURNIN,
	       int nfade  = DEFAULT_NFADE)
	: m_lf(fs,DEFAULT_LF_Q,DEFAULT_LF_F,DEFAULT_LF_DB,
	       SECTYPE_LOWSHELF,burnin,nfade),
	  m_lmf(fs,DEFAULT_LMF_Q,DEFAULT_LMF_F,DEFAULT_LMF_DB,
		SECTYPE_PEAK,burnin,nfade),
	  m_mf(fs,DEFAULT_MF_Q,DEFAULT_MF_F,DEFAULT_MF_DB,
	       SECTYPE_PEAK,burnin,nfade),
	  m_hmf(fs,DEFAULT_HMF_Q,DEFAULT_HMF_F,DEFAULT_HMF_DB,
		SECTYPE_PEAK,burnin,nfade),
	  m_hf(fs,DEFAULT_HF_Q,DEFAULT_HF_F,DEFAULT_HF_DB,
	       SECTYPE_HIGHSHELF,burnin,nfade) {}

    /**
     * Return reference to the low filter section.
     *
     * @return Reference to the section
     */
    DepoppedSts2ndSection &getLF()  { return m_lf;  }

    /**
     * Return reference to the low middle filter section.
     *
     * @return Reference to the section
     */
    DepoppedSts2ndSection &getLMF() { return m_lmf; }

    /**
     * Return reference to the middle filter section.
     *
     * @return Reference to the section
     */
    DepoppedSts2ndSection &getMF()  { return m_mf;  }

    /**
     * Return reference to the high middle filter section.
     *
     * @return Reference to the section
     */
    DepoppedSts2ndSection &getHMF() { return m_hmf; }

    /**
     * Return reference to the high filter section.
     *
     * @return Reference to the section
     */
    DepoppedSts2ndSection &getHF()  { return m_hf;  }

    /**
     * Reset (zero) the equalizer state.
     */
    void reset() {
	m_lf.reset();
	m_lmf.reset();
	m_mf.reset();
	m_hmf.reset();
	m_hf.reset();
    }

    /**
     * Run the input through all the filter section and
     * return the output. 
     *
     * @param input Input sample
     * @return Output of equalizer
     */
    double process(double input) {
	double y = input;
	y = m_lf.process(y);
	y = m_lmf.process(y);
	y = m_mf.process(y);
	y = m_hmf.process(y);
	y = m_hf.process(y);
	return y;
    }

    /**
     * Set sampling frequency (typically 44100 or 48000)
     *
     * @param fs Sampling frequency in Hz
     */
    void setSamplingFrequency(double fs) {
	m_lf.setSamplingFrequency(fs);
	m_lmf.setSamplingFrequency(fs);
	m_mf.setSamplingFrequency(fs);
	m_hmf.setSamplingFrequency(fs);
	m_hf.setSamplingFrequency(fs);
    }

    /**
     * Get sampling frequency
     *
     * @return Sampling frequency in Hz
     */
    double getSamplingFrequency() const
    { return m_lf.getSamplingFrequency(); }

private:
    DepoppedSts2ndSection m_lf;  // Low filter section
    DepoppedSts2ndSection m_lmf; // Low middle filter section
    DepoppedSts2ndSection m_mf;  // Middle filter section
    DepoppedSts2ndSection m_hmf; // High middle filter section
    DepoppedSts2ndSection m_hf;  // High filter section
};

#endif

