
(* 2017 (C) Jussi Rintanen *)

signature XDIMACS =
  sig
      type graph = int * (int * int * int) list
      type instance = int                      (* node count *)
		      * graph option           (* the graph *)
		      * bool                   (* acyclicity required? *)
		      * (int * int * lit) list (* s,t,v reachable *)
		      * (int * int * lit) list (* s,t,v non-reachable *)
		      * lit list list          (* clauses *)
		      * (int * string) list    (* symbol table *)
      val extract : entry list -> instance
      val output : TextIO.outstream * instance * bool -> unit
  end;

structure xDIMACS : XDIMACS =
  struct
    type graph = int * (int * int * int) list
    type instance = int * graph option * bool * (int * int * lit) list * (int * int * lit) list * lit list list * (int * string) list

    fun test(true,_) = ()
      | test(false,s) = ERROR s

    fun extract entries =
      let
	  fun getGraph (nodecnt,nodes,arcs) = (nodecnt,arcs)
	  val graph0 = fold (fn (Cgraph g,_) => SOME g | (_,ac) => ac) entries NONE
	  val graph = inOption getGraph graph0
	  val pcnf = fold (fn (Pcnf(m,c),ac) => SOME(m,c) | (_,ac) => ac) entries NONE
	  val (maxvar,clausecnt) = (case pcnf of
					(SOME mc) => mc
				      | NONE => ERROR "'p cnf' missing")
	  val symtab = fold (fn (Csymtab(i,s),ac) => (i-1,s)::ac | (_,ac) => ac) entries []
	  val clauses = fold (fn (Clause c,ac) => c::ac | (_,ac) => ac) entries []
	  val nonreacha = fold (fn (Cnonreach(cnt,non),ac) => (test(cnt=(length non),"wrong number of entries in gnonreach");
							       ac@non)
			       | (_,ac) => ac) entries []
	  val reacha = fold (fn (Creachable(s,cnt,tvlist),ac) => (test(cnt=(length tvlist),"wrong number of entries in greachable for "^(Int.toString s));
								  (map (fn (t,v) => (s,t,v)) tvlist)@ac)
			    | (_,ac) => ac) entries []
      in
	  (maxvar,graph,false,reacha,nonreacha,clauses,symtab)
      end

    fun output (fo,(varcnt,gro,acyc,reachables,nonreachables,clauses,symtab),smbolic) =
      let fun out s = TextIO.output(fo,s)
	  fun svar2str i = (assoc(i,symtab)) handle Match => (app print ["Assoc ",Int.toString i];raise Match)
	  fun lit2str (Pos n) = Int.toString(n+1)
	    | lit2str (Neg n) = "-"^(Int.toString(n+1))
	  fun slit2str (Pos n) = svar2str n
	    | slit2str (Neg n) = "-"^(svar2str n)
	  fun var2str v = Int.toString(v+1)
	  fun outclause c = app out [String.concatWith " " (map lit2str c)," 0\n"]
	  fun soutclause c = app out ["[ ",String.concatWith " " (map slit2str c)," ]\n"]
	  fun outreachables [] = ()
	    | outreachables _ = ()
	  fun outnonreachables [] = ()
	    | outnonreachables cs =
	    let val a = 1
	    in
		app out ["g nonreach ",Int.toString(length cs)];
		app (fn (s,t,v) => app out [" ",
					    Int.toString s," ",
					    Int.toString t," ",
					    lit2str v]) cs;
		out "\n"
	    end
      in
	  app out ["p cnf ",Int.toString varcnt," ",Int.toString(length clauses),"\n"];
	  (case gro of
	      (SOME (nodecnt,arcs)) =>
	      (app out ["c graph ",Int.toString nodecnt,"\n"];
	       let fun neighbors (n,[]) = []
		     | neighbors (n,(v,s,t)::mm) = if n=s then t::(neighbors(n,mm))
						   else neighbors(n,mm)
		   fun getNodeInfo n =
		     (n,noduplicates(neighbors (n,arcs)))
		   val nodes = map getNodeInfo (fromto(0,nodecnt-1))
	       in
		   app (fn (n,neighbors) => app out ["c node ",
						     Int.toString n," ",
						     Int.toString(length neighbors),"\n"]) nodes;
		   app (fn (v,s,t) => app out ["c arc ",var2str v," ",
					       Int.toString s," ",
					       Int.toString t,"\n"]) arcs
	       end;
	       app out ["c endgraph\n"])
	    | NONE => ());
	  if acyc then out "c acyc\n" else ();
	  outreachables reachables;
	  outnonreachables nonreachables;
	  if smbolic
	  then app soutclause clauses
	  else app outclause clauses;
	  app (fn (v,s) => app out ["c ",var2str v," ",s,"\n"]) symtab
      end
  end;
