/*
	$Id: file.c,v 1.1 1999/12/12 18:18:39 rogue Exp $

	Copyright (c) 1999 Xforge project
*/

#include <Xm/Xm.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "xforge.h"
#include "file.h"
#include "glob.h"

/*
	helper function - your system may be missing this or it is
	hard to locate. How about autoconf?
*/

static char *basename(const char *s) {
	char *p;

	if ((p = strrchr(s, '/')))
		p++;
	else
		p = (char *)s;
	return p;
}

/*
	be_to_long returns unsigned 32 bit value of next 4 characters in
	char array given as argument
*/

static unsigned long be_to_long(unsigned char *s) {
	return (s[0] << 24) | (s[1] << 16) | (s[2] << 8) | s[3];
}

/*
	le_to_long returns unsigned 32 bit value of next 4 characters in
	char array given as argument
*/

static unsigned long le_to_long(unsigned char *s) {
	return s[0] | (s[1] << 8) | (s[2] << 16) | (s[3] << 24);
}

/*
	le_to_long returns unsigned 16 bit value of next 2 characters in
	char array given as argument
*/

static unsigned long le_to_short(unsigned char *s) {
	return s[0] | (s[1] << 8);
}

/*
	long_to_be writes unsigned 32 bit value to next 4 characters in
	char array given as argument
*/

static void long_to_be(unsigned char *s, unsigned long value) {
	s[0] = (value >> 24) & 0xff;
	s[1] = (value >> 16) & 0xff;
	s[2] = (value >> 8) & 0xff;
	s[3] = (value) & 0xff;
}

/*
	long_to_le writes unsigned 32 bit value to next 4 characters in
	char array given as argument
*/

static void long_to_le(unsigned char *s, unsigned long value) {
	s[0] = (value) & 0xff;
	s[1] = (value >> 8) & 0xff;
	s[2] = (value >> 16) & 0xff;
	s[3] = (value >> 24) & 0xff;
}

/*
	short_to_le writes unsigned 16 bit value to next 2 characters in
	char array given as argument
*/

static void short_to_le(unsigned char *s, unsigned long value) {
	s[0] = (value) & 0xff;
	s[1] = (value >> 8) & 0xff;
}

/*
	copy_le_8 copies channel-interleaved array of 8-bit integers to
	concatenated arrays of floats
*/

static void copy_le_8(float *q, char *p, int n, int s, int ch) {
	int i, j, skip;
	char *dp, *up;

	skip = ch;
	for (i = 0; i < ch; i++) {
		dp = p + i;
		up = (unsigned char *)dp;
		if (s)
			for (j = 0; j < n; j++)
				*q++ = (dp[j * skip]) / 128.0f;
		else
			for (j = 0; j < n; j++)
				*q++ = (((int)dp[j * skip] & 0xff) - 0x80) /
					128.0f;
	}
}

/*
	copy_le_16 copies channel-interleaved array of 16-bit little endian
	integers to concatenated arrays of floats
*/

static void copy_le_16(float *q, char *p, int n, int s, int ch) {
	int i, j, skip;
	char *dp;

	skip = ch * 2;
	for (i = 0; i < ch; i++) {
		dp = p + 2 * i;
		/* FIXME: note 0x100 underflow fix - what a hack! */
		for (j = 0; j < n; j++)
			*q++ = (dp[j * skip] + 256 * dp[j * skip + 1] -
				(s ? 0 : 32768) + (dp[j * skip] < 0 ?
				0x100 : 0)) / 32768.0f;
	}
}

/*
	copy_le_24 copies channel-interleaved array of 24-bit little endian
	integers to concatenated arrays of floats
*/

static void copy_le_24(float *q, char *p, int n, int s, int ch) {
	int i, j, skip;
	char *dp;

	skip = ch * 3;
	for (i = 0; i < ch; i++) {
		dp = p + 3 * i;
		/* FIXME: see above */
		for (j = 0; j < n; j++)
			*q++ = (dp[j * skip] + 256 * dp[j * skip + 1] +
				65536 * dp[j * skip + 2] - (s ? 0 : 8388608) +
				(dp[j * skip] < 0 ? 0x100 : 0) +
				(dp[j * skip + 1] < 0 ? 0x10000 : 0)) /
				8388608.0f;
	}
}

/*
	copy_le_32 copies channel-interleaved array of 32-bit little endian
	integers to concatenated arrays of floats
*/

static void copy_le_32(float *q, char *p, int n, int s, int ch) {
	int i, j, skip;
	char *dp;

	skip = ch * 3;
	for (i = 0; i < ch; i++) {
		dp = p + 4 * i;
		/* FIXME: see above */
		for (j = 0; j < n; j++) 
			*q++ = (dp[j * skip] + 256 * dp[j * skip + 1] +
				65536 * dp[j * skip + 2] + 16777216 *
				dp[j * skip + 3] - (s ? 0 : -I32WRAP) +
				(dp[j * skip] < 0 ? 0x100 : 0) +
				(dp[j * skip + 1] < 0 ? 0x10000 : 0) +
				(dp[j * skip + 2] < 0 ? 0x1000000 : 0)) /
				2147483648.0f;
	}
}

int io_openfile(struct buffer *buf, char *path) {
	struct stat statbuf;
	FILE *fp;
	int i, filef;
	char *p, tmp;
	float *q;
	unsigned long file_size, data_offset, data_size, samples, magic;
	char msgbuf[1024];
	unsigned char scanbuf[0x2c];	/* RIFF/RIFX header size */
	int bits, issigned, endianness, channels, rate, datalen;

	/* clear the scan buffer */
	memset(scanbuf, 0, sizeof(scanbuf));

	if (stat(path, &statbuf) == -1) {
		syserr(path);
		return -1;
	}

	file_size = statbuf.st_size;

	if (!(fp = fopen(path, "rb"))) {
		syserr(path);
		return -1;
	}

	/* resolve file type */
	if (file_size < 4) {		/* too small for magic */
		filef = FILEF_RAW;	/* assume 16 bit raw, little endian */
		data_offset = 0;
		data_size = file_size;
		rate = 44100;
		endianness = ENDIAN_LITTLE;
		bits = 16;
		issigned = 1;
		channels = 1;
	} else {
		fread(scanbuf, 1, sizeof(scanbuf), fp);
		rewind(fp);
		magic = be_to_long(scanbuf);
		switch (magic) {
			case MAGIC_SUN:
				filef = FILEF_SUN;
				data_offset = be_to_long(&scanbuf[4]);
				data_size = be_to_long(&scanbuf[8]);
				rate = be_to_long(&scanbuf[16]);
				endianness = ENDIAN_BIG;
				switch (be_to_long(&scanbuf[12])) {
					case SND_FORMAT_LINEAR_8:
						bits = 8;
						break;
					case SND_FORMAT_LINEAR_16:
						bits = 16;
						break;
					case SND_FORMAT_LINEAR_24:
						bits = 24;
						break;
					case SND_FORMAT_LINEAR_32:
						bits = 32;
						break;
					default:
						sprintf(msgbuf, "%s: "
							"format not "
							"supported, loading "
							"as raw file\n",
							path);
						xferr(msgbuf);
						filef = FILEF_RAW;
						data_offset = 0;
						data_size = file_size;
						rate = 44100;
						endianness = ENDIAN_BIG;
						bits = 16;
						issigned = 1;
						channels = 1;
						break;
				}
				issigned = 1;
				channels = be_to_long(&scanbuf[20]);
				break;
			case MAGIC_RIFF:
				filef = FILEF_RIFF;
				/* format tag must be PCM */
				if (le_to_short(&scanbuf[0x14]) != 1) {
					sprintf(msgbuf, "%s: format not "
						"supported, loading as raw "
						"file\n", path);
					xferr(msgbuf);
					filef = FILEF_RAW;
					data_offset = 0;
					data_size = file_size;
					rate = 44100;
					endianness = ENDIAN_LITTLE;
					bits = 16;
					issigned = 1;
					channels = 1;
					break;
				}
				data_offset = 0x2c;
				data_size = le_to_long(&scanbuf[0x28]);
				rate = le_to_long(&scanbuf[0x18]);
				endianness = ENDIAN_LITTLE;
				bits = le_to_short(&scanbuf[0x22]);
				issigned = (bits == 8 ? 0 : 1);
				channels = le_to_short(&scanbuf[0x16]);
				break;
			case MAGIC_RIFX:
				filef = FILEF_RIFX;
				/* format tag must be PCM */
				if (le_to_short(&scanbuf[0x14]) != 1) {
					sprintf(msgbuf, "%s: format not "
						"supported, loading as raw "
						"file\n", path);
					xferr(msgbuf);
					filef = FILEF_RAW;
					data_offset = 0;
					data_size = file_size;
					rate = 44100;
					endianness = ENDIAN_BIG;
					bits = 16;
					issigned = 1;
					channels = 1;
					break;
				}
				data_offset = 0x2c;
				data_size = le_to_long(&scanbuf[0x28]);
				rate = le_to_long(&scanbuf[0x18]);
				endianness = ENDIAN_BIG;
				bits = le_to_short(&scanbuf[0x22]);
				issigned = (bits == 8 ? 0 : 1);
				channels = le_to_short(&scanbuf[0x16]);
				break;
			default:
				/* FIXME: use some defaults or ask these */
				filef = FILEF_RAW;
				data_offset = 0;
				data_size = file_size;
				rate = 44100;
				endianness = ENDIAN_LITTLE;
				bits = 16;
				issigned = 1;
				channels = 1;
		}
	}

	/* go to beginning of data */
	fseek(fp, data_offset, SEEK_SET);

	if (!(p = malloc(data_size))) {
		syserr(EMEM_MSG);
		fclose(fp);
		return -1;
	}

	if (data_size != fread(p, 1, data_size, fp)) {
		syserr(path);
	}

	samples = data_size / (bits / 8);
	datalen = samples / channels;

	if (!(q = malloc(samples * sizeof(float)))) {
		syserr(EMEM_MSG);
		free(p);
		fclose(fp);
		return -1;
	}

	/* convert the buffer to little endian */
	if (endianness == ENDIAN_BIG) {
		switch (bits) {
			case 16:
				for (i = 0; i < data_size; i = i + 2) {
					tmp = p[i];
					p[i] = p[i + 1];
					p[i + 1] = tmp;
				}
				break;
			case 24:
				for (i = 0; i < data_size; i = i + 3) {
					tmp = p[i];
					p[i] = p[i + 1];
					p[i + 1] = p[i + 2];
					p[i + 2] = tmp;
				}
				break;
			case 32:
				for (i = 0; i < data_size; i = i + 4) {
					tmp = p[i];
					p[i] = p[i + 1];
					p[i + 1] = p[i + 2];
					p[i + 2] = p[i + 3];
					p[i + 3] = tmp;
				}
				break;
		}
	}

	switch (bits) {
		case 8:
			copy_le_8(q, p, datalen, issigned, channels);
			break;
		case 16:
			copy_le_16(q, p, datalen, issigned, channels);
			break;
		case 24:
			copy_le_24(q, p, datalen, issigned, channels);
			break;
		case 32:
			copy_le_32(q, p, datalen, issigned, channels);
			break;
	}

	free(p);
	fclose(fp);

	buf->data = q;
	buf->len = datalen;
	buf->dispstart = 0;
	buf->dispend = buf->len - 1;
	buf->path = strdup(path);
	buf->name = strdup(basename(path));
	buf->filef = filef;
	buf->channels = channels;
	buf->rate = rate;
	buf->bits = bits;
	buf->mod = 0;
	buf->selmask = 0;
	buf->selstart = 0;
	buf->selend = 0;
	return 0;
}

/* FIXME: quad channel samples are not yet implemented */

static void write_data(FILE *fp, struct buffer *buf, int endianness,
	int issigned) {
	int pmode = 0, i, sample;
	float *p, *q, *r, *s, fs;

	p = buf->data;
	q = p + buf->len;
	r = q + buf->len;
	s = r + buf->len;

	switch (buf->channels) {
		case 1:
			pmode = PMODE_MONO;
			break;
		case 2:
			pmode = PMODE_STEREO;
			break;
		case 4:
			pmode = PMODE_QUAD;
			xferr("quad channel saving not implemented");
			break;
	}

	switch (buf->bits) {
		case 8:
			pmode |= PMODE_8;
			break;
		case 16:
			pmode |= PMODE_16;
			break;
		case 24:
			pmode |= PMODE_24;
			break;
		case 32:
			pmode |= PMODE_32;
			break;
	}

	switch (endianness) {
		case ENDIAN_LITTLE:
			switch (pmode) {
				case PMODE_M8:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 128.0f + 0.5f;
						sample = (int)fs;
						if (sample > 127)
							sample = 127;
						if (sample < -128)
							sample = -128;
						sample += issigned ? 0 : 128;
						fputc(sample, fp);
					}
					break;
				case PMODE_M16:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 32768.0f;
						sample = (int)fs;
						if (sample > 32767)
							sample = 32767;
						if (sample < -32768)
							sample = -32768;
						sample += issigned ? 0 :
							32768;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
					}
					break;
				case PMODE_M24:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 8388608.0f;
						sample = (int)fs;
						if (sample > 8388607)
							sample = 8388607;
						if (sample < -8388608)
							sample = -8388608;
						sample += issigned ? 0 :
							8388608;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
					}
					break;
				case PMODE_M32:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 2147483648.0f;
						sample = (int)fs;
						if (fs > 2147483647.0f)
							sample = 2147483647;
						if (fs < -2147483648.0f)
							sample = -I32WRAP;
						sample += issigned ? 0 :
							I32WRAP;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 24 & 0xff, fp);
					}
					break;
				case PMODE_S8:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 128.0f + 0.5f;
						sample = (int)fs;
						if (sample > 127)
							sample = 127;
						if (sample < -128)
							sample = -128;
						sample += issigned ? 0 : 128;
						fputc(sample, fp);
						fs = *q++ * 128.0f + 0.5f;
						sample = (int)fs;
						if (sample > 127)
							sample = 127;
						if (sample < -128)
							sample = -128;
						sample += issigned ? 0 : 128;
						fputc(sample, fp);
					}
					break;
				case PMODE_S16:
					for (i = 0; i < buf->len; i++) {							fs = *p++ * 32768.0f;
						sample = (int)fs;
						if (sample > 32767)
							sample = 32767;
						if (sample < -32768)
							sample = -32768;
						sample += issigned ? 0 :
							32768;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fs = *q++ * 32768.0f;
						sample = (int)fs;
						if (sample > 32767)
							sample = 32767;
						if (sample < -32768)
							sample = -32768;
						sample += issigned ? 0 :
							32768;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
				}
					break;
				case PMODE_S24:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 8388608.0f;
						sample = (int)fs;
						if (sample > 8388607)
							sample = 8388607;
						if (sample < -8388608)
							sample = -8388608;
						sample += issigned ? 0 :
							8388608;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
						fs = *q++ * 8388608.0f;
						sample = (int)fs;
						if (sample > 8388607)
							sample = 8388607;
						if (sample < -8388608)
							sample = -8388608;
						sample += issigned ? 0 :
							8388608;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
					}
					break;
				case PMODE_S32:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 2147483648.0f;
						sample = (int)fs;
						if (fs > 2147483647.0f)
							sample = 2147483647;
						if (fs < -2147483648.0f)
							sample = -I32WRAP;
						sample += issigned ? 0 :
							I32WRAP;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 24 & 0xff, fp);
						fs = *q++ * 2147483648.0f;
						sample = (int)fs;
						if (fs > 2147483647.0f)
							sample = 2147483647;
						if (fs < -2147483648.0f)
							sample = -I32WRAP;
						sample += issigned ? 0 :
							I32WRAP;
						fputc(sample & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 24 & 0xff, fp);
					}
					break;
			}
			break;
		case ENDIAN_BIG:
			switch (pmode) {
				case PMODE_M8:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 128.0f + 0.5f;
						fs += issigned ? 0.0f : 128.0f;
						sample = (int)fs;
						if (sample > 127)
							sample = 127;
						if (sample < -128)
							sample = -128;
						fputc(sample, fp);
					}
					break;
				case PMODE_M16:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 32768.0f;
						sample = (int)fs;
						if (sample > 32767)
							sample = 32767;
						if (sample < -32768)
							sample = -32768;
						sample += issigned ? 0 :
							32768;
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
					}
					break;
				case PMODE_M24:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 8388608.0f;
						sample = (int)fs;
						if (sample > 8388607)
							sample = 8388607;
						if (sample < -8388608)
							sample = -8388608;
						sample += issigned ? 0 :
							8388608;
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
					}
					break;
				case PMODE_M32:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 2147483648.0f;
						sample = (int)fs;
						if (fs > 2147483647.0f)
							sample = 2147483647;
						if (fs < -2147483648.0f)
							sample = -I32WRAP;
						sample += issigned ? 0 :
							I32WRAP;
						fputc(sample >> 24 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
					}
					break;
				case PMODE_S8:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 128.0f + 0.5f;
						sample = (int)fs;
						if (sample > 127)
							sample = 127;
						if (sample < -128)
							sample = -128;
						sample += issigned ? 0 : 128;
						fputc(sample, fp);
						fs = *q++ * 128.0f + 0.5f;
						sample = (int)fs;
						if (sample > 127)
							sample = 127;
						if (sample < -128)
							sample = -128;
						sample += issigned ? 0 : 128;
						fputc(sample, fp);
					}
					break;
				case PMODE_S16:
					for (i = 0; i < buf->len; i++) {							fs = *p++ * 32768.0f;
						sample = (int)fs;
						if (sample > 32767)
							sample = 32767;
						if (sample < -32768)
							sample = -32768;
						sample += issigned ? 0 :
							32768;
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
						fs = *q++ * 32768.0f;
						sample = (int)fs;
						if (sample > 32767)
							sample = 32767;
						if (sample < -32768)
							sample = -32768;
						sample += issigned ? 0 :
							32768;
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
				}
					break;
				case PMODE_S24:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 8388608.0f;
						sample = (int)fs;
						if (sample > 8388607)
							sample = 8388607;
						if (sample < -8388608)
							sample = -8388608;
						sample += issigned ? 0 :
							8388608;
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
						fs = *q++ * 8388608.0f;
						sample = (int)fs;
						if (sample > 8388607)
							sample = 8388607;
						if (sample < -8388608)
							sample = -8388608;
						sample += issigned ? 0 :
							8388608;
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
					}
					break;
				case PMODE_S32:
					for (i = 0; i < buf->len; i++) {
						fs = *p++ * 2147483648.0f;
						sample = (int)fs;
						if (fs > 2147483647.0f)
							sample = 2147483647;
						if (fs < -2147483648.0f)
							sample = -I32WRAP;
						sample += issigned ? 0 :
							I32WRAP;
						fputc(sample >> 24 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
						fs = *q++ * 2147483648.0f;
						sample = (int)fs;
						if (fs > 2147483647.0f)
							sample = 2147483647;
						if (fs < -2147483648.0f)
							sample = -I32WRAP;
						sample += issigned ? 0 :
							I32WRAP;
						fputc(sample >> 24 & 0xff, fp);
						fputc(sample >> 16 & 0xff, fp);
						fputc(sample >> 8 & 0xff, fp);
						fputc(sample & 0xff, fp);
					}
					break;
			}
			break;
	}
}

int io_savefile(struct buffer *buf, char *path) {
	FILE *fp;
	unsigned char header[0x2c];	/* RIFF/RIFX header size */
	int datasize, issigned = 1, endianness = ENDIAN_LITTLE;

	if (!(fp = fopen(path, "wb"))) {
		syserr(path);
		return -1;
	}

	datasize = buf->channels * buf->len;

	switch (buf->filef) {
		case FILEF_SUN:
			long_to_be(&header[0], MAGIC_SUN);
			long_to_be(&header[4], 24);
			long_to_be(&header[8], datasize * buf->bits / 8);
			switch (buf->bits) {
				case 8:
					long_to_be(&header[12],
						SND_FORMAT_LINEAR_8);
					break;
				case 16:
					long_to_be(&header[12],
						SND_FORMAT_LINEAR_16);
					break;
				case 24:
					long_to_be(&header[12],
						SND_FORMAT_LINEAR_24);
					break;
				case 32:
					long_to_be(&header[12],
						SND_FORMAT_LINEAR_32);
					break;
			}
			long_to_be(&header[16], buf->rate);
			long_to_be(&header[20], buf->channels);
			if (fwrite(header, 24, 1, fp) != 1) {
				syserr(path);
				fclose(fp);
				return -1;
			}
			issigned = 1;
			endianness = ENDIAN_BIG;
			break;
		case FILEF_RIFF:
			long_to_be(&header[0x0], MAGIC_RIFF);
			/* exclude ckid and size */
			long_to_le(&header[0x4], datasize * buf->bits / 8
				+ 0x2c - 8);
			/* "WAVE" */
			long_to_be(&header[0x8], 0x57415645);
			/* "fmt " */
			long_to_be(&header[0xc], 0x666d7420);
			/* fmt ck size */
			long_to_le(&header[0x10], 0x10);
			/* fmt tag: PCM */
			short_to_le(&header[0x14], 0x01);
			/* # channels */
			short_to_le(&header[0x16], buf->channels);
			/* sample rate */
			long_to_le(&header[0x18], buf->rate);
			/* avg bytes per second */
			long_to_le(&header[0x1c], buf->rate * buf->channels *
				buf->bits / 8);
			/* bytes / sample */
			short_to_le(&header[0x20], buf->channels * buf->bits /
				8);
			/* bits / sample */
			short_to_le(&header[0x22], buf->bits);
			/* "data" */
			long_to_be(&header[0x24], 0x64617461);
			/* data length */
			long_to_le(&header[0x28], datasize * buf->bits / 8);
			if (fwrite(header, 0x2c, 1, fp) != 1) {
				syserr(path);
				fclose(fp);
				return -1;
			}
			issigned = (buf->bits == 8 ? 0 : 1);
			endianness = ENDIAN_LITTLE;
			break;
		case FILEF_RIFX:
			long_to_be(&header[0x0], MAGIC_RIFX);
			/* exclude ckid and size */
			long_to_le(&header[0x4], datasize * buf->bits / 8
				+ 0x2c - 8);
			/* "WAVE" */
			long_to_be(&header[0x8], 0x57415645);
			/* "fmt " */
			long_to_be(&header[0xc], 0x666d7420);
			/* fmt ck size */
			long_to_le(&header[0x10], 0x10);
			/* fmt tag: PCM */
			short_to_le(&header[0x14], 0x01);
			/* # channels */
			short_to_le(&header[0x16], buf->channels);
			/* sample rate */
			long_to_le(&header[0x18], buf->rate);
			/* avg bytes per second */
			long_to_le(&header[0x1c], buf->rate * buf->channels *
				buf->bits / 8);
			/* bytes / sample */
			short_to_le(&header[0x20], buf->channels * buf->bits /
				8);
			/* bits / sample */
			short_to_le(&header[0x22], buf->bits);
			/* "data" */
			long_to_be(&header[0x24], 0x64617461);
			/* data length */
			long_to_le(&header[0x28], datasize * buf->bits / 8);
			if (fwrite(header, 0x2c, 1, fp) != 1) {
				syserr(path);
				fclose(fp);
				return -1;
			}
			issigned = (buf->bits == 8 ? 0 : 1);
			endianness = ENDIAN_BIG;
			break;
		case FILEF_RAW:
			issigned = 1;
			endianness = ENDIAN_LITTLE;
			break;
	}

	write_data(fp, buf, endianness, issigned);

	buf->path = strdup(path);
	buf->name = strdup(basename(path));
	buf->mod = 0;

	fclose(fp);
	return 0;
}
