/*
	$Id: xforge.c,v 1.2 2000/01/03 17:08:48 rogue Exp $

	Copyright (c) 1999 Xforge project
*/

#include <Xm/Xm.h>
#include <Xm/AtomMgr.h>
#include <Xm/CascadeB.h>
#include <Xm/DrawingA.h>
#include <Xm/FileSB.h>
#include <Xm/Frame.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/MainW.h>
#include <Xm/MessageB.h>
#include <Xm/Protocols.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Scale.h>
#include <Xm/ScrollBar.h>
#include <Xm/SelectioB.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>
#include <X11/cursorfont.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>
#include "xforge.h"
#include "glob.h"

/* common variables */
static XtAppContext app;
static Widget quit_dialog, about_dialog, license_dialog, top_wid;
static XmString xmstring1, xmstring2,		/* temporal XmStrings */
		xmstring3, xmstring4;
static int top_argc;				/* copy of initial args */
static char **top_argv;
char *progname;

/* forward declarations */
static void spawn_gui(char *);

/* syserr dialog */

static void syserr_destroy_widget(Widget w) {
	XtUnmanageChild(w);
	XtDestroyWidget(w);
}

void syserr(char *msg) {
	char buffer[1024];	/* FIXME: figure out good buffer size */
	Widget widget, w;

	sprintf(buffer, "%s: %s", msg, strerror(errno));
	widget = XmCreateErrorDialog(top_wid, "errorDialog", NULL, 0);
	/* remove "Cancel" and "Help" buttons */
	w = XmMessageBoxGetChild(widget, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(w);
	w = XmMessageBoxGetChild(widget, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(w);
	xmstring1 = XmStringCreateLocalized(buffer);
	xmstring2 = XmStringCreateLocalized(DEFAULT_TITLE);
	XtVaSetValues(widget, XmNmessageString, xmstring1,
		XmNdialogTitle, xmstring2, NULL);
	XmStringFree(xmstring1);
	XmStringFree(xmstring2);
	XtAddCallback(widget, XmNokCallback,
			(XtCallbackProc)syserr_destroy_widget, NULL);
	XtManageChild(widget);
}

/* xforge err dialog */

void xferr(char *msg) {
	Widget widget, w;

	widget = XmCreateErrorDialog(top_wid, "errorDialog", NULL, 0);
	/* remove "Cancel" and "Help" buttons */
	w = XmMessageBoxGetChild(widget, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(w);
	w = XmMessageBoxGetChild(widget, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(w);
	xmstring1 = XmStringCreateLocalized(msg);
	xmstring2 = XmStringCreateLocalized(DEFAULT_TITLE);
	XtVaSetValues(widget, XmNmessageString, xmstring1,
		XmNdialogTitle, xmstring2, NULL);
	XmStringFree(xmstring1);
	XmStringFree(xmstring2);
	XtAddCallback(widget, XmNokCallback,
			(XtCallbackProc)syserr_destroy_widget, NULL);
	XtManageChild(widget);
}

/* use XC_watch cursor - note you cannot call this if toplevel is unmanaged */

static void use_xcwatch(void) {
	struct buffer *p;

	for (p = root; p; p = p->succ) {
		XDefineCursor(XtDisplay(p->top_wid),
			XtWindow(p->top_wid),
			p->xcwatch);
		if (XtIsManaged(p->param_dialog))
			XDefineCursor(XtDisplay(p->param_dialog),
				XtWindow(p->param_dialog),
				p->xcwatch);
		if (XtIsManaged(p->vol_dialog))
			XDefineCursor(XtDisplay(p->vol_dialog),
				XtWindow(p->vol_dialog),
				p->xcwatch);
		if (XtIsManaged(p->echo_dialog))
			XDefineCursor(XtDisplay(p->echo_dialog),
				XtWindow(p->echo_dialog),
				p->xcwatch);
		if (XtIsManaged(p->filt_dialog))
			XDefineCursor(XtDisplay(p->filt_dialog),
				XtWindow(p->filt_dialog),
				p->xcwatch);
		if (XtIsManaged(p->comp_dialog))
			XDefineCursor(XtDisplay(p->comp_dialog),
				XtWindow(p->comp_dialog),
				p->xcwatch);
		XSync(XtDisplay(p->top_wid), False);
	}
}

/* don't use XC_watch cursor - note you cannot call this is toplevel is */
/* unmanaged */

static void use_pointer(void) {
	struct buffer *p;

	for (p = root; p; p = p->succ) {
		XUndefineCursor(XtDisplay(p->top_wid),
			XtWindow(p->top_wid));
		if (XtIsManaged(p->param_dialog))
			XUndefineCursor(XtDisplay(p->param_dialog),
				XtWindow(p->param_dialog));
		if (XtIsManaged(p->vol_dialog))
			XUndefineCursor(XtDisplay(p->vol_dialog),
				XtWindow(p->vol_dialog));
		if (XtIsManaged(p->echo_dialog))
			XUndefineCursor(XtDisplay(p->echo_dialog),
				XtWindow(p->echo_dialog));
		if (XtIsManaged(p->filt_dialog))
			XUndefineCursor(XtDisplay(p->filt_dialog),
				XtWindow(p->filt_dialog));
		if (XtIsManaged(p->comp_dialog))
			XUndefineCursor(XtDisplay(p->comp_dialog),
				XtWindow(p->comp_dialog));
	}
}

/* check if string p has only nonnumeric [0123456789.] characters */

static int numchk(const char *p) {
	while (*p)
		if (isdigit(*p) || *p == '.')
			p++;
		else
			return -1;
	return 0;
}

/* callbacks */

static void render_buffer(Widget w, struct buffer *buf,
	XmDrawingAreaCallbackStruct *cbs) {
	int i, j;
	short height, width;
	XPoint *points;
	double p, s;

	XtVaGetValues(w, XmNheight, &height, XmNwidth, &width, NULL);
	if (!(points = malloc(width * sizeof(*points)))) {
		syserr(EMEM_MSG);
		return;
	}

	XClearArea(buf->display, XtWindow(w), 0, 0, width, height, FALSE);

	/* render horizontal line(s) */
	for (i = 0; i < buf->channels; i++) {
		points[0].x = 0;
		points[0].y = (1 + 2 * i) * height / buf->channels / 2;
		points[1].x = width - 1;
		points[1].y = points[0].y;
		XDrawLines(buf->display, XtWindow(w), buf->lgc, points, 2,
			CoordModeOrigin);
	}

	/* render waveform and selection */
	if (buf->data && width > 1) {
		/* FIXME: is double always accurate enough? */
		s = (double)(buf->dispend - buf->dispstart) /
			(double)(width - 1);
		for (i = 0; i < buf->channels; i++) {
			p = (double)(buf->dispstart + i * buf->len);
			for (j = 0; j < width; j++) {
				points[j].x = j;
				points[j].y = (short)((1 + 2 * i) * height /
					buf->channels / 2 + height /
					buf->channels / 2 * buf->data[(int)p]);
				p += s;
			}
			XDrawLines(buf->display, XtWindow(w), buf->lgc, points,
				width, CoordModeOrigin);
		}
	}
	free(points);
}

static void render_selection(Widget w, struct buffer *buf, int clearold) {
	short height, width, selx0, selx1, selw;

	XtVaGetValues(w, XmNheight, &height, XmNwidth, &width, NULL);

	if (buf->data) {
		/* wipe out old selection first if render_buffer not called */
		if (clearold) {
			if (buf->channels == 1) {
				if (buf->old_selmask) {
					XFillRectangle(buf->display,
						XtWindow(w),
						buf->sgc, buf->old_selx0, 0,
						buf->old_selw, height);
				}
			} else {
				if (buf->old_selmask & 0x1) {
					XFillRectangle(buf->display,
						XtWindow(w),
						buf->sgc, buf->old_selx0, 0,
						buf->old_selw, height / 2);
				}
				if (buf->old_selmask & 0x2) {
					XFillRectangle(buf->display,
						XtWindow(w),
						buf->sgc, buf->old_selx0,
						height / 2,
						buf->old_selw, height / 2);
				}
			}
		}
		/* draw new selection */
		selx0 = (float)(buf->selstart - buf->dispstart) *
			(float)width / (float)(buf->dispend - buf->dispstart);
		selx1 = (float)(buf->selend - buf->dispstart) *
			(float)width / (float)(buf->dispend - buf->dispstart);
		/* do nothing if selection is not visible */
		if ((selx0 < 0 && selx1 < 0) ||
			(selx0 > width && selx1 > width))
			return;
		if (selx0 < 0)
			selx0 = 0;
		if (selx1 >= width)
			selx1 = width;
		selw = selx1 - selx0;
		/* force selection width to at least 1 pixel */
		if (!selw)
			selw = 1;
		buf->old_selx0 = selx0;
		buf->old_selx1 = selx1;
		buf->old_selw = selw;
		buf->old_selmask = buf->selmask;
		if (buf->channels == 1) {
			if (buf->selmask) {
				XFillRectangle(buf->display, XtWindow(w),
					buf->sgc, selx0, 0,
					selw, height);
			}
		} else {
			if (buf->selmask & 0x1) {
				XFillRectangle(buf->display, XtWindow(w),
					buf->sgc, selx0, 0,
					selw, height / 2);
			}
			if (buf->selmask & 0x2) {
				XFillRectangle(buf->display, XtWindow(w),
					buf->sgc, selx0, height / 2,
					selw, height / 2);
			}
		}
	}
}

static void render_statbar(struct buffer *buf) {
	char *formatname = "empty buffer", buffer[1024];

	if (buf->data)
		switch (buf->filef) {
			case FILEF_RAW:
				formatname = "Raw data";
				break;
			case FILEF_SUN:
				formatname = "Sun Audio";
				break;
			case FILEF_RIFF:
				formatname = "RIFF Audio";
				break;
			case FILEF_RIFX:
				formatname = "RIFX Audio";
				break;
		}

	sprintf(buffer, "Length: %d DispStart: %d DispEnd: %d SelStart: %d "
		"SelEnd: %d %s %d S/s %s", buf->len, buf->dispstart,
		buf->dispend, buf->selstart, buf->selend, formatname,
		buf->rate, buf->mod ? "(Modified)" : "");
	xmstring1 = XmStringCreateLocalized(buffer);
	XtVaSetValues(buf->statlabel, XmNlabelString, xmstring1, NULL);
	XmStringFree(xmstring1);
}

static void refresh_scrollbar(struct buffer *buf) {
	if (buf->data && buf->len) {
		XtVaSetValues(buf->scrollbar,
			XmNminimum, 0,
			XmNmaximum, buf->len,
			XmNvalue, buf->dispstart,
			XmNsliderSize, buf->dispend - buf->dispstart + 1,
			XmNincrement, buf->len / 100,
			XmNpageIncrement, buf->dispend - buf->dispstart + 1,
			NULL);
	} else {
		XtVaSetValues(buf->scrollbar,
			XmNminimum, 0,
			XmNmaximum, 10,
			XmNvalue, 0,
			XmNsliderSize, 10,
			NULL);
	}
}

static void render_all(Widget w, struct buffer *buf,
	XmDrawingAreaCallbackStruct *cbs) {
	render_buffer(w, buf, cbs);
	render_selection(w, buf, 0);
	render_statbar(buf);
	refresh_scrollbar(buf);
}

static void scroll_canvas(Widget w, struct buffer *buf,
	XmScrollBarCallbackStruct *cbs) {
	int value, diff;

	XtVaGetValues(w, XmNvalue, &value, NULL);
	diff = value - buf->dispstart;
	buf->dispstart += diff;
	buf->dispend += diff;
	buf->old_selmask = 0;
	render_all(buf->canvas, buf, NULL);
}

/* menu callbacks */

static void new_window(Widget w, struct buffer *buf) {
	spawn_gui(NULL);
}

static void open_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->open_dialog);
}

static void open_really(Widget w, struct buffer *buf,
	XmFileSelectionBoxCallbackStruct *cbs) {
	char *path;

	XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &path);
	XtUnmanageChild(buf->open_dialog);
	/* if buffer is empty, load the file in this buffer */
	/* else spawn another window */
	if (buf->data) {
		spawn_gui(path);
	} else {
		use_xcwatch();
		if (io_openfile(buf, path) == -1) {
			use_pointer();
			return;
		}
		XtVaSetValues(buf->top_wid, XtNtitle, buf->name, NULL);
		/* FIXME: warning, cbs = NULL */
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void open_query(Widget w, struct buffer *buf) {
	XtManageChild(buf->open_dialog);
}

static void save_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->save_dialog);
}

static void save_really(Widget w, struct buffer *buf,
	XmFileSelectionBoxCallbackStruct *cbs) {
	char *path;

	XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &path);
	XtUnmanageChild(buf->save_dialog);
	use_xcwatch();
	io_savefile(buf, path);
	XtVaSetValues(buf->top_wid, XtNtitle, buf->name, NULL);
	render_all(buf->canvas, buf, NULL);
	use_pointer();
}

static void save_inst(Widget w, struct buffer *buf) {
	if (buf->data) {
		if (buf->path) {
			use_xcwatch();
			io_savefile(buf, buf->path);
			render_all(buf->canvas, buf, NULL);
			use_pointer();
		} else
			XtManageChild(buf->save_dialog);
	}
}

static void save_query(Widget w, struct buffer *buf) {
	if (buf->data)
		XtManageChild(buf->save_dialog);
}

static void revert_destroy_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(w);
	XtDestroyWidget(w);
}

static void revert_really(Widget w, struct buffer *buf) {
	revert_destroy_dialog(w, buf);
	use_xcwatch();
	io_openfile(buf, buf->path);
	if (buf->selend >= buf->len)
		buf->selend = buf->len - 1;
	render_all(buf->canvas, buf, NULL);
	use_pointer();
}

static void revert_query(Widget w, struct buffer *buf) {
	Widget widget;
	char buffer[1024];	/* FIXME: figure out a good buffer size */

	/* do nothing if buffer is not modified or is not from disk */
	if (buf->mod && buf->path) {
		sprintf(buffer, "Revert to %s?", buf->path);
		widget = XmCreateQuestionDialog(buf->top_wid, "revertDialog",
			NULL, 0);
		xmstring1 = XmStringCreateLocalized(buffer);
		xmstring2 = XmStringCreateLocalized(DEFAULT_TITLE);
		XtVaSetValues(widget, XmNmessageString, xmstring1,
			XmNdialogTitle, xmstring2, NULL);
		XmStringFree(xmstring1);
		XmStringFree(xmstring2);
		XtAddCallback(widget, XmNokCallback,
			(XtCallbackProc)revert_really, buf);
		XtAddCallback(widget, XmNcancelCallback,
			(XtCallbackProc)revert_destroy_dialog, buf);
		XtManageChild(widget);
	}
}

static void quit_really(void) {
	struct buffer *p;

	for (p = root; p; p = p->succ)
		if (p->p_pid != -1)
			oss_play_stop(p);
	exit(EXIT_SUCCESS);
}

static void quit_query(Widget w, struct buffer *buf) {
	XtManageChild(quit_dialog);
}

static void buf_undo(Widget w, struct buffer *buf) {
	undo(buf);
	buf->selmask = 0x0;
	buf->dispstart = buf->selstart = 0;
	buf->dispend = buf->selend = buf->len - 1;
	render_all(buf->canvas, buf, NULL);
}

static void clip_copy(Widget w, struct buffer *buf) {
	if (buf->data) {
		use_xcwatch();
		buf_copy(buf);
		use_pointer();
	}
}

static void clip_cut(Widget w, struct buffer *buf) {
	if (buf->data) {
		use_xcwatch();
		buf_cut(buf);
		buf->dispstart = 0;
		buf->dispend = buf->len - 1;
		buf->selend = buf->selstart;
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void clip_paste(Widget w, struct buffer *buf) {
	use_xcwatch();
	buf_paste(buf);
	buf->dispstart = 0;
	buf->dispend = buf->len - 1;
	render_all(buf->canvas, buf, NULL);
	use_pointer();
}

static void clip_swap(Widget w, struct buffer *buf) {
	buf_swaptoclip(buf);
	buf->selmask = 0x0;
	buf->dispstart = buf->selstart = 0;
	buf->dispend = buf->selend = buf->len - 1;
	render_all(buf->canvas, buf, NULL);
}

static void clear_really(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->clear_dialog);
	if (buf->data)
		free(buf->data);
	buf->selmask = 0x0;
	buf->data = NULL;
	buf->len = buf->selstart = buf->selend = buf->dispstart =
		buf->dispend = buf->mod = 0;
	XtVaSetValues(buf->top_wid, XtNtitle, DEFAULT_TITLE, NULL);
	render_all(buf->canvas, buf, NULL);
}

static void clear_query(Widget w, struct buffer *buf) {
	if (buf->mod)
		XtManageChild(buf->clear_dialog);
	else
		/* FIXME: should be Ok button of clear_dialog */
		clear_really(buf->clear_dialog, buf);
}

static void clip_delete(Widget w, struct buffer *buf) {
	if (buf->data) {
		use_xcwatch();
		buf_delete(buf);
		buf->dispstart = 0;
		buf->dispend = buf->len - 1;
		buf->selend = buf->selstart;
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void select_chg(Widget w, XButtonEvent *event, String *args,
	int *argc) {
	short width, height;
	short xstart, xend;		/* sorted x positions */
	struct buffer *buf;
	float c;
	int selmask = 0;

	XtVaGetValues(w, XmNheight, &height, XmNwidth, &width,
			XmNuserData, &buf, NULL);

	if (strcmp(args[0], "down") == 0) {
		buf->m_x0 = event->x;
		buf->m_y0 = event->y;
	}

	if (strcmp(args[0], "motion") == 0 || strcmp(args[0], "up") == 0) {
		buf->m_x1 = event->x;
		buf->m_y1 = event->y;
		if (buf->m_x1 > buf->m_x0) {
			xstart = buf->m_x0;
			xend = buf->m_x1;
		} else {
			xstart = buf->m_x1;
			xend = buf->m_x0;
		}
		if (xstart < 0)
			xstart = 0;
		if (xend >= width)
			xend = width - 1;
		if ((buf->m_y0 < height / 2) || (buf->m_y1 < height / 2))
			selmask |= 0x1;
		if ((buf->m_y0 >= height / 2) || (buf->m_y1 >= height / 2))
			selmask |= 0x2;
		buf->selmask = selmask;
		c = (float)(buf->dispend - buf->dispstart) /
			(float)(width - 1);
		buf->selstart = c * xstart + buf->dispstart;
		buf->selend = c * xend + buf->dispstart;
		render_selection(buf->canvas, buf, 1);
		render_statbar(buf);
	}
}

static void select_all(Widget w, struct buffer *buf) {
	if (buf->data) {
		if (buf->channels == 1)
			buf->selmask = 0x1;
		else
			buf->selmask = 0x3;
		buf->selstart = 0;
		buf->selend = buf->len - 1;
	}
	render_selection(buf->canvas, buf, 1);
	render_statbar(buf);
}

static void select_left(Widget w, struct buffer *buf) {
	if (buf->data) {
		buf->selmask = 0x1;
		buf->selstart = 0;
		buf->selend = buf->len - 1;
	}
	render_selection(buf->canvas, buf, 1);
	render_statbar(buf);
}

static void select_right(Widget w, struct buffer *buf) {
	if (buf->data) {
		buf->selmask = 0x2;
		buf->selstart = 0;
		buf->selend = buf->len - 1;
	}
	render_selection(buf->canvas, buf, 1);
	render_statbar(buf);
}

static void select_none(Widget w, struct buffer *buf) {
	buf->selmask = 0;
	render_selection(buf->canvas, buf, 1);
	render_statbar(buf);
}

static void begin_data(Widget w, struct buffer *buf) {
	if (buf->data) {
		buf->selstart = buf->selend = 0;
		/* select all channels if nothing is selected */
		if (!(buf->selmask))
			buf->selmask = (1 << buf->channels) - 1;
		render_all(buf->canvas, buf, NULL);
	}
}

static void ext_begin_data(Widget w, struct buffer *buf) {
	if (buf->data) {
		buf->selstart = 0;
		/* select the first sample of all channels if nothing */
		/* is selected */
		if (!(buf->selmask)) {
			buf->selmask = (1 << buf->channels) - 1;
			buf->selend = 0;
		}
		render_all(buf->canvas, buf, NULL);
	}
}

static void end_data(Widget w, struct buffer *buf) {
	if (buf->data) {
		buf->selstart = buf->selend = buf->len - 1;
		/* select all channels if nothing is selected */
		if (!(buf->selmask))
			buf->selmask = (1 << buf->channels) - 1;
		render_all(buf->canvas, buf, NULL);
	}
}

static void ext_end_data(Widget w, struct buffer *buf) {
	if (buf->data) {
		buf->selend = buf->len - 1;
		/* select the last sample of all channels if nothing */
		/* is selected */
		if (!(buf->selmask)) {
			buf->selmask = (1 << buf->channels) - 1;
			buf->selend = buf->len - 1;
		}
		render_all(buf->canvas, buf, NULL);
	}
}

static void addws_really(Widget w, struct buffer *buf,
	XmSelectionBoxCallbackStruct *cbs) {
	char *p;
	int n = 0;

	XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &p);
	XtUnmanageChild(w);
	XtDestroyWidget(w);
	if (numchk(p))
		xferr(ENUM_MSG);
	else
		n = atoi(p);
	if (n) {
		use_xcwatch();
		buf_addws(buf, n);
		buf->dispstart = 0;
		buf->dispend = buf->len - 1;
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void addws_query(Widget w, struct buffer *buf) {
	Widget dialog;

	dialog = XmCreatePromptDialog(buf->top_wid, "addwsDialog", NULL, 0);
	XtAddCallback(dialog, XmNokCallback,
		(XtCallbackProc)addws_really, buf);
	XtManageChild(dialog);
}

static void clip_remdc(Widget w, struct buffer *buf) {
	if (buf->data) {
		use_xcwatch();
		buf_remdc(buf);
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void show_range(Widget w, struct buffer *buf) {
	buf->dispstart = buf->selstart;
	buf->dispend = buf->selend;
	render_all(buf->canvas, buf, NULL);
}

static void show_all(Widget w, struct buffer *buf) {
	buf->dispstart = 0;
	buf->dispend = buf->len - 1;
	render_all(buf->canvas, buf, NULL);
}

static void zoom_out(Widget w, struct buffer *buf) {
	int center, span;

	center = (buf->dispend + buf->dispstart) / 2;
	span = buf->dispend - buf->dispstart + 1;
	buf->dispstart = center - span;
	if (buf->dispstart < 0)
		buf->dispstart = 0;
	buf->dispend = center + span;
	if (buf->dispend >= buf->len)
		buf->dispend = buf->len - 1;
	render_all(buf->canvas, buf, NULL);
}

static void close_really(Widget w, struct buffer *buf) {
	/* last one? then quit, otherwise just close buffer */
	if (nbuf == 1)
		quit_query(NULL, NULL);
	else {
		XtDestroyWidget(buf->top_wid);
		if (buf->p_pid != -1)
			oss_play_stop(buf);
		remove_buf(buf);
	}
}

static void close_query(Widget w, struct buffer *buf) {
	if (buf->mod)
		XtManageChild(buf->close_dialog);
	else
		close_really(w, buf);
}

static void refresh_window(Widget w, struct buffer *buf) {
	render_all(buf->canvas, buf, NULL);
}

static void play_range(Widget w, struct buffer *buf) {
	if (buf->data)
		oss_play_range(buf);
}

static void play_all(Widget w, struct buffer *buf) {
	if (buf->data)
		oss_play_all(buf);
}

static void play_stop(Widget w, struct buffer *buf) {
	if (buf->p_pid != -1)
		oss_play_stop(buf);
}

static void play_loop(Widget w, struct buffer *buf) {
	XtVaGetValues(w, XmNset, &(buf->p_loop), NULL);
}

static void param_set(Widget w, struct buffer *buf,
	XmPushButtonCallbackStruct *cbs) {
	char *rate;
	int testv = 0;

	XtVaGetValues(buf->param_dialog_srw, XmNvalue, &rate, NULL);
	if (numchk(rate))
		xferr(ENUM_MSG);
	else
		buf->rate = atoi(rate);
	XtVaGetValues(buf->param_dialog_ch1, XmNset, &testv, NULL);
	if (testv && buf->channels == 2) {
		use_xcwatch();
		if (!buf_stomshuffle(buf)) {
			buf->channels = 1;
			/* transition from 2 channels to 1: double length */
			buf->len *= 2;
			buf->selmask = 0;
			buf->dispstart = buf->selstart = 0;
			buf->dispend = buf->selend = buf->len - 1;
		}
		use_pointer();
	}
	XtVaGetValues(buf->param_dialog_ch2, XmNset, &testv, NULL);
	if (testv && buf->channels == 1) {
		use_xcwatch();
		if (!buf_mtosshuffle(buf)) {
			buf->channels = 2;
			/* transition from 1 channel to 2: halve length */
			/* note this truncates wave length */
			buf->len /= 2;
			buf->selmask = 0;
			buf->dispstart = buf->selstart = 0;
			buf->dispend = buf->selend = buf->len - 1;
		}
		use_pointer();
	}
	XtVaGetValues(buf->param_dialog_b8, XmNset, &testv, NULL);
	if (testv)
		buf->bits = 8;
	XtVaGetValues(buf->param_dialog_b16, XmNset, &testv, NULL);
	if (testv)
		buf->bits = 16;
	XtVaGetValues(buf->param_dialog_b24, XmNset, &testv, NULL);
	if (testv)
		buf->bits = 24;
	XtVaGetValues(buf->param_dialog_b32, XmNset, &testv, NULL);
	if (testv)
		buf->bits = 32;
	XtVaGetValues(buf->param_dialog_fraw, XmNset, &testv, NULL);
	if (testv)
		buf->filef = FILEF_RAW;
	XtVaGetValues(buf->param_dialog_fsun, XmNset, &testv, NULL);
	if (testv)
		buf->filef = FILEF_SUN;
	XtVaGetValues(buf->param_dialog_friff, XmNset, &testv, NULL);
	if (testv)
		buf->filef = FILEF_RIFF;
	XtVaGetValues(buf->param_dialog_frifx, XmNset, &testv, NULL);
	if (testv)
		buf->filef = FILEF_RIFX;
	render_all(buf->canvas, buf, NULL);
}

static void param_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->param_dialog);
}

static void param_query(Widget w, struct buffer *buf) {
	char buffer[1024];	/* FIXME */
	char *title = DEFAULT_TITLE;

	if (buf->name)
		title = buf->name;
	xmstring1 = XmStringCreateLocalized(title);
	XtVaSetValues(buf->param_dialog,
		XmNdialogTitle, xmstring1,
		NULL);
	XmStringFree(xmstring1);

	/* set widget states to those of buffer */
	sprintf(buffer, "%d", buf->rate);
	XtVaSetValues(buf->param_dialog_srw,
		XmNvalue, buffer,
		NULL);
	XtVaSetValues(buf->param_dialog_ch1,
		XmNset, (buf->channels == 1 ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_ch2,
		XmNset, (buf->channels == 2 ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_b8,
		XmNset, (buf->bits == 8 ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_b16,
		XmNset, (buf->bits == 16 ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_b24,
		XmNset, (buf->bits == 24 ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_b32,
		XmNset, (buf->bits == 32 ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_fraw,
		XmNset, (buf->filef == FILEF_RAW ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_fsun,
		XmNset, (buf->filef == FILEF_SUN ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_friff,
		XmNset, (buf->filef == FILEF_RIFF ? True : False),
		NULL);
	XtVaSetValues(buf->param_dialog_frifx,
		XmNset, (buf->filef == FILEF_RIFX ? True : False),
		NULL);

	XtManageChild(buf->param_dialog);
}

static void resample_really(Widget w, struct buffer *buf) {
	char *text;
	int testv, newrate, wlen, wfunc = WFUNC_RECTANGULAR;

	if (buf->data) {
		XtVaGetValues(buf->resample_dialog_srw, XmNvalue, &text, NULL);
		if (numchk(text)) {
			xferr(ENUM_MSG);
			return;
		} else
			newrate = atoi(text);
		XtVaGetValues(buf->resample_dialog_wlw, XmNvalue, &text, NULL);
		if (numchk(text)) {
			xferr(ENUM_MSG);
			return;
		} else
			wlen = atoi(text);
		XtVaGetValues(buf->resample_dialog_rect, XmNset, &testv, NULL);
		if (testv)
			wfunc = WFUNC_RECTANGULAR;
		XtVaGetValues(buf->resample_dialog_bart, XmNset, &testv, NULL);
		if (testv)
			wfunc = WFUNC_BARTLETT;
		XtVaGetValues(buf->resample_dialog_black, XmNset, &testv,
			NULL);
		if (testv)
			wfunc = WFUNC_BLACKMAN;
		XtVaGetValues(buf->resample_dialog_ham, XmNset, &testv, NULL);
		if (testv)
			wfunc = WFUNC_HAMMING;
		XtVaGetValues(buf->resample_dialog_han, XmNset, &testv, NULL);
		if (testv)
			wfunc = WFUNC_HANNING;
		use_xcwatch();
		resample(buf, newrate, wlen, wfunc);
		use_pointer();
	}
}

static void resample_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->resample_dialog);
}

static void resample_query(Widget w, struct buffer *buf) {
	char buffer[1024];	/* FIXME */

	sprintf(buffer, "%d", buf->rate);
	XtVaSetValues(buf->resample_dialog_srw,
		XmNvalue, buffer,
		NULL);
	XtVaSetValues(buf->resample_dialog_wlw,
		XmNvalue, "15",
		NULL);
	XtManageChild(buf->resample_dialog);
}

static void mono_really(Widget w, struct buffer *buf) {
	int ibalance;
	float balance;

	if (buf->data && buf->channels == 2) {
		XtVaGetValues(buf->mono_dialog_bal, XmNvalue, &ibalance, NULL);
		balance = (float)ibalance / 1000.0f;
		use_xcwatch();
		buf_tomono(buf, balance);
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void mono_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->mono_dialog);
}

static void mono_query(Widget w, struct buffer *buf) {
	XtManageChild(buf->mono_dialog);
}

static void stereo_query(Widget w, struct buffer *buf) {
}

static void phase_invert(Widget w, struct buffer *buf) {
	if (buf->data) {
		use_xcwatch();
		buf_invert(buf);
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void range_reverse(Widget w, struct buffer *buf) {
	if (buf->data) {
		use_xcwatch();
		buf_reverse(buf);
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void mix_query(Widget w, struct buffer *buf) {
}

static void vol_really(Widget w, struct buffer *buf) {
	int istart, iend;
	float fstart, fend;

	if (buf->data) {
		XtVaGetValues(buf->vol_dialog_st, XmNvalue, &istart, NULL);
		XtVaGetValues(buf->vol_dialog_end, XmNvalue, &iend, NULL);
		fstart = (float)istart / 1000.0f;
		fend = (float)iend / 1000.0f;
		use_xcwatch();
		eff_volume(buf, fstart, fend);
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void vol_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->vol_dialog);
}

static void vol_query(Widget w, struct buffer *buf) {
	XtManageChild(buf->vol_dialog);
}

static void echo_really(Widget w, struct buffer *buf) {
	int igain, ifeedback, idelay;
	float gain, feedback, delay;

	XtVaGetValues(buf->echo_dialog_gain, XmNvalue, &igain, NULL);
	XtVaGetValues(buf->echo_dialog_fb, XmNvalue, &ifeedback, NULL);
	XtVaGetValues(buf->echo_dialog_delay, XmNvalue, &idelay, NULL);
	gain = (float)igain / 1000.0f;
	feedback = (float)ifeedback / 1000.0f;
	delay = (float)idelay / 1000.0f;
	use_xcwatch();
	eff_echo(buf, gain, feedback, delay);
	render_all(buf->canvas, buf, NULL);
	use_pointer();
}

static void echo_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->echo_dialog);
}

static void echo_query(Widget w, struct buffer *buf) {
	XtManageChild(buf->echo_dialog);
}

static void reverb_query(Widget w, struct buffer *buf) {
}

static void filt_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->filt_dialog);
}

static void filt_apply(Widget w, struct buffer *buf) {
	char *freq;
	double lcf, ucf, tmp;		/* FIXME: check if low > high is OK */
	int testv = 0;
	int order = 1, type = F_LP;

	if (buf->data) {
		XtVaGetValues(buf->filt_dialog_lcw, XmNvalue, &freq, NULL);
		if (numchk(freq)) {
			xferr(ENUM_MSG);
			return;
		} else
			lcf = atof(freq);
		XtVaGetValues(buf->filt_dialog_ucw, XmNvalue, &freq, NULL);
		if (numchk(freq)) {
			xferr(ENUM_MSG);
			return;
		} else
			ucf = atof(freq);
		XtVaGetValues(buf->filt_dialog_o1, XmNset, &testv, NULL);
		if (testv) {
			order = 1;
		}
		XtVaGetValues(buf->filt_dialog_o2, XmNset, &testv, NULL);
		if (testv) {
			order = 2;
		}
		XtVaGetValues(buf->filt_dialog_tlp, XmNset, &testv, NULL);
		if (testv)
			type = F_LP;
		XtVaGetValues(buf->filt_dialog_thp, XmNset, &testv, NULL);
		if (testv)
			type = F_HP;
		XtVaGetValues(buf->filt_dialog_tbp, XmNset, &testv, NULL);
		if (testv)
			type = F_BP;
		XtVaGetValues(buf->filt_dialog_tbr, XmNset, &testv, NULL);
		if (testv)
			type = F_BR;
		/* if bandpass or bandreject, ensure that lower cutoff */
		/* really is lower than higher */
		if (lcf > ucf && (type == F_BP || type == F_BR)) {
			tmp = lcf;
			lcf = ucf;
			ucf = tmp;
		}
		use_xcwatch();
		eff_bfilter(buf, (float)lcf, (float)ucf, order, type);
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void filt_query(Widget w, struct buffer *buf) {
	char *title = DEFAULT_TITLE;

	if (buf->name)
		title = buf->name;
	xmstring1 = XmStringCreateLocalized(title);
	XtVaSetValues(buf->filt_dialog,
		XmNdialogTitle, xmstring1,
		NULL);
	XmStringFree(xmstring1);

	XtManageChild(buf->filt_dialog);
}

static void comp_really(Widget w, struct buffer *buf) {
	int idyn, ivel;
	float fdyn, fvel;

	if (buf->data) {
		XtVaGetValues(buf->comp_dialog_dyn, XmNvalue, &idyn, NULL);
		XtVaGetValues(buf->comp_dialog_vel, XmNvalue, &ivel, NULL);
		fdyn = (float)idyn / 1000.0f;
		fvel = (float)ivel;
		use_xcwatch();
		eff_compress(buf, fdyn, fvel);
		render_all(buf->canvas, buf, NULL);
		use_pointer();
	}
}

static void comp_hide_dialog(Widget w, struct buffer *buf) {
	XtUnmanageChild(buf->comp_dialog);
}

static void comp_query(Widget w, struct buffer *buf) {
	XtManageChild(buf->comp_dialog);
}

static void about_manage(Widget w, struct buffer *buf) {
	XtManageChild(about_dialog);
}

static void about_unmanage(Widget w, struct buffer *buf) {
	XtUnmanageChild(about_dialog);
}

static void license_manage(Widget w, struct buffer *buf) {
	XtManageChild(license_dialog);
}

/* menu structures */

static struct menubar_entry {
	String name;
	int id;
} menubar_entries[] = {
	{ "file",	0 },
	{ "edit",	1 },
	{ "view",	2 },
	{ "sound",	3 },
	{ "effects",	4 },
	{ "options",	5 },
	{ "help",	6 },
	{ NULL,		0 }
};

static struct menuitem_entry {
	String name;
#define M_PUSHB 0		/* push button */
#define M_TOGGLEB 1		/* toggle button */
#define M_SEP 2			/* separator */
	int type;
	void (*callback)(Widget, struct buffer *);
	int pid;		/* parent menu */
} menuitem_entries[] = {
	/* File */
	{ "new",	M_PUSHB,	new_window,	0 },
	{ "open",	M_PUSHB,	open_query,	0 },
	{ "save",	M_PUSHB,	save_inst,	0 },
	{ "saveAs",	M_PUSHB,	save_query,	0 },
	{ "revert",	M_PUSHB,	revert_query,	0 },
	{ "--------",	M_SEP,		NULL,		0 },
	{ "exit",	M_PUSHB,	quit_query,	0 },
	/* Edit */
	{ "undo",	M_PUSHB,	buf_undo,	1 },
	{ "--------",	M_SEP,		NULL,		1 },
	{ "copy",	M_PUSHB,	clip_copy,	1 },
	{ "cut",	M_PUSHB,	clip_cut,	1 },
	{ "paste",	M_PUSHB,	clip_paste,	1 },
	{ "swap",	M_PUSHB,	clip_swap,     	1 },
	{ "clear",	M_PUSHB,	clear_query,	1 },
	{ "delete",	M_PUSHB,	clip_delete,	1 },
	{ "--------",	M_SEP,		NULL,		1 },
	{ "selectAll",	M_PUSHB,	select_all,	1 },
	{ "selectLeft",	M_PUSHB,	select_left,	1 },
	{ "selectRight",M_PUSHB,	select_right,	1 },
	{ "deselect",	M_PUSHB,	select_none,	1 },
	{ "--------",	M_SEP,		NULL,		1 },
	{ "beginData",	M_PUSHB,	begin_data,	1 },
	{ "extBegin",	M_PUSHB,	ext_begin_data,	1 },
	{ "endData",	M_PUSHB,	end_data,	1 },
	{ "extEnd",	M_PUSHB,	ext_end_data,	1 },
	{ "--------",	M_SEP,		NULL,		1 },
	{ "removeDC",	M_PUSHB,	clip_remdc,	1 },
	{ "addWS",	M_PUSHB,	addws_query,	1 },
	/* View */
	{ "showRange",	M_PUSHB,	show_range,	2 },
	{ "showAll",	M_PUSHB,	show_all,	2 },
	{ "zoomOut",	M_PUSHB,	zoom_out,	2 },
	{ "--------",	M_SEP,		NULL,		2 },
	{ "closeView",	M_PUSHB,	close_query,	2 },
	{ "--------",	M_SEP,		NULL,		2 },
	{ "refresh",	M_PUSHB,	refresh_window,	2 },
	/* Sound */
	{ "playRange",	M_PUSHB,	play_range,	3 },
	{ "playAll",	M_PUSHB,	play_all,	3 },
	{ "stop",	M_PUSHB,	play_stop,	3 },
	{ "loop",	M_TOGGLEB,	play_loop,	3 },
	{ "sampler",	M_PUSHB,	NULL,		3 },
	{ "--------",	M_SEP,		NULL,		3 },
	{ "params",	M_PUSHB,	param_query,	3 },
	{ "resample",	M_PUSHB,	resample_query,	3 },
	{ "toMono",	M_PUSHB,	mono_query,	3 },
	{ "toStereo",	M_PUSHB,	stereo_query,	3 },
	{ "--------",	M_SEP,		NULL,		3 },
	{ "invert",	M_PUSHB,	phase_invert,	3 },
	{ "reverse",	M_PUSHB,	range_reverse,	3 },
	{ "mix",	M_PUSHB,	mix_query,	3 },
	/* Effects */
	{ "volume",	M_PUSHB,	vol_query,	4 },
	{ "echo",	M_PUSHB,	echo_query,	4 },
	{ "reverb",	M_PUSHB,	reverb_query,	4 },
	{ "filter",	M_PUSHB,	filt_query,	4 },
	{ "compress",	M_PUSHB,	comp_query,	4 },
	/* Options */
	{ "saveOpts",	M_PUSHB,	NULL,		5 },
	/* Help */
	{ "about",	M_PUSHB,	about_manage,	6 },
	{ "license",	M_PUSHB,	license_manage,	6 },
	{ NULL,		0,		NULL,		0 },
};

static void spawn_gui(char *path) {
	Widget widget;		/* temporal, general-purpose widget */
	struct menubar_entry *mp = menubar_entries;
	struct menuitem_entry *ip = menuitem_entries;
	struct buffer *buf;
	char *title = DEFAULT_TITLE;
	XtActionsRec canvas_actions;
	String canvas_translations =
		"<Btn1Down>: select_chg(down) ManagerGadgetArm() \n "
		"<Btn1Up>: select_chg(up) ManagerGadgetActivate() \n "
		"<Btn1Motion>: select_chg(motion) ManagerGadgetButtonMotion()";
	Atom delete_window_protocol;

	if (!(buf = add_buf()))
		return;

	if (path) {
		if (io_openfile(buf, path) == -1) {
			remove_buf(buf);
			return;
		}
		title = buf->name;
	}

	/* first make a new application shell */
	buf->display = XtOpenDisplay(app, NULL, NULL, "Xforge", NULL, 0,
		&top_argc, top_argv);
	buf->top_wid = XtVaAppCreateShell(NULL, "Xforge",
		applicationShellWidgetClass, buf->display,
		XtNtitle, title, NULL);

	/* catch WM_DELETE_WINDOW and let close_query() handle it */
	delete_window_protocol = XmInternAtom(XtDisplay(buf->top_wid),
		"WM_DELETE_WINDOW", True);
	XtVaSetValues(buf->top_wid,
		XmNdeleteResponse, XmDO_NOTHING,
		NULL);
	XmAddWMProtocolCallback(buf->top_wid, delete_window_protocol,
		(XtCallbackProc)close_query, buf);

	/* create a watch cursor */
	buf->xcwatch = XCreateFontCursor(XtDisplay(buf->top_wid), XC_watch);

	/* create main window */
	buf->mainw = XtVaCreateManagedWidget("mainWindow",
		xmMainWindowWidgetClass, buf->top_wid, NULL);

	/* create menubar */
	buf->menubar = XmCreateMenuBar(buf->mainw, "menuBar", NULL, 0);

	/* create menu bar buttons */
	do {
		widget = buf->mbe[mp->id] =
			XtVaCreateManagedWidget(mp->name,
				xmCascadeButtonWidgetClass, buf->menubar,
				NULL);
		buf->pulld[mp->id] =
			XmCreatePulldownMenu(buf->menubar, mp->name, NULL, 0);
		XtVaSetValues(buf->pulld[mp->id],
			XmNtearOffModel, XmTEAR_OFF_ENABLED,
			NULL);
		XtVaSetValues(buf->mbe[mp->id],
			XmNsubMenuId, buf->pulld[mp->id],
			NULL);
		mp++;
	} while (mp->name);
	/* widget now contains the last menu bar button, i.e. Help */
	/* mark it as a menuHelpWidget */
	XtVaSetValues(buf->menubar, XmNmenuHelpWidget, widget, NULL);

	/* create menu entries */
	while (ip->name) {
		switch(ip->type) {
			case M_PUSHB:
				widget = XtVaCreateManagedWidget(ip->name,
					xmPushButtonWidgetClass,
					buf->pulld[ip->pid], NULL);
				if (ip->callback)
					XtAddCallback(widget,
						XmNactivateCallback,
						(XtCallbackProc)ip->callback,
						buf);
				break;
			case M_TOGGLEB:
				widget = XtVaCreateManagedWidget(ip->name,
					xmToggleButtonWidgetClass,
					buf->pulld[ip->pid], NULL);
				if (ip->callback) {
					XtAddCallback(widget,
						XmNarmCallback,
						(XtCallbackProc)ip->callback,
						buf);
					XtAddCallback(widget,
						XmNdisarmCallback,
						(XtCallbackProc)ip->callback,
						buf);
				}
				break;
			case M_SEP:
				widget = XtVaCreateManagedWidget(ip->name,
					xmSeparatorWidgetClass,
					buf->pulld[ip->pid], NULL);
				break;
		}
		ip++;
	}

	XtManageChild(buf->menubar);

	/* create a Form to place work area */
	buf->form = XtVaCreateManagedWidget("form", xmFormWidgetClass,
		buf->mainw, NULL);

	/* status bar */
	buf->statframe = XtVaCreateManagedWidget("statFrame",
		xmFrameWidgetClass, buf->form,
		XmNshadowType, XmSHADOW_OUT,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	buf->statlabel = XtVaCreateManagedWidget("statLabel",
		xmLabelWidgetClass, buf->statframe,
		XmNalignment, XmALIGNMENT_BEGINNING,
		NULL);

	/* scrollbar */
	buf->scrollbar = XtVaCreateManagedWidget("scrollBar",
		xmScrollBarWidgetClass, buf->form,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget, buf->statframe,
		XmNorientation, XmHORIZONTAL,
		XmNminimum, 0,
		XmNmaximum, 10,
		XmNsliderSize, 10,
		NULL);
	XtAddCallback(buf->scrollbar, XmNvalueChangedCallback,
		(XtCallbackProc)scroll_canvas, buf);

	/* create a DrawingArea for waveform display */
	canvas_actions.string = "select_chg";
	canvas_actions.proc = (XtActionProc)select_chg;
	XtAppAddActions(app, &canvas_actions, 1);
	buf->canvas = XtVaCreateManagedWidget("canvas",
		xmDrawingAreaWidgetClass, buf->form,
		XmNtranslations, XtParseTranslationTable(canvas_translations),
		XmNuserData, buf,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_WIDGET,
		XmNbottomWidget, buf->scrollbar,
		NULL);
	buf->screen = XtScreen(buf->canvas);
	buf->lgcv.foreground = BlackPixelOfScreen(buf->screen);
	buf->lgc = XCreateGC(buf->display, RootWindowOfScreen(buf->screen),
		GCForeground, &(buf->lgcv));
	buf->sgcv.function = GXinvert;
	buf->sgc = XCreateGC(buf->display, RootWindowOfScreen(buf->screen),
		GCFunction, &(buf->sgcv));
	XtAddCallback(buf->canvas, XmNexposeCallback,
		(XtCallbackProc)render_all, buf);
	XtAddCallback(buf->canvas, XmNresizeCallback,
		(XtCallbackProc)render_all, buf);
	/* initialize selection backup */
	buf->old_selx0 = buf->old_selx1 = buf->old_selw = buf->old_selmask = 0;

	XtVaSetValues(buf->mainw, XmNmenuBar, buf->menubar, XmNworkWindow,
		buf->form, NULL);

	XtRealizeWidget(buf->top_wid);

	/* create an open dialog; don't show it yet */
	buf->open_dialog = XmCreateFileSelectionDialog(buf->top_wid,
		"openDialog", NULL, 0);
	XtAddCallback(buf->open_dialog, XmNcancelCallback,
		(XtCallbackProc)open_hide_dialog, buf);
	XtAddCallback(buf->open_dialog, XmNokCallback,
		(XtCallbackProc)open_really, buf);

	/* create a save dialog; don't show it yet */
	buf->save_dialog = XmCreateFileSelectionDialog(buf->top_wid,
		"saveDialog", NULL, 0);
	XtAddCallback(buf->save_dialog, XmNcancelCallback,
		(XtCallbackProc)save_hide_dialog, buf);
	XtAddCallback(buf->save_dialog, XmNokCallback,
		(XtCallbackProc)save_really, buf);

	/* create a clear dialog; don't show it yet */
	buf->clear_dialog = XmCreateQuestionDialog(buf->top_wid, "clearDialog",
		NULL, 0);
	XtAddCallback(buf->clear_dialog, XmNokCallback,
		(XtCallbackProc)clear_really, buf);

	/* create a close dialog; don't show it yet */
	buf->close_dialog = XmCreateQuestionDialog(buf->top_wid, "closeDialog",
		NULL, 0);
	XtAddCallback(buf->close_dialog, XmNokCallback,
		(XtCallbackProc)close_really, buf);

	/* create a param dialog; don't show it yet */
	buf->param_dialog = XmCreateFormDialog(buf->top_wid, "paramDialog",
		NULL, 0);

	buf->param_dialog_rc = XtVaCreateManagedWidget("rowCol",
		xmRowColumnWidgetClass, buf->param_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNorientation, XmVERTICAL,
		XmNnumColumns, 2,
		XmNpacking, XmPACK_COLUMN,
		NULL);

	/* labels */
	XtVaCreateManagedWidget("rateLabel", xmLabelWidgetClass,
		buf->param_dialog_rc, NULL);
	XtVaCreateManagedWidget("chanLabel", xmLabelWidgetClass,
		buf->param_dialog_rc, NULL);
	XtVaCreateManagedWidget("bitsLabel", xmLabelWidgetClass,
		buf->param_dialog_rc, NULL);
	XtVaCreateManagedWidget("fmtLabel", xmLabelWidgetClass,
		buf->param_dialog_rc, NULL);

	/* data rate */
	buf->param_dialog_srw = XtVaCreateManagedWidget("rateText",
		xmTextWidgetClass, buf->param_dialog_rc, NULL);
	/* channels */
	buf->param_dialog_chm = XmCreateRadioBox(buf->param_dialog_rc,
		"channels", NULL, 0);
	XtVaSetValues(buf->param_dialog_chm, XmNorientation, XmHORIZONTAL,
		NULL);
	buf->param_dialog_ch1 = XtVaCreateManagedWidget("ch1",
		xmToggleButtonWidgetClass, buf->param_dialog_chm,
		NULL);
	buf->param_dialog_ch2 = XtVaCreateManagedWidget("ch2",
		xmToggleButtonWidgetClass, buf->param_dialog_chm,
		NULL);
	XtManageChild(buf->param_dialog_chm);

	/* sample bits */
	buf->param_dialog_bm = XmCreateRadioBox(buf->param_dialog_rc,
		"bits", NULL, 0);
	XtVaSetValues(buf->param_dialog_bm, XmNorientation, XmHORIZONTAL,
		NULL);
	buf->param_dialog_b8 = XtVaCreateManagedWidget("bits8",
		xmToggleButtonWidgetClass, buf->param_dialog_bm,
		NULL);
	buf->param_dialog_b16 = XtVaCreateManagedWidget("bits16",
		xmToggleButtonWidgetClass, buf->param_dialog_bm,
		NULL);
	buf->param_dialog_b24 = XtVaCreateManagedWidget("bits24",
		xmToggleButtonWidgetClass, buf->param_dialog_bm,
		NULL);
	buf->param_dialog_b32 = XtVaCreateManagedWidget("bits32",
		xmToggleButtonWidgetClass, buf->param_dialog_bm,
		NULL);
	XtManageChild(buf->param_dialog_bm);

	/* file format */
	buf->param_dialog_fm = XmCreateRadioBox(buf->param_dialog_rc,
		"fmt", NULL, 0);
	XtVaSetValues(buf->param_dialog_fm, XmNorientation, XmHORIZONTAL,
		NULL);
	buf->param_dialog_fraw = XtVaCreateManagedWidget("fmtRaw",
		xmToggleButtonWidgetClass, buf->param_dialog_fm,
		NULL);
	buf->param_dialog_fsun = XtVaCreateManagedWidget("fmtSun",
		xmToggleButtonWidgetClass, buf->param_dialog_fm,
		NULL);
	buf->param_dialog_friff = XtVaCreateManagedWidget("fmtRIFF",
		xmToggleButtonWidgetClass, buf->param_dialog_fm,
		NULL);
	buf->param_dialog_frifx = XtVaCreateManagedWidget("fmtRIFX",
		xmToggleButtonWidgetClass, buf->param_dialog_fm,
		NULL);
	XtManageChild(buf->param_dialog_fm);

	buf->param_dialog_sep = XtVaCreateManagedWidget("separator",
		xmSeparatorWidgetClass, buf->param_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->param_dialog_rc,
		NULL);

	buf->param_dialog_apply = XtVaCreateManagedWidget("apply",
		xmPushButtonWidgetClass, buf->param_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->param_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->param_dialog_apply, XmNactivateCallback,
		(XtCallbackProc)param_set, buf);

	buf->param_dialog_cancel = XtVaCreateManagedWidget("dismiss",
		xmPushButtonWidgetClass, buf->param_dialog,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->param_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->param_dialog_cancel, XmNactivateCallback,
		(XtCallbackProc)param_hide_dialog, buf);

	/* create a resample dialog; don't show it yet */
	buf->resample_dialog = XmCreateFormDialog(buf->top_wid,
		"resampleDialog", NULL, 0);

	buf->resample_dialog_rc = XtVaCreateManagedWidget("rowCol",
		xmFormWidgetClass, buf->resample_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		NULL);

	/* data rate */
	buf->resample_dialog_srw = XtVaCreateManagedWidget("rateText",
		xmTextWidgetClass, buf->resample_dialog_rc,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_POSITION,
		XmNleftPosition, 50,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);
	/* window length */
	buf->resample_dialog_wlw = XtVaCreateManagedWidget("windowLen",
		xmTextWidgetClass, buf->resample_dialog_rc,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->resample_dialog_srw,
		XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget, buf->resample_dialog_srw,
		XmNrightAttachment, XmATTACH_FORM,
		NULL);

	/* window function */
	buf->resample_dialog_wfm = XmCreateRadioBox(buf->resample_dialog_rc,
		"wfunc", NULL, 0);
	XtVaSetValues(buf->resample_dialog_wfm, XmNorientation, XmVERTICAL,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->resample_dialog_wlw,
		XmNleftAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget, buf->resample_dialog_wlw,
		XmNrightAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	buf->resample_dialog_rect = XtVaCreateManagedWidget("rect",
		xmToggleButtonWidgetClass, buf->resample_dialog_wfm,
		XmNset, True,
		NULL);
	buf->resample_dialog_bart = XtVaCreateManagedWidget("bart",
		xmToggleButtonWidgetClass, buf->resample_dialog_wfm,
		XmNset, False,
		NULL);
	buf->resample_dialog_black = XtVaCreateManagedWidget("black",
		xmToggleButtonWidgetClass, buf->resample_dialog_wfm,
		XmNset, False,
		NULL);
	buf->resample_dialog_ham = XtVaCreateManagedWidget("hamming",
		xmToggleButtonWidgetClass, buf->resample_dialog_wfm,
		XmNset, False,
		NULL);
	buf->resample_dialog_han = XtVaCreateManagedWidget("hanning",
		xmToggleButtonWidgetClass, buf->resample_dialog_wfm,
		XmNset, False,
		NULL);
	XtManageChild(buf->resample_dialog_wfm);

	/* labels */
	XtVaCreateManagedWidget("rateLabel",
		xmLabelWidgetClass, buf->resample_dialog_rc,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_WIDGET,
		XmNrightWidget, buf->resample_dialog_srw,
		XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNbottomWidget, buf->resample_dialog_srw,
		NULL);
	XtVaCreateManagedWidget("wlenLabel",
		xmLabelWidgetClass, buf->resample_dialog_rc,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget, buf->resample_dialog_wlw,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_WIDGET,
		XmNrightWidget, buf->resample_dialog_wlw,
		XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNbottomWidget, buf->resample_dialog_wlw,
		NULL);
	XtVaCreateManagedWidget("wfuncLabel",
		xmLabelWidgetClass, buf->resample_dialog_rc,
		XmNalignment, XmALIGNMENT_BEGINNING,
		XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget, buf->resample_dialog_wfm,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_WIDGET,
		XmNrightWidget, buf->resample_dialog_wfm,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);

	buf->resample_dialog_sep = XtVaCreateManagedWidget("separator",
		xmSeparatorWidgetClass, buf->resample_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->resample_dialog_rc,
		NULL);

	buf->resample_dialog_apply = XtVaCreateManagedWidget("apply",
		xmPushButtonWidgetClass, buf->resample_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->resample_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->resample_dialog_apply, XmNactivateCallback,
		(XtCallbackProc)resample_really, buf);

	buf->resample_dialog_cancel = XtVaCreateManagedWidget("dismiss",
		xmPushButtonWidgetClass, buf->resample_dialog,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->resample_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->resample_dialog_cancel, XmNactivateCallback,
		(XtCallbackProc)resample_hide_dialog, buf);

	/* create a mono dialog; don't show it yet */
	buf->mono_dialog = XmCreateFormDialog(buf->top_wid, "monoDialog",
		NULL, 0);

	widget = XtVaCreateManagedWidget("left", xmLabelWidgetClass,
		buf->mono_dialog,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_FORM,
		NULL);

	buf->mono_dialog_bal = XtVaCreateManagedWidget("balance",
		xmScaleWidgetClass, buf->mono_dialog,
		XmNminimum, -1000,
		XmNmaximum, 1000,
		XmNvalue, 0,
		XmNdecimalPoints, 3,
		XmNscaleMultiple, 1,
		XmNtopAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, widget,
		XmNorientation, XmHORIZONTAL,
		NULL);

	XtVaSetValues(widget,
		XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNbottomWidget, buf->mono_dialog_bal,
		NULL);

	XtVaCreateManagedWidget("right", xmLabelWidgetClass, buf->mono_dialog,
		XmNtopAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, buf->mono_dialog_bal,
		XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNbottomWidget, buf->mono_dialog_bal,
		NULL);

	buf->mono_dialog_sep = XtVaCreateManagedWidget("separator",
		xmSeparatorWidgetClass, buf->mono_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->mono_dialog_bal,
		NULL);

	buf->mono_dialog_apply = XtVaCreateManagedWidget("apply",
		xmPushButtonWidgetClass, buf->mono_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->mono_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->mono_dialog_apply, XmNactivateCallback,
		(XtCallbackProc)mono_really, buf);

	buf->mono_dialog_cancel = XtVaCreateManagedWidget("dismiss",
		xmPushButtonWidgetClass, buf->mono_dialog,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->mono_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->mono_dialog_cancel, XmNactivateCallback,
		(XtCallbackProc)mono_hide_dialog, buf);

	/* create a volume dialog; don't show it yet */
	buf->vol_dialog = XmCreateFormDialog(buf->top_wid, "volDialog",
		NULL, 0);

	buf->vol_dialog_rc = XtVaCreateManagedWidget("rowCol",
		xmRowColumnWidgetClass, buf->vol_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNorientation, XmHORIZONTAL,
		NULL);

	buf->vol_dialog_st = XtVaCreateManagedWidget("start",
		xmScaleWidgetClass, buf->vol_dialog_rc,
		XmNminimum, 0,
		XmNmaximum, 2000,
		XmNvalue, 1000,
		XmNdecimalPoints, 3,
		XmNscaleMultiple, 1,
		NULL);

	buf->vol_dialog_end = XtVaCreateManagedWidget("end",
		xmScaleWidgetClass, buf->vol_dialog_rc,
		XmNminimum, 0,
		XmNmaximum, 2000,
		XmNvalue, 1000,
		XmNdecimalPoints, 3,
		XmNscaleMultiple, 1,
		NULL);

	buf->vol_dialog_sep = XtVaCreateManagedWidget("separator",
		xmSeparatorWidgetClass, buf->vol_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->vol_dialog_rc,
		NULL);

	buf->vol_dialog_apply = XtVaCreateManagedWidget("apply",
		xmPushButtonWidgetClass, buf->vol_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->vol_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->vol_dialog_apply, XmNactivateCallback,
		(XtCallbackProc)vol_really, buf);

	buf->vol_dialog_cancel = XtVaCreateManagedWidget("dismiss",
		xmPushButtonWidgetClass, buf->vol_dialog,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->vol_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->vol_dialog_cancel, XmNactivateCallback,
		(XtCallbackProc)vol_hide_dialog, buf);

	/* create an echo dialog; don't show it yet */
	buf->echo_dialog = XmCreateFormDialog(buf->top_wid, "echoDialog",
		NULL, 0);

	buf->echo_dialog_rc = XtVaCreateManagedWidget("rowCol",
		xmRowColumnWidgetClass, buf->echo_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNorientation, XmHORIZONTAL,
		NULL);

	buf->echo_dialog_gain = XtVaCreateManagedWidget("gain",
		xmScaleWidgetClass, buf->echo_dialog_rc,
		XmNminimum, 0,
		XmNmaximum, 3000,
		XmNdecimalPoints, 3,
		XmNscaleMultiple, 1,
		NULL);

	buf->echo_dialog_fb = XtVaCreateManagedWidget("feedback",
		xmScaleWidgetClass, buf->echo_dialog_rc,
		XmNminimum, 0,
		XmNmaximum, 1000,
		XmNdecimalPoints, 3,
		XmNscaleMultiple, 1,
		NULL);

	buf->echo_dialog_delay = XtVaCreateManagedWidget("delay",
		xmScaleWidgetClass, buf->echo_dialog_rc,
		XmNminimum, 0,
		XmNmaximum, 10000,
		XmNdecimalPoints, 3,
		XmNscaleMultiple, 1,
		NULL);

	buf->echo_dialog_sep = XtVaCreateManagedWidget("separator",
		xmSeparatorWidgetClass, buf->echo_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->echo_dialog_rc,
		NULL);

	buf->echo_dialog_apply = XtVaCreateManagedWidget("apply",
		xmPushButtonWidgetClass, buf->echo_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->echo_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->echo_dialog_apply, XmNactivateCallback,
		(XtCallbackProc)echo_really, buf);

	buf->echo_dialog_cancel = XtVaCreateManagedWidget("dismiss",
		xmPushButtonWidgetClass, buf->echo_dialog,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->echo_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->echo_dialog_cancel, XmNactivateCallback,
		(XtCallbackProc)echo_hide_dialog, buf);

	/* create a filter dialog; don't show it yet */
	buf->filt_dialog = XmCreateFormDialog(buf->top_wid, "filtDialog",
		NULL, 0);

	buf->filt_dialog_rc = XtVaCreateManagedWidget("rowCol",
		xmRowColumnWidgetClass, buf->filt_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNorientation, XmVERTICAL,
		XmNnumColumns, 2,
		XmNpacking, XmPACK_COLUMN,
		NULL);

	/* labels */
	XtVaCreateManagedWidget("lowLabel", xmLabelWidgetClass,
		buf->filt_dialog_rc, NULL);
	XtVaCreateManagedWidget("hiLabel", xmLabelWidgetClass,
		buf->filt_dialog_rc, NULL);
	XtVaCreateManagedWidget("orderLabel", xmLabelWidgetClass,
		buf->filt_dialog_rc, NULL);
	XtVaCreateManagedWidget("typeLabel", xmLabelWidgetClass,
		buf->filt_dialog_rc, NULL);

	/* lower cutoff frequency */
	buf->filt_dialog_lcw = XtVaCreateManagedWidget("lowText",
		xmTextWidgetClass, buf->filt_dialog_rc,
		NULL);

	/* upper cutoff frequency */
	buf->filt_dialog_ucw = XtVaCreateManagedWidget("hiText",
		xmTextWidgetClass, buf->filt_dialog_rc,
		NULL);

	/* filter order */
	buf->filt_dialog_om = XmCreateRadioBox(buf->filt_dialog_rc,
		"order", NULL, 0);
	XtVaSetValues(buf->filt_dialog_om, XmNorientation, XmHORIZONTAL,
		NULL);
	buf->filt_dialog_o1 = XtVaCreateManagedWidget("order1",
		xmToggleButtonWidgetClass, buf->filt_dialog_om,
		NULL);
	buf->filt_dialog_o2 = XtVaCreateManagedWidget("order2",
		xmToggleButtonWidgetClass, buf->filt_dialog_om,
		NULL);
	XtManageChild(buf->filt_dialog_om);

	/* filter type */
	buf->filt_dialog_tm = XmCreateRadioBox(buf->filt_dialog_rc,
		"type", NULL, 0);
	XtVaSetValues(buf->filt_dialog_tm, XmNorientation, XmHORIZONTAL,
		NULL);
	buf->filt_dialog_tlp = XtVaCreateManagedWidget("lp",
		xmToggleButtonWidgetClass, buf->filt_dialog_tm,
		NULL);
	buf->filt_dialog_thp = XtVaCreateManagedWidget("hp",
		xmToggleButtonWidgetClass, buf->filt_dialog_tm,
		NULL);
	buf->filt_dialog_tbp = XtVaCreateManagedWidget("bp",
		xmToggleButtonWidgetClass, buf->filt_dialog_tm,
		NULL);
	buf->filt_dialog_tbr = XtVaCreateManagedWidget("br",
		xmToggleButtonWidgetClass, buf->filt_dialog_tm,
		NULL);
	XtManageChild(buf->filt_dialog_tm);

	buf->filt_dialog_sep = XtVaCreateManagedWidget("separator",
		xmSeparatorWidgetClass, buf->filt_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->filt_dialog_rc,
		NULL);

	buf->filt_dialog_apply = XtVaCreateManagedWidget("apply",
		xmPushButtonWidgetClass, buf->filt_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->filt_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->filt_dialog_apply, XmNactivateCallback,
		(XtCallbackProc)filt_apply, buf);

	buf->filt_dialog_cancel = XtVaCreateManagedWidget("dismiss",
		xmPushButtonWidgetClass, buf->filt_dialog,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->filt_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->filt_dialog_cancel, XmNactivateCallback,
		(XtCallbackProc)filt_hide_dialog, buf);

	/* set widget states to something basic */
	XtVaSetValues(buf->filt_dialog_lcw,
		XmNvalue, "1000",
		NULL);
	XtVaSetValues(buf->filt_dialog_ucw,
		XmNvalue, "10000",
		NULL);
	XtVaSetValues(buf->filt_dialog_o1,
		XmNset, True,
		NULL);
	XtVaSetValues(buf->filt_dialog_o2,
		XmNset, False,
		NULL);
	XtVaSetValues(buf->filt_dialog_tlp,
		XmNset, True,
		NULL);
	XtVaSetValues(buf->filt_dialog_thp,
		XmNset, False,
		NULL);
	XtVaSetValues(buf->filt_dialog_tbp,
		XmNset, False,
		NULL);
	XtVaSetValues(buf->filt_dialog_tbr,
		XmNset, False,
		NULL);

	/* create a compress dialog; don't show it yet */
	buf->comp_dialog = XmCreateFormDialog(buf->top_wid, "compDialog",
		NULL, 0);

	buf->comp_dialog_rc = XtVaCreateManagedWidget("rowCol",
		xmRowColumnWidgetClass, buf->comp_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNorientation, XmHORIZONTAL,
		NULL);

	buf->comp_dialog_dyn = XtVaCreateManagedWidget("dynamics",
		xmScaleWidgetClass, buf->comp_dialog_rc,
		XmNminimum, -1000,
		XmNmaximum, 1000,
		XmNdecimalPoints, 3,
		XmNscaleHeight, 200,
		XmNscaleMultiple, 1,
		NULL);

	buf->comp_dialog_vel = XtVaCreateManagedWidget("velocity",
		xmScaleWidgetClass, buf->comp_dialog_rc,
		XmNminimum, 10,
		XmNmaximum, 1000,
		XmNvalue, 20,
		XmNdecimalPoints, 0,
		XmNscaleMultiple, 1,
		NULL);

	buf->comp_dialog_sep = XtVaCreateManagedWidget("separator",
		xmSeparatorWidgetClass, buf->comp_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->comp_dialog_rc,
		NULL);

	buf->comp_dialog_apply = XtVaCreateManagedWidget("apply",
		xmPushButtonWidgetClass, buf->comp_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->comp_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->comp_dialog_apply, XmNactivateCallback,
		(XtCallbackProc)comp_really, buf);

	buf->comp_dialog_cancel = XtVaCreateManagedWidget("dismiss",
		xmPushButtonWidgetClass, buf->comp_dialog,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, buf->comp_dialog_sep,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);
	XtAddCallback(buf->comp_dialog_cancel, XmNactivateCallback,
		(XtCallbackProc)comp_hide_dialog, buf);
}

void childhandler(int sig) {
	wait(NULL);		/* FIXME: check if this is ok */
	if ((signal(SIGCHLD, childhandler)) == SIG_ERR)
		xferr("warning: cannot set signal handler");
}

int main(int argc, char **argv) {
	Widget w;
	int c;

	/* initialize application */
	progname = *argv;
	top_argc = argc;
	top_argv = argv;
	top_wid = XtVaAppInitialize(&app, "Xforge", NULL, 0, &argc, argv,
		NULL, NULL);

	while ((c = getopt(argc, argv, "D:")) != EOF) {
		switch (c) {
			case 'D':
				dspdev = optarg;
				break;
			default:
				exit(EXIT_FAILURE);
		}
	}

	argc -= optind;
	argv += optind;

	/* install signal handler for catching SIGCHLDs */
	if ((signal(SIGCHLD, childhandler)) == SIG_ERR)
		xferr("warning: cannot set signal handler");

	/* set up clipboard */
	if (!(clip = malloc(sizeof(struct buffer)))) {
		syserr(EMEM_MSG);
		exit(EXIT_FAILURE);
	}
	clip->data = NULL;
	clip->channels = 1;
	clip->len = 0;
	clip->rate = 44100;

	/* spawn a new gui for each name on command line */
	/* if no names exist, spawn a single empty wave gui */
	do {
		spawn_gui(*argv);
		if (argc) {
			argc--;
			argv++;
		}
	} while (argc);

	/* create a quit dialog; don't show it yet */
	quit_dialog = XmCreateQuestionDialog(top_wid, "quitDialog", NULL, 0);
	/* remove "Help" button */
	w = XmMessageBoxGetChild(quit_dialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(w);
	XtAddCallback(quit_dialog, XmNokCallback,
		(XtCallbackProc)quit_really, NULL);

	/* create an about dialog; don't show it yet */
	about_dialog = XmCreateInformationDialog(top_wid, "aboutDialog", NULL,
		0);
	/* remove "Cancel" and "Help" buttons */
	w = XmMessageBoxGetChild(about_dialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(w);
	w = XmMessageBoxGetChild(about_dialog, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(w);
	xmstring1 = XmStringCreateLocalized(DEFAULT_TITLE);
	XtVaSetValues(about_dialog, XmNdialogTitle, xmstring1, NULL);
	XmStringFree(xmstring1);
	xmstring2 = XmStringSeparatorCreate();
	xmstring1 = XmStringCreateLocalized(ABOUT_STRING_1);
	xmstring3 = XmStringConcat(xmstring1, xmstring2);
	XmStringFree(xmstring1);
	xmstring1 = XmStringCreateLocalized(ABOUT_STRING_2);
	xmstring4 = XmStringConcat(xmstring3, xmstring1);
	XmStringFree(xmstring1);
	XmStringFree(xmstring3);
	xmstring3 = XmStringConcat(xmstring4, xmstring2);
	XmStringFree(xmstring4);
	xmstring1 = XmStringCreateLocalized(ABOUT_STRING_3);
	xmstring4 = XmStringConcat(xmstring3, xmstring1);
	XmStringFree(xmstring1);
	XmStringFree(xmstring3);
	xmstring3 = XmStringConcat(xmstring4, xmstring2);
	XmStringFree(xmstring4);
	xmstring1 = XmStringCreateLocalized(ABOUT_STRING_4);
	xmstring4 = XmStringConcat(xmstring3, xmstring1);
	XmStringFree(xmstring1);
	XmStringFree(xmstring2);
	XmStringFree(xmstring3);
	XtVaSetValues(about_dialog, XmNmessageString, xmstring4,
		NULL);
	XmStringFree(xmstring4);
	XtAddCallback(about_dialog, XmNokCallback,
		(XtCallbackProc)about_unmanage, NULL);

	/* create a license dialog; don't show it yet */
	license_dialog = XmCreateFormDialog(top_wid, "licenseDialog",
		NULL, 0);

	w = XmCreateScrolledText(license_dialog, "licenseText", NULL, 0);
	XtVaSetValues(w,
		XmNvalue, license,
		XmNeditable, False,
		XmNcursorPositionVisible, False,
		NULL);
	XtVaSetValues(XtParent(w),
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_NONE,
		NULL);
	XtManageChild(w);

	w = XtVaCreateManagedWidget("separator",
		xmSeparatorWidgetClass, license_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, XtParent(w),
		NULL);

	w = XtVaCreateManagedWidget("dismiss",
		xmPushButtonWidgetClass, license_dialog,
		XmNleftAttachment, XmATTACH_FORM,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, w,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL);

	XtAppMainLoop(app);

	exit(EXIT_SUCCESS);
	/* NOTREACHED */
	return 0;
}
