"""
pyFTPclientS60 - an FTP client for S60 phones written in Python.

This program requires Python for Symbian S60 phones.
"""

import sys
import os.path
import time
import appuifw
import e32
import series60_console
import pyUtilsS60
import socket
import ftplib
import ftpconfig

from key_codes import EKeyLeftArrow
from key_codes import EKeyRightArrow

global options_menu_h
global ftpcons_app_body
global dirlist_app_body
global ftpconn
global disconnect_forced

global debug_file_name
global debug_file_enabled
global debug_ftplib

COMM_EXCEPTIONS = (socket.error, ftplib.error_perm, ftplib.error_temp,
                   ftplib.error_proto, ftplib.error_reply)

STATE_NOT_CONNECTED       = 0
STATE_CONNECTING          = 1
STATE_CONNECTED           = 2
STATE_CONNECTED_DIR_XFER  = 3
STATE_CONNECTED_DATA_XFER = 4

def open_ftp_connection(server=None):
    """
    Call-back function which takes care of opening the FTP connection
    and login procedures.
    """
    global ftpconn
    global disconnect_forced
    def __close__():
        global ftpconn
        global disconnect_forced
        try:
            #ftpconn.quit()
            ftpconn.close()
        except:
            pass
        disconnect_forced = True
    def __exit__():
        sys.exit(0)
        
    global options_menu_h
    global debug_ftplib
    global dirlist_app_body
    dirlist_app_body = None

    if server == None:
        server = appuifw.query(u'Connect to', 'text', u'ftp.')
        if server == None: return
    else:
        server = unicode(server)

    disconnect_forced = False
    options_menu_h.set_connection_state(STATE_CONNECTING)
    options_menu_h.set_call_backs(close=__close__, kill=__exit__)
    options_menu_h.set_menu()
    
    print 'Connecting...'
    try:
        ftpconn = ftplib.FTP()
        ftpconn.connect(server)
        if debug_ftplib:
            #= Set also ftplib debug prints
            ftpconn.set_debuglevel(debug_ftplib)
        if not disconnect_forced:
            ftpconn.login()
    except COMM_EXCEPTIONS, detail:
        appuifw.note(unicode(detail), 'error')
        print 'Connection failed: ' + str(detail)
        disconnect_forced = True
    if not disconnect_forced:
        print 'Connected'    
        appuifw.app.title = unicode(server)
        ch = conn_handler(ftpconn, options_menu_h)
        ch.run()
        try:
            ftpconn.quit()
        except COMM_EXCEPTIONS:
            pass
    appuifw.app.title = u'Not connected'    
    options_menu_h.set_connection_state(STATE_NOT_CONNECTED)
    options_menu_h.set_call_backs()
    options_menu_h.set_menu()
    print 'Connection closed'

def open_favorite_server():
    """
    Select the server to connect from the list of favorite servers.
    """
    ix = appuifw.popup_menu(ftpconfig.servers.get_unicoded_server_lst())
    if not ix == None:
        if not ftpconfig.servers.get_server(ix) == '<not selected>':
            open_ftp_connection(ftpconfig.servers.get_server(ix))

def settings():
    """
    As the name says...
    """
    ftpconfig.ftp_settings()

def show_about_message():
    """
    Show some information regarding the program.
    """
    my_dir = str(os.path.split(appuifw.app.full_name())[0])
    fv = pyUtilsS60.fileViewer('About pyFTPc')
    fv.load(os.path.join(my_dir, 'pyftpcs60about.txt'))
    fv.view()

def show_help_message():
    """
    Show the on-line help document of the program.
    """
    my_dir = str(os.path.split(appuifw.app.full_name())[0])
    fv = pyUtilsS60.fileViewer('Help pyFTPc')
    fv.load(os.path.join(my_dir, 'pyftpcs60help.txt'))
    fv.view()

def debug_setting():
    """
    Set ftplib debug level and enable/disable client debug writing.
    """
    global debug_file_enabled
    global debug_ftplib
    q = appuifw.query(u'Debug message writing?', 'query')
    if q: debug_file_enabled = True
    else: debug_file_enabled = False
    q = appuifw.query(u'FTPLIB debug level (0-2)', 'number')
    if (q == None) or (q <= 0): debug_ftplib = 0
    elif (q >= 2): debug_ftplib = 2
    else: debug_ftplib = 1

class conn_handler:
    """
    User interface functionality while the terminal is
    connected to an FTP server.
    """
    def __init__(self, conn, menu):
        global debug_file_name
        global debug_file_enabled
        if debug_file_enabled:
            self.debug_file = self.__open_file(debug_file_name, 'w')
        else:
            self.debug_file = None
        self.dl_file = None
        self.ul_file = None
        self.dir_entries_all = []
        self.dir_entries_lst = []
        self.myconn = conn
        self.menu_h = menu
        self.script_lock = e32.Ao_lock()
        self.last_typed_dir = u'/pub/'
        self.tx_bytes = 0
        self.latest_download_file_name = None
        
    def __open_file(self, file_name, mode='r'):
        try:
            fo = file(file_name, mode)
        except IOError, detail:
            appuifw.note(u'File opening failed: '+unicode(detail), 'error')
            return None
        return fo

    def __close_file(self, fo):
        if fo: fo.close()

    def __save_data_block(self, block):
        #= Exceptions should be handled...
        global debug_ftplib
        if debug_ftplib == 2:
            print '+ entering __save_data_block()'
        self.tx_bytes += len(block)
        if self.dl_file: self.dl_file.write(block)
        if self.debug_file:
            self.debug_file.write('dl block bytes: ' + str(len(block)) +'\n')
            self.debug_file.flush()
        if debug_ftplib == 2:
            print '+ leaving __save_data_block()'

    def __save_line(self, line):
        global debug_ftplib
        if debug_ftplib == 2:
            print '+ entering __save_line()'
        self.dir_entries_all.append(unicode(line))
        if self.debug_file:
            self.debug_file.write(line+'\n')
            self.debug_file.flush()
        if debug_ftplib == 2:
            print '+ leaving __save_line()'

    def __fetch_directory(self):
        print 'Fetching directory content...'
        if self.debug_file:
            self.debug_file.write('+ __fetch_directory()\n')
            self.debug_file.flush()
        self.dir_entries_all = []
        self.dir_entries_lst = []
        self.myconn.retrlines('LIST', self.__save_line)
        #self.myconn.retrlines('LIST')            
        print 'Directory content fetched'
        #= All the information cannot be shown in one line of 'Listbox'.
        #= For directories and regular files show the file name. For the
        #= links show the name of the link. For example:
        """
        ftp> dir
        227 Entering Passive Mode (127,0,0,1,22,176)
        150 Here comes the directory listing.
        drwxr-xr-x    2 500      500        4096 Aug 05 12:12 Directory
        lrwxrwxrwx    1 500      500           9 Aug 30 06:59 Link -> File.txt
        -rwxrwxr--    1 500      500          756 Aug 01 08:32 File.txt
        226 Directory send OK.
        """
        #= Any other types of entries possible? Yes, certain sites do not
        #= provide group id. Also certain sites send 'total XX' line as a
        #= first line.
        if len(self.dir_entries_all) == 0:
            self.dir_entries_lst.append(u'<empty>')
            return
        for entry in self.dir_entries_all:
            split_entry = entry.split()
            if len(split_entry) < 7:
                #= Weird entries... 'total XX' and similar stuff.
                self.dir_entries_lst.append('- '+entry)
            elif split_entry[0][0] == 'l':
                #= Link
                self.dir_entries_lst.append('l '+split_entry[-3])
            elif split_entry[0][0] == 'd':
                #= Directory
                self.dir_entries_lst.append('d '+split_entry[-1])
            else:
                #= File
                self.dir_entries_lst.append('f '+split_entry[-1])
                
    def __refresh_directory(self):
        self.menu_h.set_connection_state(STATE_CONNECTED_DIR_XFER)
        self.menu_h.set_menu()
        try:
            self.__fetch_directory()
        except COMM_EXCEPTIONS, detail:
            print 'Error: ' + str(detail)
            appuifw.note(unicode(detail), 'error')
            #= If an error occurred while fetching the content
            #= lets handle the directory as it was empty.
            self.dir_entries_all = []
            self.dir_entries_lst = []
            self.dir_entries_lst.append(u'<empty>')
        #= Create a new 'list box' to show the new directory entries.
        self.lbox.set_list(self.dir_entries_lst)
        self.menu_h.set_connection_state(STATE_CONNECTED)
        self.menu_h.set_menu()

    def __refresh(self):
        self.__set_dir_list_visibility(False)
        self.__refresh_directory()
        self.__set_dir_list_visibility(True)

    def __starting(self):
        self.old_app = pyUtilsS60.save_current_app_info()
        self.menu_h.set_connection_state(STATE_CONNECTED)
        self.menu_h.set_call_backs(self.__set_dir_list_visibility,
                                   self.__chdir,
                                   self.__upload_file,
                                   self.__refresh,
                                   self.__disconnect,
                                   self.__kill_program,
                                   self.__abort_xfer)
        self.menu_h.set_menu()
        #= What would be the best way to use the 'exit' key.
        appuifw.app.exit_key_handler = self.__disconnect
        
    def __done(self):
        global dirlist_app_body
        self.__close_file(self.debug_file)
        self.menu_h.set_connection_state(STATE_NOT_CONNECTED)
        self.menu_h.set_call_backs()
        self.menu_h.set_menu()
        dirlist_app_body = None
        self.lbox = None
        pyUtilsS60.restore_app_info(self.old_app)

    def __disconnect(self):
        self.script_lock.signal()

    def __kill_program(self):
        """
        Force program termination in the case the connection 'hangs' and
        nothing else helps.
        """
        sys.exit(0)

    def __abort_xfer(self):
        print 'Aborting...'
        try:
            self.myconn.abort()
        except COMM_EXCEPTIONS, detail:
            print 'abort: ' + str(detail)

    def __process_user_evt(self, act=None):
        #= For details of the received info refer to __fetch_directory().
        if len(self.dir_entries_all) == 0:
            entry_type = None
        else:
            split_entry = self.dir_entries_all[self.lbox.current()].split()
            if len(split_entry) < 7:
                #= Weird entries... 'total XX' and similar stuff.
                entry_type = None
            elif split_entry[0][0] == 'l':
                entry_type = 'LINK'
            elif split_entry[0][0] == 'd':
                entry_type = 'DIR'
            else:
                entry_type = 'FILE'

        if act == None:
            #= The user has pressed joystick.
            if entry_type == None: return
            ix = None
            ix = appuifw.popup_menu([u'Show Details', u'Download',
                                     u'Read on-line'] )
            if ix == None: return

            if ix == 0:
                if entry_type == 'LINK':
                    info = 'Name: '+\
                           split_entry[-3]+' '+\
                           split_entry[-2]+' '+\
                           split_entry[-1]+'\n'
                    index_count = -3
                else:
                    info = 'Name: '+\
                           split_entry[-1]+'\n'
                    index_count = -1
                info += 'Date: '+\
                        split_entry[index_count-3]+' '+\
                        split_entry[index_count-2]+' '+\
                        split_entry[index_count-1]+'\n'
                index_count -= 3
                info += 'Size: '+\
                        split_entry[index_count-1]+'\n'
                index_count -= 1
                info += 'Perm:'
                for i in range(len(split_entry)+index_count):
                    info += ' ' + split_entry[i]
                info += '\n'
                fv = pyUtilsS60.fileViewer('File details')
                fv.view(info)
            elif (ix == 1) or (ix == 2):
                if entry_type == 'LINK':
                    self.__download_file(split_entry[-3])
                elif entry_type == 'FILE':
                    self.__download_file(split_entry[-1])
                else:
                    appuifw.note(u'Cannot download directory', 'error')
                    return
                if (ix == 2) and (not self.latest_download_file_name == None):
                    fv = pyUtilsS60.fileViewer(joystick=True)
                    fv.load(self.latest_download_file_name)
                    fv.view()
        elif act == 'back':
            #= The user has turned joystick left.
            self.__chdir_command('..')
        elif act == 'next' and \
                 (entry_type == 'LINK' or entry_type == 'DIR'):
            #= The user has turned joystick right.
            self.__chdir_command(split_entry[-1])
        return    

    def __download_file(self, dfile):
        self.latest_download_file_name = None
        dst_file = os.path.join(ftpconfig.directories.get_download_dir(),
                                dfile)
        self.dl_file = self.__open_file(dst_file, 'w')
        if self.dl_file == None: return

        #= In order to make prints appear correctly
        self.__set_dir_list_visibility(False)

        appuifw.note(u'Downloading...', 'conf')
        print 'Downloading...'
        if self.debug_file:
            self.debug_file.write('+ __download_file()\n')
            self.debug_file.flush()
        self.tx_bytes = 0
        start_time = time.time()
        self.menu_h.set_connection_state(STATE_CONNECTED_DATA_XFER)
        self.menu_h.set_menu()
        try:
            cmd = 'RETR '+dfile
            self.myconn.retrbinary(cmd, self.__save_data_block)
        except COMM_EXCEPTIONS, detail:
            print 'Error: ' + str(detail)
            appuifw.note(unicode(detail), 'error')
        else:
            end_time = time.time()
            tx_time = end_time - start_time
            if tx_time == 0.0: speed = 0.0
            else:              speed = round((self.tx_bytes / tx_time), 2)
            appuifw.note(u'Download completed.', 'info')
            print 'Download completed:\n\t' + str(tx_time) + ' sec\n\t' + \
                  str(self.tx_bytes) + ' bytes\n\t' + \
                  str(speed) + ' bytes/sec'
            self.latest_download_file_name = dst_file
        self.menu_h.set_connection_state(STATE_CONNECTED)
        self.menu_h.set_menu()
        self.__close_file(self.dl_file)
        self.dl_file = None
        #= Switch dir listing visible. It must have been visible before
        #= downloading was started.
        self.__set_dir_list_visibility()

    def __upload_file(self):
        fb = pyUtilsS60.dirBrowser()
        ufile = fb.select(ftpconfig.directories.get_upload_dir())
        if ufile == None:
            return
        ul_file_name = os.path.join(ufile[0], ufile[1])
        if os.path.isdir(ul_file_name):
            appuifw.note(u'Not a file', error)
            return
        try:
            self.ul_file = self.__open_file(ul_file_name, 'r')
        except IOError, detail:
            appuifw.note(u'File opening failed', unicode(detail))
            return False

        #= Save current 'dir listing visibility', set dir listing
        #= invisible, and once uploading is completed set visibility state
        #= back to what is was before uploading started.
        global ftpcons_app_body
        global dirlist_app_body
        set_visible = False
        if appuifw.app.body == dirlist_app_body:
            self.__set_dir_list_visibility(False)
            set_visible = True

        appuifw.note(u'Uploading...', 'conf')
        print 'Uploading...'
        if self.debug_file:
            self.debug_file.write('+ __upload_file()\n')
            self.debug_file.flush()
        self.tx_bytes = 0
        start_time = time.time()
        self.menu_h.set_connection_state(STATE_CONNECTED_DATA_XFER)
        self.menu_h.set_menu()
        try:
            cmd = 'STOR ' + ufile[1]
            self.myconn.storbinary(cmd, self.ul_file)
        except COMM_EXCEPTIONS, detail:
            print 'Error ' + str(detail)
            appuifw.note(unicode(detail), 'error')
        else:
            end_time = time.time()
            tx_time = end_time - start_time
            #= Find size of the file. The position should be end of file since
            #= all data has been sent. Thus the size of sent data can be solved
            #= using tell().
            self.tx_bytes = self.ul_file.tell()
            if tx_time == 0.0: speed = 0.0
            else:              speed = round((self.tx_bytes / tx_time), 2)
            appuifw.note(u'Upload completed.', 'info')
            print 'Upload completed:\n\t' + str(tx_time) + ' sec\n\t' + \
                  str(self.tx_bytes) + ' bytes\n\t' + \
                  str(speed) + ' bytes/sec'
        self.__refresh_directory()
        #self.menu_h.set_connection_state(STATE_CONNECTED)
        #self.menu_h.set_menu()
        self.__close_file(self.ul_file)
        self.ul_file = None
        if set_visible == True:
            self.__set_dir_list_visibility(True)

    def __chdir_command(self, directory):
        self.__set_dir_list_visibility(False)
        print 'Changing directory...'
        self.menu_h.set_connection_state(STATE_CONNECTED_DIR_XFER)
        self.menu_h.set_menu()
        try:
            self.myconn.cwd(directory)
            wd = self.myconn.pwd()
            print 'Directory: ' + str(wd)
        except COMM_EXCEPTIONS, detail:
            #= The directory could not be changed. Keep the original
            #= directory listing since we are still in that directory.
            print 'Error: ' + str(detail)
            appuifw.note(unicode(detail), 'error')
        except:
            print 'Some other error...'
        else:
            self.__refresh_directory()
        global dirlist_app_body
        dirlist_app_body = self.lbox
        self.__set_dir_list_visibility(True)

    def __chdir(self):
        dir = appuifw.query(u'Change to', 'text', self.last_typed_dir)
        if dir == None: return
        self.last_typed_dir = dir
        self.__chdir_command(dir)

    def __set_dir_list_visibility(self, state=None):
        """
        Set directory listing visible or invisible. If state is not
        given then simply switch the state.
        """
        global ftpcons_app_body
        global dirlist_app_body
        if dirlist_app_body == None:
            #= No dir listing fetched yet
            return
        if state == None:
            #= Switch visibility.
            if appuifw.app.body == ftpcons_app_body:
                appuifw.app.body = dirlist_app_body
            else:
                appuifw.app.body = ftpcons_app_body
        elif state == False:
            #= Set directory listing invisible
            appuifw.app.body = ftpcons_app_body
        else:
            #= Set directory listing visible
            appuifw.app.body = dirlist_app_body

    def run(self):
        self.__starting()
        self.menu_h.set_connection_state(STATE_CONNECTED_DIR_XFER)
        self.menu_h.set_menu()
        try:
            self.__fetch_directory()
        except COMM_EXCEPTIONS, detail:
            print 'Error ' + str(detail)
            appuifw.note(unicode(detail), 'error')
            self.__done()
            return
        self.menu_h.set_connection_state(STATE_CONNECTED)
        self.menu_h.set_menu()
        self.lbox = appuifw.Listbox(self.dir_entries_lst, \
                                    self.__process_user_evt)
        self.lbox.bind(EKeyLeftArrow, lambda:self.__process_user_evt('back'))
        self.lbox.bind(EKeyRightArrow, lambda:self.__process_user_evt('next'))
        global dirlist_app_body
        dirlist_app_body = self.lbox
        self.__set_dir_list_visibility(True)
        self.script_lock.wait()
        self.__done()

class options_menu:
    """
    Handle the main options menu. The menu is 'dynamic', i.e., it changes
    according to the operations performed by the user.
    """
    def __init__(self):
        self.conn_state = STATE_NOT_CONNECTED
        self.cb_dir_list = None
        self.cb_chdir = None
        self.cb_upload = None
        self.cb_refresh = None
        self.cb_close = None
        self.cb_kill = None
        self.cb_abort_xfer = None
            
    def set_connection_state(self, state):
        self.conn_state = state

    def set_call_backs(self, dir_list=None, chdir=None, upload=None,
                       refresh=None, close=None, kill=None, abort=None):
        self.cb_dir_list = dir_list
        self.cb_chdir = chdir
        self.cb_upload = upload
        self.cb_refresh = refresh
        self.cb_close = close
        self.cb_kill = kill
        self.cb_abort_xfer = abort
    
    def set_menu(self):
        """
        Create and set main options menu. Also the joystick functionality
        will be set according to the connection state.
        """
        if self.conn_state == STATE_NOT_CONNECTED:
            #= Menu in the case the client is not connected to any server.
            appuifw.app.menu = [(u'Connect', open_ftp_connection), \
                                (u'My servers', open_favorite_server),
                                (u'Settings', settings)]
            appuifw.app.menu.append((u'Debug', debug_setting))
        elif self.conn_state == STATE_CONNECTED:
            #= Menu in the case the client is connected to a server but
            #= not communicating with it.
            appuifw.app.menu = [(u'Show/hide listing', self.cb_dir_list),
                                (u'Change dir', self.cb_chdir),
                                (u'Upload', self.cb_upload),
                                (u'Refresh', self.cb_refresh),
                                (u'Disconnect',  self.cb_close),
                                (u'Kill program',  self.cb_kill)]
        elif self.conn_state == STATE_CONNECTING:
            #= Menu in the case the client is connecting to a server.
            appuifw.app.menu = [(u'Disconnect',  self.cb_close),
                                (u'Kill program',  self.cb_kill)]            
        else:
            #= Menu in the case the client is connected to a server and
            #= really communicating with it.
            appuifw.app.menu = [(u'Abort transfer', self.cb_abort_xfer),
                                (u'Disconnect',  self.cb_close),
                                (u'Kill program',  self.cb_kill)]
        #= These menu items are always present.
        appuifw.app.menu.append((u'About', show_about_message))
        appuifw.app.menu.append((u'Help', show_help_message))

def ftp_for_s60(mypath):
    """
    Core functionality.
    """
    global options_menu_h
    global ftpcons_app_body
    global dirlist_app_body
    global debug_file_name
    global debug_file_enabled
    global debug_ftplib
        
    #= Check if the currently installed pyUtilsS60 module version is
    #= compatible with this version of pyEdit.
    if pyUtilsS60.version_compatibility((0,1)) == False:
        msg = 'pyUtilsS60\nversion %d.%d.%d\ntoo old' % pyUtilsS60.version
        appuifw.note(unicode(msg), 'error')
        return
    
    #= Save current application information and define the actions to be
    #= taken while exiting from the program.
    orig_app_gui_info = pyUtilsS60.save_current_app_info()
    orig_stderr = sys.stderr
    orig_stdout = sys.stdout
    def my_exit():
        print 'FTP exiting...'
        pyUtilsS60.restore_app_info(orig_app_gui_info)
        sys.stderr = orig_stderr
        sys.stdout = orig_stdout
        escript_lock.signal()

    #= Create a console where the most of the messages can be printed.
    #= It is used as body when not connected to any server.
    ftp_console = series60_console.Console()
    ftpcons_app_body = ftp_console.text
    sys.stderr = sys.stdout = ftp_console
    appuifw.app.body = ftpcons_app_body
    dirlist_app_body = None

    #= Set file name for debug information saving
    debug_file_name = os.path.join(mypath, 'dat/ftpdebugfile.txt')
    debug_file_enabled = False
    debug_ftplib = 0

    escript_lock = e32.Ao_lock()
    appuifw.app.exit_key_handler = my_exit
    appuifw.app.title = u'Not connected'

    #= Initialize the settings module and read the default settings.
    ftpconfig.init_ftp_settings(mypath)
    ftpconfig.read_ftp_settings()

    #= Initialize the options menu.
    options_menu_h = options_menu()
    options_menu_h.set_menu()

    print 'This is a simple FTP implementation for\nSeries 60\nReady...'
    #= Wait for exiting...
    escript_lock.wait()
    
if __name__ == '__main__':
    mypath = os.path.join(os.path.split(appuifw.app.full_name())[0], 'my')
    ftp_for_s60(mypath)
