
import os
import appuifw
import e32
import dir_iter

#=
#= The current version number of the pyUtilsS60 module.
#=
version = (0,1,0)

def version_compatibility(required_version):
    """
    Check if the version of the pyEditS60 module is older than required by
    the application utilizing the module. Only the first two most significant
    numbers are checked. The last number only indicates the bug fix level and
    thus compatibility is not an issue.
    """
    if (required_version[0] < version[0]):
        return True
    if (required_version[0] == version[0]):
        if (required_version[1] <= version[1]):
            return True
        else:
            return False
    else:
        return False

#=
#= Functions to save and restore application GUI data.
#=
def save_current_app_info():
    """
    Save the current application GUI data.
    """
    app_info = []
    app_info.append(appuifw.app.title)
    app_info.append(appuifw.app.body)
    app_info.append(appuifw.app.menu)
    app_info.append(appuifw.app.exit_key_handler)
    # The ones above seem to be present always but
    # the appuifw.app.t is sometimes missing.
    try:
        app_info.append(appuifw.app.t)
    except AttributeError:
        app_info.append(None)
    return app_info

def restore_app_info(app_info):
    """
    Restore the saved application GUI data.
    """
    appuifw.app.title            = app_info[0]
    appuifw.app.body             = app_info[1]
    appuifw.app.menu             = app_info[2]
    appuifw.app.exit_key_handler = app_info[3]
    appuifw.app.t                = app_info[4]
    return

#=
#= Directory browser
#=
class dirBrowser:
    """
    Browse the S60 phone directory structure interactively using joystick and
    select a file and/or directory. The user may also define an options
    menu with callback functions.
    """
    def __init__(self, title='Select', menu=None):
        self.my_title = unicode(title)
        self.script_lock = e32.Ao_lock()
        self.dir_stack = []
        self.current_dir = dir_iter.Directory_iter(e32.drive_list())
        self.dir_empty = False
        self.selection = None
        self.user_menu = menu
        
    def __exit_key_handler(self):
        appuifw.app.exit_key_handler = None # Not needed
        self.script_lock.signal()
        
    def __selecting(self):
        """
        Save the GUI parameters of the 'parent program' and set the new ones.
        """
        self.old_app = save_current_app_info()

        appuifw.app.title = self.my_title
        if self.user_menu == None:
            appuifw.app.menu = []
        else:
            appuifw.app.menu = self.user_menu
        appuifw.app.exit_key_handler = self.__exit_key_handler
        
    def __selected(self):
        """
        Restore the GUI parameters of the 'parent program'.
        """
        restore_app_info(self.old_app)
        self.lbox = None
        
    def select(self, default_path=None):
        """
        The selection will return a tuple containing the selected directory
        file item. The selected file item may also be a directory and
        either part may be empty.

        If the user does not select anything (presses the exit key)
        the return value will be None.
        """
        self.__selecting()
        from key_codes import EKeyLeftArrow
        from key_codes import EKeyRightArrow
        entries = self.current_dir.list_repr()
        
        self.lbox = appuifw.Listbox(entries, self.__process_user_evt)
        self.lbox.bind(EKeyLeftArrow, lambda: self.__process_user_evt('back'))
        self.lbox.bind(EKeyRightArrow, lambda: self.__process_user_evt('next'))
        appuifw.app.body = self.lbox

        self.__open_default_directory(default_path)
        
        self.script_lock.wait()
        self.__selected()
        return self.selection

    def __process_user_evt(self, dir=None):
        if self.dir_empty:
            index = 0
        else:
            index = self.lbox.current()
        if dir == None:
            # The user has made up her mind...
            # If on the root level, no selection made!
            focused_item = 0
            if not self.current_dir.at_root:
                if self.dir_empty == False:
                    item = self.current_dir.entry(index)
                    self.selection = os.path.split(item)
                else:
                    # If the directory is empty we have to do a little trick.
                    # Lets go to the parent directory and then the focused
                    # item is our current directory path.
                    index = self.dir_stack.pop()
                    self.current_dir.pop()
                    item = self.current_dir.entry(index)
                    self.selection = (item, '')
            self.__exit_key_handler()

        elif dir == 'back':
            # Go to the parent directory unless already in the root directory.
            if not self.current_dir.at_root:
                focused_item = self.dir_stack.pop()
                self.current_dir.pop()
            else:
                focused_item = index

        elif dir == 'next':
            # Go to the focused sub-directory provided it's a directory.
            if self.current_dir.at_root:
                self.dir_stack.append(index)
                self.current_dir.add(index)
                focused_item = 0
            elif self.dir_empty:
                focused_item = 0
            elif os.path.isdir(self.current_dir.entry(index)):
                self.dir_stack.append(index)
                self.current_dir.add(index)
                focused_item = 0
            else:
                focused_item = index

        entries = self.current_dir.list_repr()
        
        if entries == []:
            entries.insert(0, (u'<empty>', u''))
            self.dir_empty = True
        else:
            self.dir_empty = False
        self.lbox.set_list(entries, focused_item)
        return

    def __open_default_directory(self, dir=None):
        """
        Parse and set the default directory. If the default directory is not
        given, it does not exist, or there are problems parsing the given
        path then default to the drive selection.
        """
        if dir == None:
            return
        dir = os.path.normpath(dir)
        if not os.path.isdir(dir):
            # Should raise an exception!
            print 'ERROR: Given default path not a directory'
            return
        drv, path = os.path.splitdrive(dir)

        index = self.__get_index(drv)
        if index == None:
            # Would be a bug!
            self.dir_stack = []
            self.current_dir = dir_iter.Directory_iter(e32.drive_list())
            return 
        self.dir_stack.append(index)
        self.current_dir.add(index)

        dir_items = path.split('\\')
        for item in dir_items:
            if item == u'': continue
            index = self.__get_index(u'['+item+u']')
            if index == None:
                # Would be a bug!
                self.dir_stack = []
                self.current_dir = dir_iter.Directory_iter(e32.drive_list())
                return 
            self.dir_stack.append(index)
            self.current_dir.add(index)

        focused_item = 0
        entries = self.current_dir.list_repr()
        if entries == []:
            entries.insert(0, (u'<empty>', u''))
            self.dir_empty = True
        else:
            self.dir_empty = False        
        self.lbox.set_list(entries, focused_item)

        return
    
    def __get_index(self, item):
        """
        Find the directory list index of a file or directory name in
        the current directory.
        """
        index = 0
        entries = self.current_dir.list_repr()
        for entry in entries:
            if entry[0] == item:
                return index
            index += 1
        return None

#=
#= File viewer
#=
class fileViewer:
    """
    Show a long message on the screen. The message may be given as a
    parameter or it can be read from a file.
    """
    def __init__(self, title=None):
        if title: self.my_title = unicode(title)
        else:     self.my_title = None
        self.script_lock = e32.Ao_lock()
        self.loaded_text = None
        
    def __exit_key_handler(self):
        appuifw.app.exit_key_handler = None # Not needed
        self.script_lock.signal()

    def __set_new_gui(self):
        """
        Save the GUI parameters of the 'parent program' and set the new ones.
        """
        self.old_app = save_current_app_info()
        if self.my_title:
            appuifw.app.title = self.my_title
        appuifw.app.t = appuifw.Text(u'')
        appuifw.app.body = appuifw.app.t
        appuifw.app.menu = []
        appuifw.app.exit_key_handler = self.__exit_key_handler

    def __read_file(self, file_name):
        mfile = file(file_name, 'r')
        msg = mfile.read()  # Note: reads everything!
        mfile.close()
        return msg.decode('iso-8859-1')

    def load(self, file_name=None):
        """
        Load the message to be viewed from a file.
        """
        if file_name == None: return
        fname = os.path.normpath(file_name)
        if not os.path.isfile(fname):
            # Should raise an exception!
            print 'ERROR: Given file ' + str(file_name) + ' not a regular file'
            return
        try:
            self.loaded_text = self.__read_file(fname)
            # The exception should be catched by the main app!!!
        except UnicodeError, detail:
            appuifw.note(u'Error while reading!\n' + unicode(detail),
                         'error')
        #except:
        #    appuifw.note(u'Error while reading!', 'error')

    def view(self, text=None):
        """
        View the content of the selected file or given text string.
        """
        if not text == None:
            # Message to be shown given as a parameter.
            txt = unicode(text)
        elif self.loaded_text == None:
            # No message to show.
            return
        else:
            # Message loaded from a file.
            txt = self.loaded_text
        self.__set_new_gui()
        appuifw.app.t.set(txt)
        appuifw.app.t.set_pos(0)
        self.script_lock.wait()
        restore_app_info(self.old_app)

#=
#= Stand-alone testing
#=
def stand_alone_test_dirBrowser():
    fb = dirBrowser('Select a file or directory...')
    selection = fb.select(u'C:\\Nokia\\')
    if selection == None:
        print 'Nothing selected'
    else:
        print 'Selected path', str(selection[0])
        print 'Selected file', str(selection[1])

if __name__ == '__main__':
    stand_alone_test_dirBrowser()
