"""
pyExchangeS60 - Exchange Rate Converter

A program for converting currencies based on up to date exchange rates
that can be downloaded from the Internet.

This program requires Python for Symbian S60 phones.
"""

import os.path
import sys
import socket
import appuifw
import e32
import pyUtilsS60

#=
#= The remote host
#=
HOST = 'www1.nordea.fi'
#=
#= The http port used by the server
#=
PORT = 80
#=
#= This is the required 'http get command'
#=
COMMAND = 'GET /s/merita/kurssit/val-tilival_eur.asp HTTP/1.0\r\nUser-Agent: Wget/1.8.2\r\nHost: www.nordea.fi\r\nAccept: */*\r\nConnection: Keep-Alive\r\n\r\n'
###############################################################################

class rate_loader:
    """
    Handles communication with the server offering
    the exchange rate information.
    """
    
    def __init__(self, local_file, server_host, server_port, http_get_command):
        self.local_file = local_file
        self.command = http_get_command
        self.host = server_host
        self.port = server_port
        
    def download(self):
        """
        This method performs the actual downloading.
        """
        f = file(self.local_file, 'w+')        
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        s.connect((self.host, self.port))
        s.send(self.command)

        #print 'Starting to download...'
        while 1:
            data = s.recv(1024)
            if not data: break
            try:
                f.write(data)
            except IOError, detail:
                print 'Error (' + str(detail) + ') in writing file ' + \
                      self.file
                break
        #print 'download completed.'
        s.close()
        f.close()

###############################################################################

class rate_extractor:
    """
    A class for extracting the exchange rate information from
    the downloaded file.
    """

    def __init__(self, data_file):
        self.file_obj  = None
        self.file_name = data_file

    def open(self):
        """
        Open the file containing the exchange rate data.
        """
        try:
            self.file_obj = file(self.file_name) # only for reading
        except IOError, detail:
            raise IOError, detail
        
    def close(self):
        """
        Close the file containing the exchange rate data.
        """
        if not self.file_obj == None:
            self.file_obj.close()
            self.file_obj = None

    def getDataDate(self):
        """
        Returns a string containing the date when exchange rates were updated.
        """
        data_date = None
        if self.file_obj == None:
            return data_date
        self.file_obj.seek(0)
        while 1:
            s = self.file_obj.readline()
            if s == '': break
            if not (s.find('Updated') == -1):
                data_date = s.split()[1]
                break
        return data_date

    def getExchangeRate(self, currency):
        """
        Returns a string containing exchange rate against EUR for
        the currency input in the argument 'currency'.
        """        
        # The parameter 'currency' may be unicoded.
        # At least Amaretto (Nokia Python) does not handle the
        # stuff below correctly if the normal and
        # unicode string are used in mix =>
        # lets use ASCII strings here.
        c = str(currency)
        rate = None
        if self.file_obj == None:
            return rate
        self.file_obj.seek(0)
        while 1:
            s = self.file_obj.readline()
            if s == '': break
            if not (s.find(c) == -1):
                rate = s.split()[1]           # This is the middle rate
                rate = rate.replace(',', '.') # Can convert string to float
                break
        return rate

    def getAllExchangeRates(self):
        list_of_rates = []
        """
        Returns a list of tuples where each tuple contains name of
        the currency and exchange rate against EUR.
        """
        # EUR is the first line in the table of exchange
        # rates. Now we search for the EUR and then we
        # read all the lines until we hit 'XDR' which is
        # the last one in the table.
        self.getExchangeRate('EUR')
        while 1:
            s = self.file_obj.readline()
            if s == '': break
            c = s.split()
            tmp = c[1].replace(',', '.')
            list_of_rates.append((c[0], tmp))
            if c[0] == 'XDR':
                break
        return list_of_rates

###############################################################################
        
class s60_main:
    """
    For S60 phones.
    """
    def __init__(self, script_dir):
        global HOST
        global PORT
        global COMMAND
        self.host = HOST
        self.port = PORT
        self.command = COMMAND
        self.extractor = None
        self.script_lock = e32.Ao_lock()
        self.old_app_data = pyUtilsS60.save_current_app_info()

        full_dir = os.path.join(script_dir, 'dat')
        if not os.path.isdir(full_dir):
            try:
                os.mkdir(full_dir)
            except:
                type, value = sys.exc_info() [:2]
                appuifw.note(unicode(str(type)+'\n'+str(value)), 'error')

        self.script_dir = script_dir
        self.local_file = os.path.join(full_dir, 'exchangerates.txt')

        
    def __exit_key_handler(self):
        """
        Things to do while exiting.
        """
        try:
            self.extractor.close()
        finally:
            # The lock has to be released as well as the exit_key_handler
            # must be set to None no matter what happens above.
            # Otherwise the program never exits properly!
            #appuifw.app.exit_key_handler = None
            pyUtilsS60.restore_app_info(self.old_app_data)
            self.script_lock.signal()

    def __download_exchange_rate_info(self):
        """
        Handle downloading of the latest exchange rate data.
        """
        appuifw.note(u'Starting to download', 'info')
        self.extractor.close()
        loader = \
               rate_loader(self.local_file, self.host, self.port, self.command)
        try:
            loader.download()
        except IOError, detail:
            appuifw.note(unicode(detail), 'error')
            return False
        except socket.error, detail:
            err = u'socket():' + unicode(detail)
            appuifw.note(err, 'error')
            return False
        appuifw.note(u'Downloading completed', 'info')
        try:
            self.extractor.open()
        except IOError, detail:
            appuifw.note(unicode(detail), 'error')
            return False
        return True
        
    def run(self):
        """
        Main program when run in S60 phone
        """        
        old_title = appuifw.app.title
        appuifw.app.title = u'pyExchangeS60'
        appuifw.note(u'pyExchangeS60 started', 'conf')

        self.extractor = rate_extractor(self.local_file)
        try:
            self.extractor.open()
        except IOError, detail:
            appuifw.note(u'Exchange rate data not available. Downloading it.',\
                         'info')
            if self.__download_exchange_rate_info() == False:
                # If we cannot download the data lets exit the program.
                # This can be sending signal for the 'script_lock'
                # even before script_lock.wait().
                self.script_lock.signal()
                print 'Sorry, cannot continue! Exiting...'
        
        appuifw.app.menu = \
                         [(u'Convert currencies', self.convert_currencies),
                          (u'Show selected rate', self.show_selected_rate),
                          (u'Show all rates', self.show_all_rates),
                          (u'Show data date', self.show_data_date),
                          (u'Update rates', self.__download_exchange_rate_info),
                          (u'Help', self.__help),
                          (u'About', self.__about),
                          (u'Exit', self.__exit_key_handler)]
        appuifw.app.exit_key_handler = self.__exit_key_handler
        self.script_lock.wait()

    def show_data_date(self):
        """
        Displays the date when the exchange rates were updated.
        """
        data_date = self.extractor.getDataDate()
        update_info = 'Exchange rates dated on ' + data_date
        appuifw.note(unicode(update_info), 'info')

    def convert_currencies(self):
        """
        Converts different currencies.
        """
        s = ' '
        t = ' '
        amount = 1
        while 1:
            p = unicode(s)
            s = appuifw.query(u'Give source currency', 'text', p)
            if s == None: break
            s = s.upper()
            srate = self.extractor.getExchangeRate(s)
            if srate == None:
                err = u'Currency ' + s + u' not available'
                appuifw.note(err, 'error')
                continue

            p = unicode(t)
            t = appuifw.query(u'Give target currency', 'text', p)
            if t == None: break
            t = t.upper()
            trate = self.extractor.getExchangeRate(t)
            if trate == None:
                err = u'Currency ' + t + u' not available'
                appuifw.note(err, 'error')
                continue

            p = amount
            amount = appuifw.query(u'Amount to convert', 'number', p)
            if amount == None: break

            rs = float(srate)
            rt = float(trate)
            r = rt/rs
            #a = s + u'->' + t
            #b = unicode(r)
            f = appuifw.Form([(s,    'text', unicode(str(amount))),
                              (u'=', 'text', u''),
                              (t,    'text', unicode(str(r*amount)))],
                             appuifw.FFormViewModeOnly)
            f.execute()
        return

    def show_selected_rate(self):
        """
        Shows exchange rate of selected currency against EUR.
        """
        s = 'USD'
        while 1:
            p = unicode(s)
            s = appuifw.query(u'Give currency', 'text', p)
            if s == None: break

            s = s.upper()
            rate = self.extractor.getExchangeRate(s)
            if rate == None:
                err = u'Currency ' + s + u' not available'
                appuifw.note(err, 'error')
                continue
            r = float(rate)
            if r == 0.0: continue
            q = 1/r
            
            f = appuifw.Form([(u'EUR', 'text', u''),
                              (u'to',  'text', unicode(r)),
                              (s,      'text', u''),
                              (s,      'text', u''),
                              (u'to',  'text', unicode(q)),
                              (u'EUR', 'text', u'')],
                             appuifw.FFormViewModeOnly)
            f.execute()
        return

    def __display_page_of_rates(self, form_data):
        """
        Ref. show_all_rates()
        """
        form_data.append((u'Back', 'text', u'to continue'))
        f = appuifw.Form(form_data,
                         appuifw.FFormViewModeOnly)
        f.execute()

    def show_all_rates(self):
        """
        Shows all the available exchange rates against EUR in table format.
        """
        nr_of_lines_per_page = 5
        list_of_rates = self.extractor.getAllExchangeRates()
        form_data = []
        k = 0
        for i, j in list_of_rates:
            form_data.append((unicode(i), 'text', unicode(j)))
            k = k + 1
            if (k % 5) == 0:
                self.__display_page_of_rates(form_data)
                form_data = []
        if not (k % 5) == 0:
            # The last (non-full) page
            self.__display_page_of_rates(form_data)
        return

    def __help(self):
        fv = pyUtilsS60.fileViewer('Help on pyExchangeS60')
        fv.load(os.path.join(self.script_dir, 'exchanges60help.txt'))
        fv.view()

    def __about(self):
        fv = pyUtilsS60.fileViewer('About pyExchangeS60')
        fv.load(os.path.join(self.script_dir, 'exchanges60about.txt'))
        fv.view()

###############################################################################

if __name__ == '__main__':
    mypath = os.path.join(os.path.split(appuifw.app.full_name())[0], 'my')
    m = s60_main(mypath)
    m.run()
