"""
sms_ui.py - yet another SMS (short message service) user interface.

Copyright (c) 2006 - 2007 Lasse Huovinen
"""

import e32
import appuifw
import messaging
import contacts
import inbox
import os.path
import pyS60uiutil

class NumberFromContacts:
    def __init__(self):

        self.app_info = pyS60uiutil.save_current_app_info()
        appuifw.app.screen = 'normal'
        appuifw.app.menu = [(u'Select', self.__selected),
                            (u'Cancel', self.__canceled)]
        self.sms_db_lock = e32.Ao_lock()
        self.recipients = []
        self._lbox = None
        self.selected_number = None

    def __selected(self):
        nbrs = []
        for i in self.recipients[self._lbox.current()][1]:
            nbrs.append(unicode(i.value))
        ix = appuifw.popup_menu(nbrs)
        if not ix == None:
            self.selected_number = nbrs[ix]
            self.sms_db_lock.signal()

    def __canceled(self):
        self.selected_number = None
        self.sms_db_lock.signal()

    def select(self):
        """
        Fetch all the phone numbers from the contacts database and ask
        the user to select one.
        """
        self.selected_number = None
        db = contacts.open()        
        self.recipients = []
        for id in db:
            contact = db[id]
            nr_list = []
            nr_list += contact.find(type='mobile_number')
            nr_list += contact.find(type='phone_number')
            if len(nr_list) > 0:
                self.recipients.append((contact.title, nr_list))
        if len(self.recipients) == 0:
            appuifw.note(u'No numbers in Contacts', 'error')
        else:
            names = []
            self.recipients.sort() #= This is cool...
            for r in self.recipients:
                names.append(unicode(r[0]))
            self._lbox = appuifw.Listbox(names, self.__selected)
            appuifw.app.body = self._lbox
            appuifw.app.exit_key_handler = self.__canceled
            self.sms_db_lock.wait()
        pyS60uiutil.restore_app_info(self.app_info)
        return self.selected_number

def __dummy_exit_handler():
    pass

def send_sms(message):
    """
    Send the given message over the SMS. Maximum message length is 160
    characters. The longer messages will be send in several separate messages.
    Note: The given message must be unicoded.
    """
    #= Note: cannot use the number mode for querying the recipient number for
    #=       two reasons. 1) not possible to add '+' 2) if the number begins
    #=       with zero the leading zero(es) is dropped from then answer.
    length = len(message)
    if not length:
        appuifw.note(u'Nothing to send!', 'info')
        return
    nr_of_messages = (length / 160) + 1
    if nr_of_messages > 1:
        if not appuifw.query(unicode('Requires ' + str(nr_of_messages) + \
                                     ' messages. Continue?'), 'query'): return
    recipient = None
    if appuifw.query(u'Select recipient from Contacts?', 'query'):
        recipient = NumberFromContacts().select()
    if recipient == None:
        recipient = appuifw.query(u'Recipients number', 'text')
    if recipient == None:
        appuifw.note(u'Sending canceled', 'conf')
        return
    appuifw.note(u'Sending...', 'conf')
    stored_gui_data = pyS60uiutil.save_current_app_info()
    appuifw.app.body = None
    appuifw.app.exit_key_handler = __dummy_exit_handler
    appuifw.app.menu = []
    i = 0
    while i < nr_of_messages:
        messaging.sms_send(str(recipient), message[i*160:(i+1)*160])
        i += 1
    pyS60uiutil.restore_app_info(stored_gui_data)
    appuifw.note(unicode(str(nr_of_messages)) + u' message(s) sent.', 'conf')

def send_mms(message, default_path=None):
    """
    Send the given message as MMS. Optionally an attachment may be sent along
    with the message.
    Note1: the given message must be unicoded.
    Note2: works only from 3rd edition onwards.
    """
    if e32.s60_version_info[0] < 3:
        appuifw.note(u'Not supported!', 'info')
        return 
    if len(message) == 0:
        appuifw.note(u'Nothing to send!', 'info')
        return
    
    attachment = None
    if appuifw.query(u'Add attachment?', 'query'):
        while True:
            db = pyS60uiutil.dirBrowser('Select attachment')
            sel = db.select(default_path)
            if sel == None or sel == ('', ''): break #= Canceled
            attachment = os.path.join(sel[0], sel[1])
            if not os.path.isfile(attachment):
                appuifw.note(u'Not a regular file!', 'error')
                default_path = sel[0]
                attachment = None
            else: break

    recipient = None
    if appuifw.query(u'Select recipient from Contacts?', 'query'):
        recipient = NumberFromContacts().select()
    if recipient == None:
        recipient = appuifw.query(u'Recipients number', 'text')
    if recipient == None:
        appuifw.note(u'Sending canceled', 'conf')
        return

    appuifw.note(u'Sending...', 'conf')
    stored_gui_data = pyS60uiutil.save_current_app_info()
    appuifw.app.body = None
    appuifw.app.exit_key_handler = __dummy_exit_handler
    appuifw.app.menu = []

    messaging.mms_send(str(recipient), message, attachment)

    pyS60uiutil.restore_app_info(stored_gui_data)
    appuifw.note(unicode(str(nr_of_messages)) + u' message(s) sent.', 'conf')
    
class sms_importer:
    """
    Select an SMS from the Inbox, Outbox, Sent, or Drafts folder and
    read the content.
    """
    def __init__(self):
        self.box = None
        self.box_type = None
        #= The following two list must be kept in sync all the time!!!
        self.sms_addr_list = []
        self.sms_id_list = []        
        self.lbox = None
        self.selected_content = None
        self.my_lock = e32.Ao_lock()

    def __get_time_str(self, id):
        import time
        try:
            s = unicode(time.asctime(time.localtime(self.box.time(id))))
        except:
            import sys
            type, value = sys.exc_info() [:2]
            appuifw.note(unicode(str(type)+'\n'+str(value)), 'error')
            s = u'invalid'
        return s
        
    def __lbcb(self):
        """
        Joystick pressed -> selection made.
        """
        sel = appuifw.popup_menu([u'Insert content', u'Insert all'])
        if sel == None:
            return
        id = self.sms_id_list[self.lbox.current()]
        if sel == 0:
            self.selected_content = self.box.content(id)
        elif sel == 1:
            self.selected_content = self.box.address(id) + u'\n' + \
                                    self.__get_time_str(id) + u'\n' + \
                                    self.box.content(id)
        self.my_lock.signal()
        
    def __exit_handler(self):
        self.selected_content = None
        self.my_lock.signal()

    def __show_content(self):
        """
        Preview a full message.
        """
        id = self.sms_id_list[self.lbox.current()]
        if self.box_type == inbox.EInbox:
            preview = u'--- Sender ---\n'
        else:
            preview = u'--- Receiver ---\n'
        preview += self.box.address(id) + \
                   u'\n--- Time ---\n' + self.__get_time_str(id) + \
                   u'\n--- Content ---\n' + self.box.content(id)
        pyS60uiutil.fileViewer().view(preview)

    def __select_box(self):
        boxes = [u'Inbox', u'Sent', u'Drafts', u'Outbox']
        ix = appuifw.popup_menu(boxes)
        if ix == 0: return inbox.EInbox
        if ix == 1: return inbox.ESent
        if ix == 2: return inbox.EDraft
        if ix == 3: return inbox.EOutbox
        return None
    
    def select(self):
        self.box_type = self.__select_box()
        if self.box_type == None: return None
        self.box = inbox.Inbox(self.box_type)
        stored_gui_data = pyS60uiutil.save_current_app_info()
        self.selected_content = None
        self.sms_addr_list = []
        self.sms_id_list = self.box.sms_messages()
        for i in self.sms_id_list:
            self.sms_addr_list.append(self.box.address(i))
        if self.sms_addr_list == []:
            appuifw.note(u'No SMS in the folder', 'error')
            pyS60uiutil.restore_app_info(stored_gui_data)
            return None
        else:
            self.lbox = appuifw.Listbox(self.sms_addr_list, self.__lbcb)
            appuifw.app.body = self.lbox
            appuifw.app.menu = [(u'Preview', self.__show_content),]
            appuifw.app.screen = 'normal'

        appuifw.app.exit_key_handler = self.__exit_handler
        self.my_lock.wait()

        pyS60uiutil.restore_app_info(stored_gui_data)
        return self.selected_content
    
if __name__ == '__main__':
    #send_sms(u'this is just a test message')
    #send_mms(u'this is just a test message')
    #print (sms_importer().select())
    pass
