"""
pyS60gfxutil.py - a collection of graphic manipulation functions for PyS60.

Copyright (C) 2007 Lasse Huovinen

Latest version available at www.iki.fi/lasse/mobile/python.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
MA 02110-1301 USA
"""
    
def positionedTextImage(lines, image, border, lineSpacing, horizontalPosition,
                        font=None, fontColor=0, checkIfFits=False,
                        visualInspectionLines=False):
    """
    Function for writing and positioning a set of text lines to an image.
    
    lines - list of tuples (<vertical position>, <text line>). These lines will
    be written on the screen.
    - <vertical position>: 'L' - left, 'C' - center, 'R' - right
    - <text line>: line to be written
    image - image where the text lines should be placed to.
    border - how many pixels will be left clear on the edges.
    lineSpacing - space between the lines in pixels.
    horizontalPosition - 'top', 'middle', 'bottom'.
    font - font as in PyS60 Library Reference manual
    fontColor - font color as in PyS60 Library Reference manual
    checkIfFits - if True, the funtion returns False in the case the given
    text does not fit to the screen.
    visualInspectionLines - for visual testing of text positioning (True/False)
    """
    fromTop = 0
    lineInfo = []
    if border >= image.size[0] or border >= image.size[1]:
        #= The border is more than the image size.
        return False
    #= Calculate position for each individual line.
    for line in lines:
        ret_val = image.measure_text(unicode(line[1]), font)
        height = abs(ret_val[0][1]) + abs(ret_val[0][3])
        fromTop += height

        if checkIfFits:
            if ret_val[1] > image.size[0] - 2*border: return False

        if line[0] == 'L':
            xpos = border
        elif line[0] == 'C':
            xpos = border + ((image.size[0] - 2*border) - ret_val[1])/2
        elif line[0] == 'R':
            xpos = (image.size[0] - border) - ret_val[1]
        else:
            #= Should raise exception?
            return False

        info = {'txt': unicode(line[1]), 'xpos': xpos, 'ypos': fromTop}
        lineInfo.append(info)
        fromTop += lineSpacing

    #= Decrease the last added lineSpacing as it's too much.
    fromTop -= lineSpacing
    
    if checkIfFits:
        if fromTop > image.size[1] - 2*border: return False
        
    #= Offset from the top of the screen is calculated here after we know
    #= horizontal length of the full text.
    if horizontalPosition == 'top':
        topOffset = border
    elif horizontalPosition == 'middle':        
        topOffset = ((image.size[1]-(2*border))-fromTop)/2
    elif horizontalPosition == 'bottom':
        topOffset = (image.size[1]-border) - fromTop
    else:
        #= Should raise exception?
        return False

    #= Finally, draw the text lines.
    for l in lineInfo:
        image.text(( l['xpos'], topOffset + l['ypos']), l['txt'],
                   fontColor, font)

    #= Just for visual inspection to detect whether the text lines
    #= are properly positioned.
    if visualInspectionLines:
        size = image.size
        image.line([(size[0] / 2.0, 1), (size[0] / 2.0, size[1])],
                   outline=(0x11,0xCE,0x11), width=1)
        image.line([(1, size[1] / 2.0), (size[0], size[1] / 2.0)],
                   outline=(0x11,0xCE,0x11), width=1)
        image.line([(border, border), (size[0] - border, border)],
                   outline=(0x11,0xCE,0x11), width=1)
        image.line([(border, border), (border,size[1]-border)],
                   outline=(0x11,0xCE,0x11), width=1)
        image.line([(size[0]-border, border), (size[0]-border,size[1]-border)],
                   outline=(0x11,0xCE,0x11), width=1)
        image.line([(border, size[1]-border), (size[0]-border,size[1]-border)],
                   outline=(0x11,0xCE,0x11), width=1)        
        image.point((size[0] / 2.0, size[1] / 2.0),
                    outline=(255,0,0), width=2)
    return True

#=
#= Self testing and examples
#=
def _test_positionedTextImage():
    import e32
    import appuifw
    import graphics

    img = None
    def redraw(area=None):
        cnvs.clear()
        if img != None: cnvs.blit(img)

    appuifw.app.screen = 'full'
    cnvs = appuifw.Canvas(redraw)
    appuifw.app.body = cnvs
    appuifw.app.menu = []


    mylines = [('C', 'Short line'),
               ('L', 'This line is pretty long line.'),
               ('C', 'a'),
               ('R', 'Medium length')]

    try:
        img = graphics.Image.new(cnvs.size)
        img.clear(0x000000)
        fnt = None
        positionedTextImage(mylines, img, 5, 5, 'top', fnt, 0xffffff,
                            True, True)
        redraw()
        e32.ao_sleep(5)

        img.clear(0x000000)
        fnt = (u'LatinBold19', 50)
        positionedTextImage(mylines, img, 5, 5, 'middle', fnt, 0xffff00,
                            False, True)
        redraw()
        e32.ao_sleep(5)

        img.clear(0x000000)
        fnt = (u'LatinBold19', 50, graphics.FONT_ITALIC)
        positionedTextImage(mylines, img, 5, 5, 'bottom', None, 0xff0000,
                            True, True)
        redraw()
        e32.ao_sleep(5)

    except:
        import sys
        type, value = sys.exc_info()[:2]
        print type, value
        return

def _test_positionedTextImage2():
    import e32
    import appuifw
    import graphics

    img = None
    def redraw(area=None):
        cnvs.clear()
        if img != None: cnvs.blit(img)

    appuifw.app.screen = 'full'    
    cnvs = appuifw.Canvas(redraw)
    appuifw.app.body = cnvs
    appuifw.app.menu = []


    mylines1 = [('C', 'This line is pretty long one and should not fit.'),]
    mylines2 = [('C', 'short line'),]*50

    try:
        #= Test too long line
        img = graphics.Image.new(cnvs.size)
        img.clear(0x000000)
        fnt = None
        if not positionedTextImage(mylines1, img, 5, 5, 'top',
                                   fnt, 0xffffff, True, True):
            print 'Error1 (fit test passed)'
        else:
            redraw()
            e32.ao_sleep(5)

        #= Test too many lines.
        img.clear(0x000000)
        if not positionedTextImage(mylines2, img, 5, 20, 'middle',
                                   fnt, 0xffff00, True, True):
            print 'Error2 (fit test passed)'
        else:        
            redraw()
            e32.ao_sleep(5)
    except:
        import sys
        type, value = sys.exc_info()[:2]
        print type, value

if __name__ == '__main__':
    #_test_positionedTextImage()
    #_test_positionedTextImage2()
    pass
