"""
pyS60cnfutil.py - a simple program configuration database utility based on
                  Python's dictionary.

Copyright (C) 2006 - 2007 Lasse Huovinen

Latest version available at www.iki.fi/lasse/mobile/python.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
MA 02110-1301 USA
"""

import os

class configDb:
    """
    A simple configuration database utility for programs.

    The configuration data is divided into classes and each class has a
    set of parameters with values. For example,
    class 'credentials' having parameters 'username' and 'password',
    class 'bookmarks' having parameters 'bmark1', 'bmark2', 'bmark3'.

    The configuration data is stored as a text file having
    Python dictionary as content.
    """
    def __init__(self, configFile=None):
        self.db = {}
        self.configFile = configFile

    def __create_full_path(self, full_path):
        """
        Creates a full path. Returns True if successful, False otherwise.
        """
        full_path = os.path.normpath(full_path)
        drv, path = os.path.splitdrive(full_path)
        elems = []
        while path != '' and path != '/' and path != '\\':
            path, elem = os.path.split(path)
            elems.insert(0, elem)
        if path != '': elems.insert(0, path) #= leading '/'
        if elems == []: return
        dir = drv
        for e in elems:
            dir = os.path.join(dir, e)
            if not os.path.isdir(dir):
                #= Cannot overwrite a file
                if os.path.exists(dir): return False
                try:
                    os.mkdir(dir)
                except:
                    return False
        return True
        
    def loadConfigData(self, configFile=None):
        """
        Loads the configuration data from the file.
        Returns True if the file was properly read and evaluated,
        False otherwise.
        """
        if self.configFile == None and configFile != None:
            self.configFile = configFile
        if self.configFile == None: return False
        try:
            cf = file(self.configFile, 'r')
            self.db = eval(cf.read())
            cf.close()
        except:
            self.db = {}
            return False
        #= The evaluated content of the file must be of type
        #= Python dictionary. 
        if type(self.db) != type({}):
            self.db = {}
            return False
        return True

    def saveConfigData(self):
        """
        Save the current configuration data. Returns True if the configuration
        is saved successfully, False otherwise.
        """
        p = os.path.split(self.configFile)[0]
        if p != '':
            if not os.path.isdir(p):
                if not self.__create_full_path(p): return False
        try:
            cf = file(self.configFile, 'w')
            cf.write(repr(self.db))
            cf.close()
        except:
            return False
        return True
    
    def setConfigClass(self, classname, content={}):
        """
        Add a new class of the configuration parameters.
        """
        self.db[classname] = content
        if type(content) != type({}):
            return False
        del content
        return True

    def getConfigClass(self, classname):
        """
        Returns a dictionary containing all the parameters of
        the given configuration class.
        """
        if self.db.has_key(classname):
            return self.db[classname]
        else:
            return None

    def delConfigClass(self, classname):
        """
        Delete class of parameters.
        """
        if self.db.has_key(classname):
            del self.db[classname]
            return True
        else:
            return False

    def setClassMissingParameters(self, classname, defaults):
        """
        Set the default values to all the parameters of the given class that
        cannot be found from the configuration database. The 'defaults' should
        be a dictionary containing all the parameters with their default
        values of the class defined by 'classname'.
        """
        if not self.db.has_key(classname):
            return False
        for prm in defaults.keys():
            if not self.db[classname].has_key(prm):
                self.db[classname][prm] = defaults[prm]
        return True

    def setConfigParameter(self, classname,
                           parametername, parametervalue=None):
        """
        Add a new parameter (optionally with value) to the given class.
        """
        if not self.db.has_key(classname):
            return False
        self.db[classname][parametername] = parametervalue
        return True

    def getConfigParameter(self, classname, parametername):
        """
        Returns the value of the given parameter of the given class.
        """
        if not self.db.has_key(classname):
            return None
        if not self.db[classname].has_key(parametername):
            return None
        return self.db[classname][parametername]

    def delConfigParameter(self, classname, parametername):
        """
        Deletes a parameter from the given class.
        """
        if not self.db.has_key(classname):
            return False
        if not self.db[classname].has_key(parametername):
            return False
        del self.db[classname][parametername]
        return True
        
    def showContent(self):
        """
        Print out the content of the configuration database (dictionary).
        """
        print str(self.db)

if __name__ == '__main__':
    """
    #fname = '/tmp/this/is/a/long/path/test_pyS60cnfutil.txt'
    fname = '/tmp/test_pyS60cnfutil.txt'
    testCfg = configDb(fname)
    testCfg.setConfigClass('credentials')
    testCfg.setConfigParameter('credentials', 'username', 'jdoe')
    testCfg.setConfigParameter('credentials', 'password', 'foo')
    testCfg.saveConfigData()
    testCfg.showContent()

    credentialsDefaults = {'username': None, 'password': None, 'minlength': 4}
    testNewCfg = configDb(fname)
    testNewCfg.loadConfigData()
    testNewCfg.setClassMissingParameters('credentials', credentialsDefaults)
    testNewCfg.setConfigParameter('credentials', 'password', 'bar')
    testNewCfg.showContent()
    """

    """
    savetst = configDb('C:\\Data\\test.txt')
    savetst.saveConfigData()
    """
    pass
