#include "raytracer.h"
#include "objloader.h"
#include "textures.h"

RayTracer::RayTracer()
{
}

RayTracer::~RayTracer()
{
}

void RayTracer::feedObject(const class ObjLoader& obj)
{
	int n = 0;
	for (unsigned int i = 0; i < obj.polygons.size(); i++)
		n += (int)obj.polygons[i].vertices.size() - 2;

	triangleNormals.reserve(triangleNormals.size() + n);
	faceNormals.reserve(faceNormals.size() + n);
	faceColors.reserve(faceNormals.size() + n);

	int* triangles = new int [n * 3];

	int j = 0;
	for (unsigned int i = 0; i < obj.polygons.size(); i++)
	{
		const ObjLoader::Polygon& poly = obj.polygons[i];

		/* TODO: Better triangulation. */

		for (unsigned int k = 2; k < poly.vertices.size(); k++)
		{
			int v0 = poly.vertices[0].v;
			int v1 = poly.vertices[k-1].v;
			int v2 = poly.vertices[k].v;

			/* Calculate face normals. */

			m::Vector3 normal =
				cross(obj.vertices[v1] - obj.vertices[v0],
				      obj.vertices[v2] - obj.vertices[v0]);
			if (normal.length() == 0.0f)
			{
				n--;
				continue;
			}

			normal.normalize();
			faceNormals.push_back(normal);

			triangles[j++] = v0;
			triangles[j++] = v1;
			triangles[j++] = v2;
			assert(j <= n * 3);

			Triangle tri;
			tri.a = poly.vertices[0].n;
			tri.b = poly.vertices[k-1].n;
			tri.c = poly.vertices[k].n;

			if (tri.a == -1 || tri.b == -1 || tri.c == -1)
			{
				tri.a = tri.b = tri.c = -1;
			}
			else
			{
				tri.a += (int)vertexNormals.size();
				tri.b += (int)vertexNormals.size();
				tri.c += (int)vertexNormals.size();
			}

			triangleNormals.push_back(tri);

			if (poly.material >= 0)
			{
				m::Vector3 c = textures.load(obj.materials[poly.material].diffuseMap).average;
				faceColors.push_back(m::Vector3(
						obj.materials[poly.material].diffuse[0] * c.x,
						obj.materials[poly.material].diffuse[1] * c.y,
						obj.materials[poly.material].diffuse[2] * c.z));
			}
			else
				faceColors.push_back(m::Vector3(1.0f, 1.0f, 1.0f));
		}
	}

	assert(j == n * 3);

	kdtree.feedTriangles(obj.vertices, triangles, n);

	vertexNormals.reserve(vertexNormals.size() + obj.normals.size());
	for (unsigned int i = 0; i < obj.normals.size(); i++)
		vertexNormals.push_back(obj.normals[i]);

	delete [] triangles;
}

void RayTracer::update()
{
	kdtree.update();
}

bool RayTracer::intersect(const m::Vector3& pos, const m::Vector3& dir, m::Vector3& hit, m::Vector3& normal, m::Vector3& color)
{
	float t, u, v;
	int id;

	id = kdtree.intersect(pos, dir, t, u, v);
	if (id == -1)
		return false;

	hit = pos + dir * t;

	if (triangleNormals[id].a == -1)
	{
		normal = faceNormals[id];
	}
	else
	{
		normal = vertexNormals[triangleNormals[id].a] * (1 - u - v);
		normal += vertexNormals[triangleNormals[id].b] * u;
		normal += vertexNormals[triangleNormals[id].c] * v;
		normal.normalize();
	}

	if (dot(dir, normal) > 0.0f)
		normal = -normal;

	color = faceColors[id];

	return true;
}
