#include "cubemap.h"
#include "configuration.h"
#include "glutil.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include "GL/glew.h"

static void checkFramebufferStatus(void)
{
	GLenum status;
	status = glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	if (status != GL_FRAMEBUFFER_COMPLETE_EXT)
		printf("FBO STATUS %x\n", status);
	assert(status == GL_FRAMEBUFFER_COMPLETE_EXT);
}

CubeMap::CubeMap(int size) : size(size)
{
	assert(size > 4);

	glGenTextures(1, &texture);
	glGenTextures(1, &texture2);

	assert(glGetError() == GL_NO_ERROR);

	glBindTexture(GL_TEXTURE_2D, texture2);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT16, size, size, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

#if 1
	glBindTexture(GL_TEXTURE_CUBE_MAP, texture);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_COMPARE_MODE_ARB, GL_COMPARE_R_TO_TEXTURE_ARB);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_COMPARE_FUNC_ARB, GL_LEQUAL);
#else
	glBindTexture(GL_TEXTURE_2D, texture);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT16, size, size, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE_ARB, GL_COMPARE_R_TO_TEXTURE_ARB);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC_ARB, GL_LEQUAL);
#endif

#if 0
	for (unsigned int i = 0; i < 6; i++)
	{
		GLenum side = GL_TEXTURE_CUBE_MAP_POSITIVE_X + i;

		assert(glGetError() == GL_NO_ERROR);
		glBindTexture(GL_TEXTURE_CUBE_MAP, texture);
		assert(glGetError() == GL_NO_ERROR);
		glTexImage2D(side, 0, GL_DEPTH_COMPONENT16, size, size, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
		assert(glGetError() == GL_NO_ERROR);
		glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
		assert(glGetError() == GL_NO_ERROR);
	}
#endif

	glGenFramebuffersEXT(1, fb);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fb[0]);

	glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, texture2, 0);

#if 0
	GLuint rb;
	glGenRenderbuffersEXT(1, &rb);
	glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, rb);
	glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT16, size, size);
	glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, rb);
#endif

	glDrawBuffer(GL_NONE);
	assert(glGetError() == GL_NO_ERROR);
	glReadBuffer(GL_NONE);
	assert(glGetError() == GL_NO_ERROR);

#if 0
	for (unsigned int i = 0; i < 6; i++)
	{
		GLenum side = GL_TEXTURE_CUBE_MAP_POSITIVE_X + i;

		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fb[i]);
		assert(glGetError() == GL_NO_ERROR);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, side, texture, 0);
		assert(glGetError() == GL_NO_ERROR);

		glDrawBuffer(GL_NONE);
		assert(glGetError() == GL_NO_ERROR);
		glReadBuffer(GL_NONE);
		assert(glGetError() == GL_NO_ERROR);

		checkFramebufferStatus();

#if 0
		assert(glGetError() == GL_NO_ERROR);
		glClear(GL_DEPTH_BUFFER_BIT);
		printf("%x\n", glGetError());
		assert(glGetError() == GL_NO_ERROR);
#endif
	}
#endif

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	assert(glGetError() == GL_NO_ERROR);

#if 0
	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X_EXT, 0, GL_RGBA, 256, 256, 0, GL_RED, GL_UNSIGNED_BYTE, p);
	assert(glGetError() == GL_NO_ERROR);
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_X_EXT, 0, GL_RGBA, 256, 256, 0, GL_RED, GL_UNSIGNED_BYTE, p);
	assert(glGetError() == GL_NO_ERROR);
	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_Y_EXT, 0, GL_RGBA, 256, 256, 0, GL_GREEN, GL_UNSIGNED_BYTE, p);
	assert(glGetError() == GL_NO_ERROR);
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_EXT, 0, GL_RGBA, 256, 256, 0, GL_BLUE, GL_UNSIGNED_BYTE, p);
	assert(glGetError() == GL_NO_ERROR);
	glTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_Z_EXT, 0, GL_RGBA, 256, 256, 0, GL_RED, GL_UNSIGNED_BYTE, p);
	assert(glGetError() == GL_NO_ERROR);
	glTexImage2D(GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_EXT, 0, GL_RGBA, 256, 256, 0, GL_GREEN, GL_UNSIGNED_BYTE, p);
	assert(glGetError() == GL_NO_ERROR);
#endif
}

CubeMap::~CubeMap()
{
	glDeleteTextures(1, &texture);
}

void CubeMap::setPosition(const m::Vector3& p)
{
	pos = p;
}

void CubeMap::setDirection(const m::Vector3& d)
{
	dir = d;

	if (fabs(dir.y) < 0.5f)
		right = m::normalize(m::cross(m::Vector3(0.0f, 1.0f, 0.0f), dir));
	else
		right = m::normalize(m::cross(dir, m::Vector3(1.0f, 0.0f, 0.0f)));

	up = m::cross(dir, right);
}

void CubeMap::getMatrix(float* m)
{
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glTranslatef(-pos.x, -pos.y, -pos.z);
	//lookat(pos, dir);
	glGetFloatv(GL_MODELVIEW_MATRIX, m);
	glPopMatrix();
}

void CubeMap::render(void (*render)(void*), void* user)
{
	if (CONFI(no_depth_cubemaps))
		return;

	static const float dirs[6][3] = {
		{ 1.0f, 0.0f, 0.0f },
		{-1.0f, 0.0f, 0.0f },
		{ 0.0f, 1.0f, 0.0f },
		{ 0.0f,-1.0f, 0.0f },
		{ 0.0f, 0.0f, 1.0f },
		{0.0f, 0.0f, -1.0f }
	};

	static const float ups[6][3] = {
		{ 0.0f, -1.0f, 0.0f },
		{ 0.0f, -1.0f, 0.0f },
		{ 0.0f, 0.0f, 1.0f },
		{ 0.0f, 0.0f, -1.0f },
		{ 0.0f, -1.0f, 0.0f },
		{ 0.0f, -1.0f, 0.0f }
	};

	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluPerspective(90.0f, 1.0f, 0.1f, 50.0f);

	glActiveTexture(GL_TEXTURE0);

	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
	glBindTexture(GL_TEXTURE_2D, 0);
	glDisable(GL_TEXTURE_CUBE_MAP);
	glDisable(GL_TEXTURE_2D);
	glEnable(GL_DEPTH_TEST);
	assert(glGetError() == GL_NO_ERROR);

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fb[0]);
	assert(glGetError() == GL_NO_ERROR);

	GLint vp[4];
	glGetIntegerv(GL_VIEWPORT, vp);
	glViewport(0, 0, size, size);

	for (int i = 0; i < 6; i++)
	{
		glClear(GL_DEPTH_BUFFER_BIT);
		assert(glGetError() == GL_NO_ERROR);

		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();
		gluLookAt(
			pos.x, pos.y, pos.z,
			pos.x + dirs[i][0], pos.y + dirs[i][1], pos.z + dirs[i][2],
			ups[i][0], ups[i][1], ups[i][2]);

		assert(glGetError() == GL_NO_ERROR);
		render(user);

		glPopMatrix();
		assert(glGetError() == GL_NO_ERROR);

		glEnable(GL_TEXTURE_CUBE_MAP);
		glBindTexture(GL_TEXTURE_CUBE_MAP, texture);

		//glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_COMPARE_MODE_ARB, GL_NONE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE_ARB, GL_COMPARE_R_TO_TEXTURE_ARB);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC_ARB, GL_LEQUAL);

		glCopyTexImage2D(GL_TEXTURE_CUBE_MAP_POSITIVE_X_EXT + i, 0, GL_DEPTH_COMPONENT16, 0, 0, size, size, 0);
		glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
		glDisable(GL_TEXTURE_CUBE_MAP);
		//glBindTexture(GL_TEXTURE_2D, texture);
		//glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT16, 0, 0, size, size, 0);
		//glBindTexture(GL_TEXTURE_2D, 0);
	}

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	glViewport(vp[0], vp[1], vp[2], vp[3]);

	assert(glGetError() == GL_NO_ERROR);

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	assert(glGetError() == GL_NO_ERROR);
}

#if 0
void CubeMap::prepareRendering(int n)
{
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluPerspective(90.0f, 1.0f, 0.1f, 20.0f);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	float m[16] =
	{
		1.0f, 0.0f, 0.0f, 0.0f,
		0.0f, 1.0f, 0.0f, 0.0f,
		0.0f, 0.0f, 1.0f, 0.0f,
		0.0f, 0.0f, 0.0f, 1.0f
	};

	m::Vector3 v1;
	m::Vector3 v2;
	m::Vector3 v3;

	switch (n)
	{
		/* Pos x */
	case 0:
		v1 = -dir;
		v2 = -up;
		v3 = -right;
		break;
		/* Neg x */
	case 1:
		v1 = dir;
		v2 = -up;
		v3 = right;
		break;
		/* Pos y */
	case 2:
		v1 = right;
		v2 = dir;
		v3 = -up;
		break;
		/* Neg y */
	case 3:
		v1 = right;
		v2 = -dir;
		v3 = up;
		break;
		/* Pos z */
	case 4:
		v1 = right;
		v2 = -up;
		v3 = -dir;
		break;
		/* Neg z */
	case 5:
		v1 = -right;
		v2 = -up;
		v3 = dir;
		break;
	}

	m[0]  = v1.x;
	m[4]  = v1.y;
	m[8]  = v1.z;
	m[1]  = v2.x;
	m[5]  = v2.y;
	m[9]  = v2.z;
	m[2]  = v3.x;
	m[6]  = v3.y;
	m[10] = v3.z;

	m[12] = m[0] * -pos.x + m[4] * -pos.y + m[8] * -pos.z;
	m[13] = m[1] * -pos.x + m[5] * -pos.y + m[9] * -pos.z;
	m[14] = m[2] * -pos.x + m[6] * -pos.y + m[10] * -pos.z;

#if 0
	glLoadIdentity();
	gluLookAt(pos.x, pos.y, pos.z,
			  pos.x+dir.x, pos.y+dir.y, pos.z+dir.z,
			  0, 1, 0);
#else
	glLoadMatrixf(m);
#endif

#if 0
	printf("M\n");
	for (int i = 0; i < 16; i++)
	{
		printf("%.2f\t", m[i]);
		if ((i % 4) == 3)
			printf("\n");
	}

	printf("NARRAR\n");
	float f[16];
	glGetFloatv(GL_MODELVIEW_MATRIX, f);
	for (int i = 0; i < 16; i++)
	{
		printf("%.2f\t", f[i]);
		if ((i % 4) == 3)
			printf("\n");
	}

	fflush(NULL);
#endif
}

void CubeMap::doneRendering(int n)
{
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	int w = 256;
	int h = 256;
	GLuint tex;

	switch (n)
	{
	case 0: tex = GL_TEXTURE_CUBE_MAP_POSITIVE_X_EXT; break;
	case 1: tex = GL_TEXTURE_CUBE_MAP_NEGATIVE_X_EXT; break;
	case 2: tex = GL_TEXTURE_CUBE_MAP_POSITIVE_Y_EXT; break;
	case 3: tex = GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_EXT; break;
	case 4: tex = GL_TEXTURE_CUBE_MAP_POSITIVE_Z_EXT; break;
	case 5: tex = GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_EXT; break;
	}

	unsigned char p[256][256][4];
	glReadPixels(0, 0, 256, 256, GL_RGBA, GL_UNSIGNED_BYTE, p);

	glTexImage2D(tex, 0, GL_RGBA, 256, 256, 0, GL_RGBA, GL_UNSIGNED_BYTE, p);
	assert(glGetError() == GL_NO_ERROR);
}
#endif
