#ifdef WIN32
#  pragma warning(disable: 4514)		//unreferenced inline function removed
#  pragma warning(disable: 4710)		//function not inlined
#  pragma warning(disable: 4514)		//unreferenced inline function has been removed
#endif

#include "VectorMatrix.h"

namespace JC
{
namespace VECTORMATRIX
{

//==============================================================================================
//==============================================================================================

static inline T det2x2( T m00, T m10, T m01, T m11 )
{
	return m00 * m11 - m01 * m10;
}

//==============================================================================================
//==============================================================================================

static inline T det3x3( T m00, T m10, T m20, T m01, T m11, T m21, T m02, T m12, T m22 )
{
	return m00 * det2x2( m11, m21, m12, m22 )
		 - m01 * det2x2( m10, m20, m12, m22 )
		 + m02 * det2x2( m10, m20, m11, m21 );
}

//==============================================================================================
//==============================================================================================






//==============================================================================================
//==============================================================================================

bool Matrix2::invert()
{
	T d = det();
	if( d == T(0) ) return false;
	d = T(1) / d;
	// inv(A) =       [ a   -a   ]
	//			  1   [  11   01 ]
	//			----- [-a    a   ]
	//			det A [  10   00 ]
	//
	T t = matrix[0][0]; matrix[0][0] = matrix[1][1]; matrix[1][1] = t;
	matrix[0][1] = -matrix[0][1];
	matrix[1][0] = -matrix[1][0];
	*this *= d;
	return true;
}

//==============================================================================================
//==============================================================================================

void Matrix2::rotate( T radians, MultiplicationOrder o )
{
	T c = COS( radians );
	T s = SIN( radians );
	if( o == LEFT_MULTIPLY )
		s = -s;	//transposes the matrix
	Matrix2 m( c, s,
			  -s, c );
	multiply( m, o );
}

//==============================================================================================
//==============================================================================================





//==============================================================================================
//==============================================================================================

bool Matrix3::invert()
{
	T det00 = matrix[1][1]*matrix[2][2] - matrix[2][1]*matrix[1][2];
	T det01 = matrix[2][0]*matrix[1][2] - matrix[1][0]*matrix[2][2];
	T det02 = matrix[1][0]*matrix[2][1] - matrix[2][0]*matrix[1][1];

	T det = matrix[0][0]*det00 + matrix[0][1]*det01 + matrix[0][2]*det02;
	if( det == T(0) ) return false;	//singular, leave the matrix unmodified and return false
	det = T(1) / det;

	Matrix3 t;
	t[0][0] = det * det00;
	t[1][0] = det * det01;
	t[2][0] = det * det02;
	t[0][1] = det * (matrix[2][1]*matrix[0][2] - matrix[0][1]*matrix[2][2]);
	t[1][1] = det * (matrix[0][0]*matrix[2][2] - matrix[2][0]*matrix[0][2]);
	t[2][1] = det * (matrix[2][0]*matrix[0][1] - matrix[0][0]*matrix[2][1]);
	t[0][2] = det * (matrix[0][1]*matrix[1][2] - matrix[1][1]*matrix[0][2]);
	t[1][2] = det * (matrix[1][0]*matrix[0][2] - matrix[0][0]*matrix[1][2]);
	t[2][2] = det * (matrix[0][0]*matrix[1][1] - matrix[1][0]*matrix[0][1]);
	*this = t;
	return true;
}

//==============================================================================================
//==============================================================================================

void Matrix3::rotate( T radians, const Vector3& aboutThis, MultiplicationOrder o )
{
	T c = COS( radians );
	T t = T(1) - c;
	T s = SIN( radians );
	if( o == LEFT_MULTIPLY )
		s = -s;	//transposes the matrix
	Vector3 vn( aboutThis );
	if( !vn.normalize( ) )
		return;	//in case the about vector is zero, do nothing
	Matrix3 m( t*vn.x*vn.x+c,	   t*vn.x*vn.y+s*vn.z, t*vn.x*vn.z-s*vn.y,
			   t*vn.x*vn.y-s*vn.z, t*vn.y*vn.y+c,	   t*vn.y*vn.z+s*vn.x,
			   t*vn.x*vn.z+s*vn.y, t*vn.y*vn.z-s*vn.x, t*vn.z*vn.z+c	   );
	multiply( m, o );
}

//==============================================================================================
//==============================================================================================





//==============================================================================================
//==============================================================================================

static void adjoint( Matrix4& m )
{
	T m00 = m[0][0], m01 = m[0][1], m02 = m[0][2], m03 = m[0][3];
	T m10 = m[1][0], m11 = m[1][1], m12 = m[1][2], m13 = m[1][3];
	T m20 = m[2][0], m21 = m[2][1], m22 = m[2][2], m23 = m[2][3];
	T m30 = m[3][0], m31 = m[3][1], m32 = m[3][2], m33 = m[3][3];

	// row column labeling reversed since we transpose rows and columns
	m[0][0]  =  det3x3( m11, m21, m31, m12, m22, m32, m13, m23, m33 );
	m[1][0]  = -det3x3( m10, m20, m30, m12, m22, m32, m13, m23, m33 );
	m[2][0]  =  det3x3( m10, m20, m30, m11, m21, m31, m13, m23, m33 );
	m[3][0]  = -det3x3( m10, m20, m30, m11, m21, m31, m12, m22, m32 );

	m[0][1]  = -det3x3( m01, m21, m31, m02, m22, m32, m03, m23, m33 );
	m[1][1]  =  det3x3( m00, m20, m30, m02, m22, m32, m03, m23, m33 );
	m[2][1]  = -det3x3( m00, m20, m30, m01, m21, m31, m03, m23, m33 );
	m[3][1]  =  det3x3( m00, m20, m30, m01, m21, m31, m02, m22, m32 );

	m[0][2]  =  det3x3( m01, m11, m31, m02, m12, m32, m03, m13, m33 );
	m[1][2]  = -det3x3( m00, m10, m30, m02, m12, m32, m03, m13, m33 );
	m[2][2]  =  det3x3( m00, m10, m30, m01, m11, m31, m03, m13, m33 );
	m[3][2]  = -det3x3( m00, m10, m30, m01, m11, m31, m02, m12, m32 );

	m[0][3]  = -det3x3( m01, m11, m21, m02, m12, m22, m03, m13, m23 );
	m[1][3]  =  det3x3( m00, m10, m20, m02, m12, m22, m03, m13, m23 );
	m[2][3]  = -det3x3( m00, m10, m20, m01, m11, m21, m03, m13, m23 );
	m[3][3]  =  det3x3( m00, m10, m20, m01, m11, m21, m02, m12, m22 );
}

//==============================================================================================
//==============================================================================================

bool Matrix4::invert()
{
	T d = det( );
	if( d == T(0) ) return false;
    adjoint( *this );
	*this *= (T(1) / d);
	return true;
}

//==============================================================================================
//==============================================================================================

T Matrix4::det() const
{
	T m00 = matrix[0][0], m01 = matrix[0][1], m02 = matrix[0][2], m03 = matrix[0][3];
	T m10 = matrix[1][0], m11 = matrix[1][1], m12 = matrix[1][2], m13 = matrix[1][3];
	T m20 = matrix[2][0], m21 = matrix[2][1], m22 = matrix[2][2], m23 = matrix[2][3];
	T m30 = matrix[3][0], m31 = matrix[3][1], m32 = matrix[3][2], m33 = matrix[3][3];

	return m00 * det3x3( m11, m21, m31, m12, m22, m32, m13, m23, m33 ) -
		   m01 * det3x3( m10, m20, m30, m12, m22, m32, m13, m23, m33 ) +
		   m02 * det3x3( m10, m20, m30, m11, m21, m31, m13, m23, m33 ) -
		   m03 * det3x3( m10, m20, m30, m11, m21, m31, m12, m22, m32 );
}

//==============================================================================================
//==============================================================================================

void Matrix4::rotate( T radians, const Vector3& aboutThis, MultiplicationOrder o )
{
	T c = COS( radians );
	T t = T(1) - c;
	T s = SIN( radians );
	if( o == LEFT_MULTIPLY )
		s = -s;	//transposes the matrix
	Vector3 vn( aboutThis );
	if( !vn.normalize( ) )
		return;	//in case the about vector is zero, do nothing
	Matrix4 m( t*vn.x*vn.x+c,	   t*vn.x*vn.y+s*vn.z, t*vn.x*vn.z-s*vn.y, T(0),
			   t*vn.x*vn.y-s*vn.z, t*vn.y*vn.y+c,	   t*vn.y*vn.z+s*vn.x, T(0),
			   t*vn.x*vn.z+s*vn.y, t*vn.y*vn.z-s*vn.x, t*vn.z*vn.z+c,      T(0),
			   T(0),               T(0),               T(0),               T(1) );
	multiply( m, o );
}

//==============================================================================================
//==============================================================================================





//==============================================================================================
//==============================================================================================

bool Matrix2x3::invert()
{
	//translation = -(invert(rotation) * translation)
	//rotation = invert(rotation)

	Matrix2 rotation( getRotation() );
	if( !rotation.invert( ) )
		return false;

	Vector2 translation( matrix[0][2], matrix[1][2] );
	translation.multiply( rotation, LEFT_MULTIPLY );

	set( rotation[0][0], rotation[0][1], -translation[0],
		 rotation[1][0], rotation[1][1], -translation[1] );
	return true;
}

//==============================================================================================
//==============================================================================================

const Matrix2x3 operator*( const Matrix2x3& m1, const Matrix2x3& m2 )
{
	// dest.rotation = m1.rotation * m2.rotation
	// dest.translation = m1.rotation * m2.translation + m1.translation

	Matrix2 rotation1( m1.getRotation() ), rotation2( m2.getRotation() );
	Vector2 translation1( m1[0][2], m1[1][2] ), translation2( m2[0][2], m2[1][2] );

	rotation2.multiply( rotation1, LEFT_MULTIPLY );
	translation2.multiply( rotation1, LEFT_MULTIPLY );
	translation2 += translation1;

	return Matrix2x3( rotation2[0][0], rotation2[0][1], translation2[0],
					  rotation2[1][0], rotation2[1][1], translation2[1] );
}

//==============================================================================================
//==============================================================================================

void Matrix2x3::rotate( T radians )
{
	T c = COS( radians );
	T s = SIN( radians );
	//always a left multiply
	Matrix2x3 m( c, -s, T(0),
			     s,  c, T(0) );
	multiply( m );
}

//==============================================================================================
//==============================================================================================






//==============================================================================================
//==============================================================================================

bool Matrix3x2::invert()
{
	//translation = -(translation * invert(rotation))
	//rotation = invert(rotation)

	Matrix2& rotation = (Matrix2&)(*this);	//works because upperleft corner of Matrix3x2 is Matrix2x2
	if( !rotation.invert( ) )
		return false;

	Vector2& translation = (*this)[2];
	translation.multiply( rotation, RIGHT_MULTIPLY );
	translation.negate( );
	return true;
}

//==============================================================================================
//==============================================================================================

const Matrix3x2 operator*( const Matrix3x2& m1, const Matrix3x2& m2 )
{
	// dest.rotation = m1.rotation * m2.rotation
	// dest.translation = m1.translation * m2.rotation + m2.translation

	Matrix3x2 t( m1 );
	Matrix2& rotation1 = (Matrix2&)t;				//works because upperleft corner of Matrix3x2 is Matrix2x2
	Vector2& translation1 = t[2];
	const Matrix2& rotation2 = (const Matrix2&)m2;	//works because upperleft corner of Matrix3x2 is Matrix2x2
	const Vector2& translation2 = m2[2];

	rotation1.multiply( rotation2, RIGHT_MULTIPLY );
	translation1.multiply( rotation2, RIGHT_MULTIPLY );
	translation1 += translation2;
	return t;
}

//==============================================================================================
//==============================================================================================

void Matrix3x2::rotate( T radians )
{
	T c = COS( radians );
	T s = SIN( radians );
	//always a right multiply
	Matrix3x2 m(   c,   s,
			      -s,   c,
			    T(0), T(0) );
	multiply( m );
}

//==============================================================================================
//==============================================================================================






//==============================================================================================
//==============================================================================================

bool Matrix3x4::invert()
{
	//translation = -(invert(rotation) * translation)
	//rotation = invert(rotation)

	Matrix3 rotation( getRotation() );
	if( !rotation.invert( ) )
		return false;

	Vector3 translation( matrix[0][3], matrix[1][3], matrix[2][3] );
	translation.multiply( rotation, LEFT_MULTIPLY );

	set( rotation[0][0], rotation[0][1], rotation[0][2], -translation[0],
		 rotation[1][0], rotation[1][1], rotation[1][2], -translation[1],
		 rotation[2][0], rotation[2][1], rotation[2][2], -translation[2] );
	return true;
}

//==============================================================================================
//==============================================================================================

const Matrix3x4 operator*( const Matrix3x4& m1, const Matrix3x4& m2 )
{
	// dest.rotation = m1.rotation * m2.rotation
	// dest.translation = m1.rotation * m2.translation + m1.translation

	Matrix3 rotation1( m1.getRotation() ), rotation2( m2.getRotation() );
	Vector3 translation1( m1[0][3], m1[1][3], m1[2][3] ), translation2( m2[0][3], m2[1][3], m2[2][3] );

	rotation2.multiply( rotation1, LEFT_MULTIPLY );
	translation2.multiply( rotation1, LEFT_MULTIPLY );
	translation2 += translation1;

	return Matrix3x4( rotation2[0][0], rotation2[0][1], rotation2[0][2], translation2[0],
					  rotation2[1][0], rotation2[1][1], rotation2[1][2], translation2[1],
					  rotation2[2][0], rotation2[2][1], rotation2[2][2], translation2[2] );
}

//==============================================================================================
//==============================================================================================

void Matrix3x4::rotate( T radians, const Vector3& aboutThis )
{
	//always a left multiply
	T c = COS( radians );
	T s = -SIN( radians );
	T t = T(1) - c;
	Vector3 vn( aboutThis );
	if( !vn.normalize( ) )
		return;	//in case the about vector is zero, do nothing
	Matrix3x4 m( t*vn.x*vn.x+c,	     t*vn.x*vn.y+s*vn.z, t*vn.x*vn.z-s*vn.y, T(0),
			     t*vn.x*vn.y-s*vn.z, t*vn.y*vn.y+c,	     t*vn.y*vn.z+s*vn.x, T(0),
			     t*vn.x*vn.z+s*vn.y, t*vn.y*vn.z-s*vn.x, t*vn.z*vn.z+c,      T(0) );
	multiply( m );
}

//==============================================================================================
//==============================================================================================






//==============================================================================================
//==============================================================================================

bool Matrix4x3::invert()
{
	//translation = -(translation * invert(rotation))
	//rotation = invert(rotation)

	Matrix3& rotation = (Matrix3&)(*this);	//works because upperleft corner of Matrix4x3 is Matrix3x3
	if( !rotation.invert( ) )
		return false;

	Vector3& translation = (*this)[3];
	translation.multiply( rotation, RIGHT_MULTIPLY );
	translation.negate( );
	return true;
}

//==============================================================================================
//==============================================================================================

const Matrix4x3 operator*( const Matrix4x3& m1, const Matrix4x3& m2 )
{
	// dest.rotation = m1.rotation * m2.rotation
	// dest.translation = m1.translation * m2.rotation + m2.translation

	Matrix4x3 t( m1 );
	Matrix3& rotation1 = (Matrix3&)t;				//works because upperleft corner of Matrix4x3 is Matrix3x3
	Vector3& translation1 = t[3];
	const Matrix3& rotation2 = (const Matrix3&)m2;	//works because upperleft corner of Matrix4x3 is Matrix3x3
	const Vector3& translation2 = m2[3];

	rotation1.multiply( rotation2, RIGHT_MULTIPLY );
	translation1.multiply( rotation2, RIGHT_MULTIPLY );
	translation1 += translation2;
	return t;
}

//==============================================================================================
//==============================================================================================

void Matrix4x3::rotate( T radians, const Vector3& aboutThis )
{
	//always a right multiply
	T c = COS( radians );
	T s = SIN( radians );
	T t = T(1) - c;
	Vector3 vn( aboutThis );
	if( !vn.normalize( ) )
		return;	//in case the about vector is zero, do nothing
	Matrix4x3 m( t*vn.x*vn.x+c,	     t*vn.x*vn.y+s*vn.z, t*vn.x*vn.z-s*vn.y,
			     t*vn.x*vn.y-s*vn.z, t*vn.y*vn.y+c,	     t*vn.y*vn.z+s*vn.x,
			     t*vn.x*vn.z+s*vn.y, t*vn.y*vn.z-s*vn.x, t*vn.z*vn.z+c,    
			     T(0),               T(0),               T(0)              );
	multiply( m );
}

//==============================================================================================
//==============================================================================================





//==============================================================================================
//==============================================================================================

const Vector2	Vector2Factory::randInSquare( T edgeLength )
{
	T e = edgeLength * T(0.5);
	return Vector2( e*RAND(), e*RAND() );
}

//==============================================================================================
//==============================================================================================

const Vector2	Vector2Factory::randInCircle( T radius )
{
	//distribution is uniform when random vectors inside a unit circle are selected
	Vector2 t;
	T d;
	do
	{
		t.set( RAND(), RAND() );
		d = dot( t, t );
	}
	while( d > T(1) );
	t *= radius;
	return t;
}

//==============================================================================================
//==============================================================================================

const Vector2	Vector2Factory::randOnCircle( T radius )
{
	//distribution is uniform when random vectors inside a unit circle are normalized
	Vector2 t;
	T d;
	do
	{
		t.set( RAND(), RAND() );
		d = dot( t, t );
	}
	while( d == T(0) || d > T(1) );

	t.normalize();
	t *= radius;
	return t;
}

//==============================================================================================
//==============================================================================================

const Vector2 Vector2Factory::perpendicular( const Vector2& v )
{
	return Vector2(	-v.y, v.x );
}

//==============================================================================================
//==============================================================================================




//==============================================================================================
//==============================================================================================

const Vector3	Vector3Factory::randInCube( T edgeLength )
{
	T e = edgeLength * T(0.5);
	return Vector3( e*RAND(), e*RAND(), e*RAND() );
}

//==============================================================================================
//==============================================================================================

const Vector3	Vector3Factory::randInSphere( T radius )
{
	//distribution is uniform when random vectors inside a unit sphere are selected
	Vector3 t;
	T d;
	do
	{
		t.set( RAND(), RAND(), RAND() );
		d = dot( t, t );
	}
	while( d > T(1) );
	t *= radius;
	return t;
}

//==============================================================================================
//==============================================================================================

const Vector3	Vector3Factory::randOnSphere( T radius )
{
	//distribution is uniform when random vectors inside a unit sphere are normalized
	Vector3 t;
	T d;
	do
	{
		t.set( RAND(), RAND(), RAND() );
		d = dot( t, t );
	}
	while( d == T(0) || d > T(1) );

	t.normalize();
	t *= radius;
	return t;
}

//==============================================================================================
//==============================================================================================

const Vector3 Vector3Factory::perpendicular( const Vector3& v )
{
	Vector3 n,t;
	n = v;
	if( !n.normalize() )
		return Vector3(0,0,0);

	//select the shortest of projections of axes on v (the closest to perpendicular to v),
	// and project it to the plane defined by v
	if( ABS(n.x) < ABS(n.y) )
	{	//x < y
		if( ABS(n.x) < ABS(n.z) )
		{	// x < y && x < z
			t.x = T(1) - n.x * n.x;
			t.y = -n.x * n.y;
			t.z = -n.x * n.z;
			return t;
		}
	}
	else
	{	//y <= x
		if( ABS(n.y) < ABS(n.z) )
		{	// y <= x && y < z
			t.x = -n.y * n.x;
			t.y = T(1) - n.y * n.y;
			t.z = -n.y * n.z;
			return t;
		}
	}

	//z <= x && z <= y
	t.x = -n.z * n.x;
	t.y = -n.z * n.y;
	t.z = T(1) - n.z * n.z;
	return t;
}

//==============================================================================================
//==============================================================================================




//==============================================================================================
//==============================================================================================

const Vector4	Vector4Factory::randInCube( T edgeLength )
{
	T e = edgeLength * T(0.5);
	return Vector4( e*RAND(), e*RAND(), e*RAND(), e*RAND() );
}

//==============================================================================================
//==============================================================================================

const Vector4	Vector4Factory::randInSphere( T radius )
{
	//distribution is uniform when random vectors inside a unit sphere are selected
	Vector4 t;
	T d;
	do
	{
		t.set( RAND(), RAND(), RAND(), RAND() );
		d = dot( t, t );
	}
	while( d > T(1) );
	t *= radius;
	return t;
}

//==============================================================================================
//==============================================================================================

const Vector4	Vector4Factory::randOnSphere( T radius )
{
	//distribution is uniform when random vectors inside a unit sphere are normalized
	Vector4 t;
	T d;
	do
	{
		t.set( RAND(), RAND(), RAND(), RAND() );
		d = dot( t, t );
	}
	while( d == T(0) || d > T(1) );

	t.normalize();
	t *= radius;
	return t;
}

//==============================================================================================
//==============================================================================================



/*
//==============================================================================================
//==============================================================================================

Matrix3 MatrixFactory::rotationX( T radians, MultiplicationOrder h )
{
	T c = T(cos( radians ));
	T s = T(sin( radians ));
	if( h == LEFT_MULTIPLY ) s = -s;	//transposes the matrix
	return Matrix3( T(1), T(0), T(0),
					T(0),    c,    s,
					T(0),   -s,    c );
}

//==============================================================================================
//==============================================================================================

Matrix3	MatrixFactory::rotationY( T radians, MultiplicationOrder h )
{
	T c = T(cos( radians ));
	T s = T(sin( radians ));
	if( h == LEFT_MULTIPLY ) s = -s;	//transposes the matrix
	return Matrix3(    c, T(0),   -s,
					T(0), T(1), T(0),
					   s, T(0),    c );
}

//==============================================================================================
//==============================================================================================

Matrix3	MatrixFactory::rotationZ( T radians, MultiplicationOrder h )
{
	T c = T(cos( radians ));
	T s = T(sin( radians ));
	if( h == LEFT_MULTIPLY ) s = -s;	//transposes the matrix
	return Matrix3(    c,    s, T(0),
					  -s,    c, T(0),
					T(0), T(0), T(1) );
}

//==============================================================================================
//==============================================================================================

Matrix3	MatrixFactory::rotationAbout( const Vector3& v, T radians, MultiplicationOrder h )
{
	T c = T(cos( radians ));
	T s = T(sin( radians ));
	T t = T(1) - c;
	Vector3 vn( v );
	bool r = vn.normalize( );
	ASSERT( r );	//input vector is a zero vector
	Matrix3 m( t*vn.x*vn.x+c,	   t*vn.x*vn.y+s*vn.z, t*vn.x*vn.z-s*vn.y,
			   t*vn.x*vn.y-s*vn.z, t*vn.y*vn.y+c,	   t*vn.y*vn.z+s*vn.x,
			   t*vn.x*vn.z+s*vn.y, t*vn.y*vn.z-s*vn.x, t*vn.z*vn.z+c	   );
	if( h == LEFT_MULTIPLY )
		m.transpose();
	return m;
}
*/
//==============================================================================================
//==============================================================================================

const Matrix3 MatrixFactory::target( const Vector3& pos, const Vector3& tgt, const Vector3& up, MultiplicationOrder o )
{
	// Advanced Animation and Rendering Techniques, Watt & Watt (p. 9)
	Matrix3 m;
	// z is the normalized direction from pos to tgt
	m[2] = tgt;
	m[2] -= pos;
	if( !m[2].normalize() )
		m[2].set( T(0), T(0), T(1) );	//z was a zero vector
	// y (up) is pointing towards the up vector, but is perpendicular to z
	m[1] = up - project( up, m[2] );	//y = up - up's projection on z
	m[1].normalize( );
	// x is the cross product between y and z
	m[0] = cross(m[1], m[2]);
//	if( righthanded )
//		m[2].negate();
	if( o == LEFT_MULTIPLY )
		m.transpose();
	return m;
}

//==============================================================================================
//==============================================================================================

//The (left, bottom, znear) and (right, top, znear) parameters specify the points on the
//near clipping plane that are mapped to the lower-left and upper-right corners of the window,
//respectively, assuming that the eye is located at (0, 0, 0)
//this maps the view frustum to a box with dimensions [-1,1]
//this corresponds to OpenGL projection matrix format
const Matrix4 MatrixFactory::frustum( T left, T right, T bottom, T top, T zNear, T zFar, MultiplicationOrder o )
{
	T ooRL = T(1)/(right-left), ooTB = T(1)/(top-bottom), ooFN = T(1)/(zFar-zNear), ZNear2 = T(2)*zNear;
	Matrix4 frustum( ZNear2 * ooRL,          T(0),                   T(0),                  T(0),
					 T(0),                   ZNear2 * ooTB,          T(0),                  T(0),
					-(right + left) * ooRL, -(top + bottom) * ooTB,  (zFar + zNear) * ooFN, T(1),
					 T(0),                   T(0),                  -ZNear2 * zFar * ooFN,  T(0) );
//	if( righthanded )
//		frustum[2].negate();
	if( o == LEFT_MULTIPLY )
		frustum.transpose();
	return frustum;
}

//==============================================================================================
//==============================================================================================

//this maps the view frustum to a box with dimensions [-1,1]
//this corresponds to OpenGL projection matrix format
//FOV is given in radians
const Matrix4 MatrixFactory::frustum( T horizontalFOV, T verticalFOV, T zNear, T zFar, MultiplicationOrder o )
{
	T w = T(1)/TAN(horizontalFOV * T(0.5)), h = T(1)/TAN(verticalFOV * T(0.5)), ooFN = T(1)/(zFar - zNear);
	Matrix4 frustum(   w, T(0),  T(0),                       T(0),
				    T(0),    h,  T(0),                       T(0),
				    T(0), T(0),  (zFar + zNear) * ooFN,      T(1),
				    T(0), T(0), -T(2) * zNear * zFar * ooFN, T(0) );
//	if( righthanded )
//		frustum[2].negate();
	if( o == LEFT_MULTIPLY )
		frustum.transpose();
	return frustum;
}

//==============================================================================================
//==============================================================================================

//The (left, bottom, near) and (right, top, near) parameters specify the points on the
//near clipping plane that are mapped to the lower-left and upper-right corners of the window,
//respectively, assuming that the eye is located at (0, 0, 0)
//this maps the view frustum to a box with dimensions [-1,1]
//this corresponds to OpenGL projection matrix format
const Matrix4 MatrixFactory::ortho( T left, T right, T bottom, T top, T zNear, T zFar, MultiplicationOrder o )
{
	T ooRL = T(1)/(right - left), ooTB = T(1)/(top - bottom), ooFN = T(1)/(zFar - zNear);
	Matrix4 ortho(  T(2) * ooRL,            T(0),                    T(0),                  T(0),
					T(0),                   T(2) * ooTB,             T(0),                  T(0),
					T(0),                   T(0),                    T(2) * ooFN,           T(0),
				   -(right + left) * ooRL, -(top + bottom) * ooTB,  -(zFar + zNear) * ooFN, T(1) );
//	if( righthanded )
//		ortho[2].negate();
	if( o == LEFT_MULTIPLY )
		ortho.transpose();
	return ortho;
}

//==============================================================================================
//==============================================================================================

//this maps the view frustum to a box with dimensions [-1,1] for x- and y-axis, and [0,1] for z-axis
//this corresponds to Direct3D projection matrix format
//FOV is given in radians
const Matrix4 MatrixFactory::frustum01( T horizontalFOV, T verticalFOV, T zNear, T zFar, MultiplicationOrder o )
{
	T w = T(1)/TAN(horizontalFOV * T(0.5)), h = T(1)/TAN(verticalFOV * T(0.5)), ooFN = T(1)/(zFar - zNear);
	Matrix4 frustum(   w, T(0),  T(0),                T(0),
				    T(0),    h,  T(0),                T(0),
				    T(0), T(0),  zFar * ooFN,         T(1),
				    T(0), T(0), -zFar * zNear * ooFN, T(0) );
//	if( righthanded )
//		frustum[2].negate();
	if( o == LEFT_MULTIPLY )
		frustum.transpose();
	return frustum;
}

//==============================================================================================
//==============================================================================================

};	//namespace VECTORMATRIX
};
