#ifndef _UTILS_H
#define _UTILS_H

#include "Vector.h"

inline m::Vector3 pointOnHemisphereCos(const m::Vector3 &n, float a, float b)
{
	float theta = acosf(sqrt(a));
	float phi = 2.0f * 3.14159265f * b;

	m::Vector3 v1, v2;

	if (fabs(n.x) < 0.5f)
	{
		v1 = cross(n, m::Vector3(1.0f, 0.0f, 0.0f));
	}
	else
	{
		v1 = cross(n, m::Vector3(0.0f, 0.0f, 1.0f));
	}

	v1.normalize();

	v2 = cross(v1, n);

	return
		v1 * (cosf(phi) * sinf(theta)) +
		v2 * (sinf(phi) * sinf(theta)) +
		n * cosf(theta);
}

inline m::Vector3 pointOnHemisphere(float u, float v)
{
	float theta = acosf(sqrtf(u)) * 2.0f - 3.14159265f;
	float phi = 2.0f * 3.14159265f * v;

	return m::Vector3(
		cosf(phi) * sinf(theta),
		sinf(phi) * sinf(theta),
		cosf(theta));
}

inline float halton2(int k)
{
	int ret = 0;
	int n = 1;

	while (k)
	{
		ret <<= 1;
		if (k & 1)
			ret |= 1;
		k >>= 1;
		n <<= 1;
	}

	return ret / (float)n;
}

inline float halton(int k, int p)
{
	int pp = p;
	int kk = k;
	float res = 0;

	while (kk > 0)
	{
		int a = kk % p;
		res += float(a) / pp;
		kk /= p;
		pp *= p;
	}

	return res;
}

inline bool triangleIntersect(const m::Vector3& O, const m::Vector3& D, const m::Vector3& V0, const m::Vector3& V1, const m::Vector3& V2, float &_t, float &_u, float &_v)
{
	const m::Vector3 E1 = V1 - V0;
	const m::Vector3 E2 = V2 - V0;
	const m::Vector3 P = cross(D, E2);

	float det = m::dot(E1, P);

	if (fabs(det) < 0.00001f)
		return false;

	float invDet = 1.0f / det;

	const m::Vector3 T = O - V0;

	float u = dot(T, P) * invDet;

	if (u < 0.0f || u > 1.0f)
		return false;

	const m::Vector3 Q = cross(T, E1);

	float v = dot(D, Q) * invDet;
	if (v < 0.0f || u + v > 1.0f)
		return false;

	float t = dot(E2, Q) * invDet;
	if (t < 0.0f)
		return false;

	_t = t;
	_u = u;
	_v = v;
	return true;
}

/* TODO: Pass inversed directions as argument. */
inline bool aaboxIntersect(const m::Vector3& p, const m::Vector3& d, const m::Vector3& min, const m::Vector3& max, float &a, float &b)
{
	float t1;
	float t2;
	float t3;
	float t4;

	/* TODO: handle these properly. */
	if (d.x == 0.0f || d.y == 0.0f || d.z == 0.0f)
		return false;

	if (d.x > 0.0f)
	{
		t1 = (min.x - p.x) / d.x;
		t2 = (max.x - p.x) / d.x;
	}
	else
	{
		t1 = (max.x - p.x) / d.x;
		t2 = (min.x - p.x) / d.x;
	}
	assert(t1 <= t2);

	if (d.y > 0.0f)
	{
		t3 = (min.y - p.y) / d.y;
		t4 = (max.y - p.y) / d.y;
	}
	else
	{
		t3 = (max.y - p.y) / d.y;
		t4 = (min.y - p.y) / d.y;
	}

	if (t2 < t3 || t4 < t1)
		return false;

	if (t1 < t3)
		t1 = t3;
	if (t2 > t4)
		t2 = t4;

	assert(t1 <= t2);

	if (d.z > 0.0f)
	{
		t3 = (min.z - p.z) / d.z;
		t4 = (max.z - p.z) / d.z;
	}
	else
	{
		t3 = (max.z - p.z) / d.z;
		t4 = (min.z - p.z) / d.z;
	}

	if (t2 < t3 || t4 < t1)
		return false;

	if (t1 < t3)
		t1 = t3;
	if (t2 > t4)
		t2 = t4;

	assert(t1 <= t2);

	a = t1;
	b = t2;

	return true;
}

#include <math.h>

#ifndef M_PI
#  define M_PI 3.14159265f
#endif

/* Disk mappings are from paper "A Low Distortion Map Between Disk and Square". */

/*
 This transforms points on [0,1]^2 to points on unit disk centered at
 origin.  Each "pie-slice" quadrant of square is handled as a seperate
 case.  The bad floating point cases are all handled appropriately.
 The regions for (a,b) are:

        phi = pi/2
       -----*-----
       |\       /|
       |  \ 2 /  |
       |   \ /   |
 phi=pi* 3  *  1 *phi = 0
       |   / \   |
       |  / 4 \  |
       |/       \|
       -----*-----
        phi = 3pi/2

change log:
    26 feb 2004.  bug fix in computation of phi (now this matches the paper,
                  which is correct).  thanks to Atilim Cetin for catching this.
*/


/* seedx, seedy is point on [0,1]^2.  x, y is point on radius 1 disk */
inline m::Vector2 mapSquareToDisk(const m::Vector2& p)
{

   double phi, r;

   double a = 2*p.x - 1;   /* (a,b) is now on [-1,1]^2 */
   double b = 2*p.y - 1;

   if (a > -b) {     /* region 1 or 2 */
       if (a > b) {  /* region 1, also |a| > |b| */
           r = a;
           phi = (M_PI/4 ) * (b/a);
       }
       else       {  /* region 2, also |b| > |a| */
           r = b;
           phi = (M_PI/4) * (2 - (a/b));
       }
   }
   else {        /* region 3 or 4 */
       if (a < b) {  /* region 3, also |a| >= |b|, a != 0 */
            r = -a;
            phi = (M_PI/4) * (4 + (b/a));
       }
       else       {  /* region 4, |b| >= |a|, but a==0 and b==0 could occur. */
            r = -b;
            if (b != 0)
                phi = (M_PI/4) * (6 - (a/b));
            else
                phi = 0;
       }
   }

	return m::Vector2((float)(r * cos(phi)), (float)(r * sin(phi)));

}

/* diskx, disky is point on radius 1 disk.  x, y is point on [0,1]^2 */
inline m::Vector2 mapDiskToSquare(const m::Vector2& p)
{
    double r = sqrt( p.x*p.x + p.y*p.y );
    double phi = atan2( p.y, p.x );
    double a, b;
    if (phi < -M_PI/4) phi += 2*M_PI;  /* in range [-pi/4,7pi/4] */
    if ( phi < M_PI/4) {         /* region 1 */
        a = r;
        b = phi * a / (M_PI/4);
    }
    else if ( phi < 3*M_PI/4 ) { /* region 2 */
        b = r;
        a = -(phi - M_PI/2) * b / (M_PI/4);
    }
    else if ( phi < 5*M_PI/4 ) { /* region 3 */
        a = -r;
        b = (phi - M_PI) * a / (M_PI/4);
    }
    else {                       /* region 4 */
        b = -r;
        a = -(phi - 3*M_PI/2) * b / (M_PI/4);
    }

	return m::Vector2((float)(a + 1) / 2.0f, (float)(b + 1) / 2.0f);
}

static inline m::Vector2 intersect_circle(const m::Vector2& a, const m::Vector2& b)
{
	assert(a.length() < 1.0001f);
	assert(b.length() > 0.9999f);

	m::Vector2 d = b - a;
	float dl = d.length();

	float D = a.x * b.y - b.x * a.y;

	float sgn = (d.y < 0.0f) ? -1.0f : 1.0f;

	float x = (D * d.y + sgn * d.x * sqrtf(dl*dl - D*D)) / (dl*dl);
	float y = (-D * d.x + fabs(d.y) * sqrtf(dl*dl - D*D)) / (dl*dl);

	float x2 = (D * d.y - sgn * d.x * sqrtf(dl*dl - D*D)) / (dl*dl);
	float y2 = (-D * d.x - fabs(d.y) * sqrtf(dl*dl - D*D)) / (dl*dl);

	if (d.y < 0)
		return m::Vector2(x2, y2);
	else
		return m::Vector2(x, y);
}

static inline float cross(m::Vector2 a, m::Vector2 b)
{
	return a.x * b.y - a.y * b.x;
}

static inline float triangle_area(const m::Vector2& a, const m::Vector2& b, const m::Vector2& c)
{
	return fabs(cross(b - a, c - a) / 2.0f);
}

static inline float area_of_segment(float r, float d)
{
	assert(d >= 0.0f);

	return r * r * acosf(d / r) - d * sqrtf(r*r - d*d);
}

static inline float safe_acosf(float f)
{
	return acosf((f < -1.0f) ? -1.0f : ((f > 1.0f) ? 1.0f : f));
}

static inline float sector_area(const m::Vector2& a, const m::Vector2& b, const m::Vector2& p)
{
	float A = safe_acosf(dot(a, b))/2.0f - triangle_area(a, b, m::Vector2(0, 0));

	if (cross(a - b, a - p) * cross(a - b, a - m::Vector2(0, 0)) > 0.0f)
		A += triangle_area(a, b, p);
	else
		A -= triangle_area(a, b, p);

	return A;
}

static inline float sphere_angle(const m::Vector3& p, const m::Vector3& v0, const m::Vector3& v1)
{
	assert(p.length() > 0.999f && p.length() < 1.001f);
	return safe_acosf(dot(normalize(v0 - p - p * dot(p, v0 - p)), normalize(v1 - p - p * dot(p, v1 - p))));
	//return 1.0f;
}

static inline float triangle_area_on_sphere(const m::Vector3& a, const m::Vector3& b, const m::Vector3& c)
{
	float sum = sphere_angle(a, b, c) + sphere_angle(b, c, a) + sphere_angle(c, a, b);
	return sum - 3.14159265f;
}

template<class T>
static inline m::Vector2 make_vec2(const T& v)
{
	return m::Vector2(v.x, v.y);
}

static inline void export_targa(const char* filename, int w, int h, unsigned char* fp)
{
	const int G_WIDTH = w;
	const int G_HEIGHT = h;
	FILE* targa = fopen(filename, "wb");
	if (!targa)
	{
		fprintf(stderr, "Can't open %s\n", filename);
		return;
	}
	int xsize = G_WIDTH;
	int ysize = G_HEIGHT;

	char hdr[18] = {0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x18, 0};
	*((short int*)(hdr + 12)) = (short int)w;
	*((short int*)(hdr + 14)) = (short int)h;
	fwrite(hdr, 1, 18, targa);
	for (int i=0; i < xsize*ysize; i++)
	{
		unsigned char r = *fp++;
		unsigned char g = *fp++;
		unsigned char b = *fp++;
		fwrite(&b, 1, 1, targa);
		fwrite(&g, 1, 1, targa);
		fwrite(&r, 1, 1, targa);
	}
	fclose(targa);
}

static inline m::Vector3 get_catmull_rom(float t, const m::Vector3& a, const m::Vector3& b, const m::Vector3& c, const m::Vector3& d)
{
	return 0.5f * ((2.0f * b) +
	               (-a + c) * t +
	               (2.0f*a - 5.0f*b + 4.0f*c - d) * t*t +
                   (-a + 3.0f*b- 3.0f*c + d) * t*t*t);
}

#endif
