#ifdef WIN32
#define _WIN32_WINNT 0x0500
//#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include "glutil.h"
#include "GL/glut.h"
#include "objloader.h"
#include "trimesh.h"
#include "light.h"
#include "glutil.h"
#include "Timer.h"
#include "deferred.h"
#include "shaders.h"
#include "kdtree.h"
#include "raytracer.h"
#include "textures.h"
#include "VectorMatrix.h"
#include "configuration.h"
#include "timing.h"
#include "utils.h"
#include "camera.h"

static Light* light;
static Trimesh* trimesh;
static Trimesh* trimeshHigh;
static Trimesh* trimeshCupid;
static Trimesh* trimeshElukka;
static Trimesh* flashLight;
static RayTracer* rt;
static Deferred* deferred;
static float timeOffset = 0.0f;
static int frame = 0;

static GLuint deferredProgram;
static GLuint filterProgram;
static GLuint deferredDirectProgram;
static GLuint ambientProgram;
static GLuint mul16Program;

static Camera camera;

static std::vector<m::Vector3> cameraPosPath;
static std::vector<m::Vector3> cameraDirPath;

static bool visualizeDepthMap = false;
static int depthMap = -1;
static bool visualizeSecondaryLights = false;
static bool visualizeDirectLighting = true;
static bool visualizeIndirectLighting = true;
static bool visualizeVoronoi = false;
static bool visualizeLightPath = false;

static const float camPath2[][6] = {
{ 7.854963, 11.968521, -0.411781, -0.841957, -0.505657, 0.188200 },
{ 8.144773, 10.357756, 1.053108, -0.908080, -0.418660, 0.010699 },
};

static const float camPath3[][6] = {
{ -1.798370, 16.739481, -9.063990, 0.419114, -0.693087, 0.586492 },
{ 0.825134, 7.511816, -6.334575, 0.514776, -0.529179, 0.674518 },
{ 6.828928, 4.162609, -6.532404, -0.423538, -0.241075, 0.873212 },
};

static const float camPath4[][6] = {
{ -5.648387, 9.744897, 7.576984, 0.785027, -0.436409, -0.439636 },
{ -0.178292, 9.270321, 7.897611, -0.189399, -0.790155, -0.582909 },
};

static void init()
{
	GLenum err = glewInit();
	assert(err == GLEW_OK);

	visualizeDirectLighting = CONFI(render_direct) ? true : false;
	visualizeIndirectLighting = CONFI(render_indirect) ? true : false;
	visualizeSecondaryLights = CONFI(render_toekoet) ? true : false;
	visualizeVoronoi = !!CONFI(render_voronoi);

	deferredProgram = MAKE_GLSL_PROGRAM(deferredDirectLightingFrag, deferredVertex);
	filterProgram = MAKE_GLSL_PROGRAM(filterDeferredFrag, NULL);
	deferredDirectProgram = MAKE_GLSL_PROGRAM(deferredDirectFrag, NULL);
	ambientProgram = MAKE_GLSL_PROGRAM(ambientFrag, NULL);
	mul16Program = MAKE_GLSL_PROGRAM(mul16Frag, NULL);

	assert(glGetError() == GL_NO_ERROR);

	deferred = new Deferred();
	assert(glGetError() == GL_NO_ERROR);

	rt = new RayTracer();

	/* Flash light. */

	ObjLoader obj2("data/lamppu.obj", CONFF(light_size));
	assert(glGetError() == GL_NO_ERROR);
	flashLight = new Trimesh(obj2, false);

	/* Load the scene. */

	ObjLoader obj((std::string("data/") + CONFS(obj_file)).c_str(), CONFF(obj_scale));
	if (CONFI(obj_swap_yz))
		obj.swapYZ();
	obj.translate(
		CONFF(obj_trans_x) * CONFF(obj_scale),
		CONFF(obj_trans_y) * CONFF(obj_scale),
		CONFF(obj_trans_z) * CONFF(obj_scale));

	rt->feedObject(obj);

	trimesh = new Trimesh(obj, false);

	trimeshHigh = new Trimesh(obj, false);
	trimeshHigh->subdivide(CONFF(subdivide_edge));

	config.setInt("flat_normals", 0);

	/* Load cubid. */

	if (CONFI(cupid))
	{
		ObjLoader objCupid("data/cupid2.obj", CONFF(cupid_scale));
		objCupid.negateXZ();
		objCupid.translate(CONFF(cupid_x), CONFF(cupid_y), CONFF(cupid_z));
		trimeshCupid = new Trimesh(objCupid, false);
		rt->feedObject(objCupid);
	}

	/* Load elukka. */

	if (CONFI(elukka))
	{
		ObjLoader objElukka("data/cupid3.obj", CONFF(elukka_scale));
		trimeshElukka = new Trimesh(objElukka, true);
	}

	/* Print statistics. */

	printf("mesh + flash light + cupid = total\n");

	printf("LO TRIANGLES %d + %d + %d = %d\n",
		trimesh->triangles.size(),
		flashLight->triangles.size(),
		trimeshCupid ? trimeshCupid->triangles.size() : 0,
		trimesh->triangles.size() + (trimeshCupid ? trimeshCupid->triangles.size() : 0) + flashLight->triangles.size());

	printf("HI TRIANGLES %d + %d + %d = %d\n",
		trimeshHigh->triangles.size(),
		flashLight->triangles.size(),
		trimeshCupid ? trimeshCupid->triangles.size() : 0,
		trimeshHigh->triangles.size() + (trimeshCupid ? trimeshCupid->triangles.size() : 0) + flashLight->triangles.size());

	/* Make ray tracer. */

	rt->update();

	/* Make light. */

	light = new Light(CONFI(secondary_lights));
	light->type = CONFI(spot) ? Light::SPOT : Light::OMNI;
	light->setPosition(m::Vector3(1.0f, 1.0f, 1.0f));
	light->setDirection(m::Vector3(0.0f, 0.0f, 1.0f));
	light->setIntensity(m::Vector3(1.0f, 1.0f, 1.0f) * (float)CONFF(light_intensity));

	switch (CONFI(camera_path))
	{
	case 2:
		for (unsigned int i = 0; i < sizeof(camPath2) / sizeof(camPath2[0]); i++)
		{
			cameraPosPath.push_back(m::Vector3(camPath2[i][0], camPath2[i][1], camPath2[i][2]));
			cameraDirPath.push_back(m::Vector3(camPath2[i][3], camPath2[i][4], camPath2[i][5]));
		}
		break;
	case 3:
		for (unsigned int i = 0; i < sizeof(camPath3) / sizeof(camPath3[0]); i++)
		{
			cameraPosPath.push_back(m::Vector3(camPath3[i][0], camPath3[i][1], camPath3[i][2]));
			cameraDirPath.push_back(m::Vector3(camPath3[i][3], camPath3[i][4], camPath3[i][5]));
		}
		break;
	case 4:
		for (unsigned int i = 0; i < sizeof(camPath4) / sizeof(camPath4[0]); i++)
		{
			cameraPosPath.push_back(m::Vector3(camPath4[i][0], camPath4[i][1], camPath4[i][2]));
			cameraDirPath.push_back(m::Vector3(camPath4[i][3], camPath4[i][4], camPath4[i][5]));
		}
		break;
	}

	timer.init();
}

static void reshape(int w, int h)
{
	glViewport(0, 0, w, h);
	glutPostRedisplay();
}

static void keyboard(unsigned char k, int x, int y)
{
	switch (k)
	{
		case 'q':
			visualizeDepthMap = visualizeDepthMap ? false : true;
			if (visualizeDepthMap == false)
				depthMap = -1;
			break;

		case 'w':
			visualizeSecondaryLights = visualizeSecondaryLights ? false : true;
			break;

		case 'e':
			visualizeDirectLighting = visualizeDirectLighting ? false : true;
			break;

		case 'r':
			visualizeIndirectLighting = visualizeIndirectLighting ? false : true;
			break;

		case 't':
			visualizeVoronoi = visualizeVoronoi ? false : true;
			break;

		case 'y':
			visualizeLightPath = visualizeLightPath ? false : true;
			break;

		case 'a':
			depthMap = (depthMap + 1) % light->subLightCount;
			break;

		case 'z':
			timeOffset -= 1.0f;
			break;

		case 'x':
			timeOffset += 1.0f;
			break;

		case 'o':
			light->outputPS();
			break;

		case 'v':
			timer.init();
			break;

		case 'c':
			{
				static int tr = 0;
				FILE* fp = fopen("campath.h", tr ? "a" : "w");
				tr = 1;
				fprintf(fp, "{ %f, %f, %f, %f, %f, %f },\n", camera.pos.x, camera.pos.y, camera.pos.z, camera.dir.x, camera.dir.y, camera.dir.z);
				fclose(fp);
			}
			break;

		case 27:
			exit(0);
	}

	glutPostRedisplay();
}

static std::set<int> keyPressed;

static void special(int k, int x, int y)
{
	keyPressed.insert(k);
}

static void specialUp(int k, int x, int y)
{
	keyPressed.erase(k);
}

static int mx, my;
static float origCameraYRot = 0.0f;
static float origCameraXRot = 0.0f;
static m::Vector3 origPosition;
static m::Vector3 origRotation;
static Trimesh* movingObject = NULL;
static Trimesh* rotatingObject = NULL;
static bool rotatingCamera = false;

static void set_perspective_mode(bool);

static void mouse(int but, int down, int x, int y)
{
	mx = x;
	my = y;
	origCameraYRot = camera.y_rot;
	origCameraXRot = camera.x_rot;

	if (but == 0 && down)
		movingObject = NULL;

	if (but == 1 && down)
		rotatingObject = NULL;

	if (but == 2)
		rotatingCamera = !down;
}

static void motion(int x, int y)
{
	int dx = x - mx;
	int dy = y - my;
	float sw = 1.0f / CONFI(screen_width);
	float sh = 1.0f / CONFI(screen_height);

	if (movingObject)
	{
		float s = dot(camera.dir, origPosition - camera.pos);
		sw *= s;
		sh *= s;
		movingObject->position = origPosition + camera.up * (dy * sh) + camera.right * (-dx * sw);
	}

	if (rotatingObject)
	{
		sw *= 180.0f;
		sh *= 180.0f;
		rotatingObject->rotation = origRotation + camera.right * (dy * sh) + camera.up * (-dx * sw);
	}

	if (rotatingCamera)
	{
		camera.y_rot = origCameraYRot - dx / 400.0f * 360.0f / 4.0f;
		camera.x_rot = origCameraXRot - dy / 400.0f * 360.0f / 4.0f;
		camera.update_dirs();
	}
}

/*
 * Scene renderings
 */

static void renderScene(void*)
{
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	if (CONFI(culling))
		glEnable(GL_CULL_FACE);
	else
		glDisable(GL_CULL_FACE);
	if (trimesh)
		trimesh->render();
	if (trimeshCupid)
		trimeshCupid->render();
	if (trimeshElukka)
		trimeshElukka->render();

#if 1
	glPushMatrix();
	glLoadIdentity();
	glRotatef(90, 1, 0, 0);
	gluLookAt(light->shadowMap.pos.x, light->shadowMap.pos.y, light->shadowMap.pos.z,
		light->shadowMap.pos.x + light->shadowMap.dir.x,
		light->shadowMap.pos.y + light->shadowMap.dir.y,
		light->shadowMap.pos.z + light->shadowMap.dir.z,
		0, 1, 0);

	JC::VECTORMATRIX::Matrix4 mtx;
	glGetFloatv(GL_MODELVIEW_MATRIX, &mtx[0][0]);
	glPopMatrix();
	mtx.invert();
	glMultMatrixf(&mtx[0][0]);
	if (CONFI(spot))
		flashLight->render();
	else
		glutSolidSphere(0.5f * CONFF(light_size), 16, 16);
	glPopMatrix();
#endif
}

static void renderStaticSceneDepth(void*)
{
	/* This is rendered using paraboloid mapping so scene must be tesselated. */
	if (CONFI(culling))
		glEnable(GL_CULL_FACE);
	else
		glDisable(GL_CULL_FACE);
	if (trimeshHigh && trimesh)
	{
		trimeshHigh->position = trimesh->position;
		trimeshHigh->rotation = trimesh->rotation;
	}
	if (trimeshHigh)
		trimeshHigh->renderDepth();
	if (trimeshCupid)
		trimeshCupid->renderDepth();
}

static void renderSceneDepth(void*)
{
	/* For direct lighting shadows. */
	if (CONFI(culling))
		glEnable(GL_CULL_FACE);
	else
		glDisable(GL_CULL_FACE);
	if (trimesh)
		trimesh->renderDepth();
	if (trimeshCupid)
		trimeshCupid->renderDepth();
	if (trimeshElukka)
		trimeshElukka->renderDepth();
}

/*
 * Misc
 */

static void renderCircle(float x, float y, float r, GLenum m)
{
	glBegin(m);
	for (int i = 0; i < 256; i++)
		glVertex2f(cos(i / 256.0f * 2 * 3.14159265f) * r + x, sin(i / 256.0f * 2 * 3.14159265f) * r + y);
	glEnd();
}

static void renderLineOnSphere(const m::Vector3& a, const m::Vector3& b)
{
	glBegin(GL_LINE_STRIP);
	for (int i = 0; i <= 32; i++)
	{
		m::Vector3 p = normalize(a + (b - a) * (i / 32.0f));
		float c = (p.z + 1.0f) / 2.0f;
		glColor3f(0.2f*c, 1.0f*c, 0.2f*c);
		glVertex3fv(&p.x);
	}
	glEnd();
}

/*
 * Paths
 */

static m::Vector3 getLightPath(float t)
{
	t += CONFF(light_time_offset);
	return m::Vector3(
		CONFF(light_x) + cosf(t * CONFF(light_speed_xz)) * CONFF(light_x_scale),
		CONFF(light_y) + cosf(t * CONFF(light_speed_y)) * CONFF(light_y_scale),
		CONFF(light_z) + sinf(t * CONFF(light_speed_xz)) * CONFF(light_z_scale));
}

static m::Vector3 getElukkaPath(float t)
{
	t += CONFF(light_time_offset);
	t = t * 0.7f - 3.5f;
	return m::Vector3(
		CONFF(light_x) + cosf(t * CONFF(light_speed_xz)) * CONFF(light_x_scale) * 1.0f,
		CONFF(light_y) + cosf(t * CONFF(light_speed_y)) * CONFF(light_y_scale) + 1.2f,
		CONFF(light_z) + sinf(t * CONFF(light_speed_xz)) * CONFF(light_z_scale) * 1.0f);
}

static void getCameraPath(float t, m::Vector3& p, m::Vector3& d)
{
	t /= CONFF(camera_path_scale);

	int i = (int)t;
	t -= (float)i;

	int i0 = std::max(0, std::min((int)cameraPosPath.size()-1, i-1));
	int i1 = std::max(0, std::min((int)cameraPosPath.size()-1, i));
	int i2 = std::max(0, std::min((int)cameraPosPath.size()-1, i+1));
	int i3 = std::max(0, std::min((int)cameraPosPath.size()-1, i+2));
	p = get_catmull_rom(t, cameraPosPath[i0], cameraPosPath[i1], cameraPosPath[i2], cameraPosPath[i3]);
	d = get_catmull_rom(t, cameraDirPath[i0], cameraDirPath[i1], cameraDirPath[i2], cameraDirPath[i3]);
}

/*
 * Pespective and ortho matrices
 */

static void set_perspective_mode(bool projectionIdentity = true)
{
	glMatrixMode(GL_PROJECTION);
	if (projectionIdentity)
		glLoadIdentity();
	gluPerspective(80.0f, CONFI(screen_width) / (float)CONFI(screen_height), 0.1f, 40.0f);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(
		camera.pos.x, camera.pos.y, camera.pos.z,
		camera.pos.x+camera.dir.x, camera.pos.y+camera.dir.y, camera.pos.z+camera.dir.z,
		0, 1, 0);

	glEnable(GL_DEPTH_TEST);

	assert(glGetError() == GL_NO_ERROR);
}

static void set_ortho_mode()
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, 1, 0, 1, -1, 1);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glDisable(GL_DEPTH_TEST);
}

/*
 * Rendering steps
 */

static void render_direct_light_shadow_map()
{
	light->renderPrimaryShadowMap(renderSceneDepth, NULL);
	REPORT_TIME("primary light");
	assert(glGetError() == GL_NO_ERROR);
}

static void render_gbuffer()
{
	set_perspective_mode();

	deferred->renderScene(renderScene, NULL);
	REPORT_TIME("gbuffer");
	assert(glGetError() == GL_NO_ERROR);
}

static void render_direct_lighting()
{
	set_ortho_mode();

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, deferred->c3.fb);

	glUseProgramObjectARB(deferredDirectProgram);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, deferred->g1.tex[0]);
	setUniform(deferredDirectProgram, "positionMap", 0U);

	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D, deferred->g1.tex[1]);
	setUniform(deferredDirectProgram, "normalMap", 1U);

	glActiveTexture(GL_TEXTURE3);
	glBindTexture(GL_TEXTURE_2D, deferred->g1.tex[2]);
	setUniform(deferredDirectProgram, "realColorMap", 3U);

	glActiveTexture(GL_TEXTURE4);
	glDisable(GL_TEXTURE_2D);
	glEnable(GL_TEXTURE_CUBE_MAP);
	glBindTexture(GL_TEXTURE_CUBE_MAP, light->shadowMap.texture);
	glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_COMPARE_MODE_ARB, GL_NONE);
	setUniform(deferredDirectProgram, "shadowMap", 4U);

	setUniform(deferredDirectProgram, "lightPosition", light->position);
	setUniform(deferredDirectProgram, "lightDirection", light->direction);
	setUniform(deferredDirectProgram, "lightIntensity", light->intensity * (visualizeDirectLighting ? 1.0f : 0.0f));
	float m[16];
	light->shadowMap.getMatrix(m);
	setUniform4x4Mat(deferredDirectProgram, "lightMatrix", m);

	glBegin(GL_QUADS);
	glVertex2f(0.0f, 0.0f);
	glVertex2f(1.0f, 0.0f);
	glVertex2f(1.0f, 1.0f);
	glVertex2f(0.0f, 1.0f);
	glEnd();

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	REPORT_TIME("direct lighting");
	assert(glGetError() == GL_NO_ERROR);
}

static void do_mosaic()
{
	deferred->do_mosaic();
	REPORT_TIME("mosaic");
	assert(glGetError() == GL_NO_ERROR);
}

static void update_secondary_lights()
{
	light->updateSecondaryLights(rt, renderStaticSceneDepth, NULL, false);
	REPORT_TIME("secondary lights");
	assert(glGetError() == GL_NO_ERROR);
}

static void blend_indirect_lights()
{
	deferred->prepare(deferredProgram);
	assert(glGetError() == GL_NO_ERROR);

	/* TODO: remove clear and use non-blending for the first pass. */
	glClear(GL_COLOR_BUFFER_BIT);

	if (visualizeIndirectLighting == false)
		return;

	glUseProgramObjectARB(deferredProgram);

	set_ortho_mode();

	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	glEnable(GL_TEXTURE_2D);
	glColor3f(1, 1, 1);

	glEnable(GL_BLEND);
	glBlendFunc(GL_ONE, GL_ONE);

	setUniform(deferredProgram, "shadowMap", 3U);
	glActiveTexture(GL_TEXTURE3);

	int split_n = CONFI(split_n);
	int split_m = CONFI(split_m);

	/* TODO: fix this to be more cache coherent. */
	for (int i = 0; i < light->subLightCount; i++)
	{
		Light::SubLight& sl = light->subLights[i];
		if (!sl.valid)
			continue;

		glBindTexture(GL_TEXTURE_2D, sl.shadowMap.texture);

		setUniform(deferredProgram, "width", (float)sl.shadowMap.size);
		if (CONFI(shadowmap_hack))
		{
			setUniform(deferredProgram, "offsetX", sl.shadowMap.offsetX / (float)(sl.shadowMap.size * 2));
			setUniform(deferredProgram, "offsetY", sl.shadowMap.offsetY / (float)(sl.shadowMap.size * 2));
		}
		setUniform(deferredProgram, "lightPosition", sl.position + sl.direction * 0.05f);
		setUniform(deferredProgram, "lightDirection", sl.direction);
		setUniform(deferredProgram, "lightIntensity", sl.intensity * (float)CONFF(indirect_boost) * (float)(split_n * split_m));
		float m[16];
		sl.shadowMap.getMatrix(m);
		setUniform4x4Mat(deferredProgram, "lightMatrix", m);

		float x = (i % split_n) / (float)split_n;
		float y = ((i / split_n) % split_m) / (float)split_m;

		glBegin(GL_QUADS);
		glVertex2f(x, y);
		glVertex2f(x + 1.0f / split_n, y);
		glVertex2f(x + 1.0f / split_n, y + 1.0f / split_m);
		glVertex2f(x, y + 1.0f / split_m);
		glEnd();
	}
	REPORT_TIME("blend sublights");

	assert(glGetError() == GL_NO_ERROR);

	glBindTexture(GL_TEXTURE_2D, 0);
	glActiveTexture(GL_TEXTURE0);

	glDisable(GL_BLEND);
}

static void do_demosaic()
{
	deferred->done();
	REPORT_TIME("demosaic");

	glDisable(GL_TEXTURE_2D);

	assert(glGetError() == GL_NO_ERROR);
}

static void filter_indirect_to_screen()
{
	set_ortho_mode();

	glUseProgramObjectARB(filterProgram);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, deferred->g1.tex[0]);
	setUniform(filterProgram, "positionMap", 0U);

	glActiveTexture(GL_TEXTURE1);
	glBindTexture(GL_TEXTURE_2D, deferred->g1.tex[1]);
	setUniform(filterProgram, "normalMap", 1U);

	glActiveTexture(GL_TEXTURE2);
	glBindTexture(GL_TEXTURE_2D, deferred->c2.tex);
	setUniform(filterProgram, "colorMap", 2U);

	glActiveTexture(GL_TEXTURE3);
	glBindTexture(GL_TEXTURE_2D, deferred->g1.tex[2]);
	setUniform(filterProgram, "realColorMap", 3U);

	glActiveTexture(GL_TEXTURE4);
	glBindTexture(GL_TEXTURE_2D, deferred->c3.tex);
	setUniform(filterProgram, "directLightingMap", 4U);

	assert(glGetError() == GL_NO_ERROR);

	glBegin(GL_QUADS);
	glVertex2f(0.0f, 0.0f);
	glVertex2f(1.0f, 0.0f);
	glVertex2f(1.0f, 1.0f);
	glVertex2f(0.0f, 1.0f);
	glEnd();

	REPORT_TIME("filtering indirect");

	/* Clean up. */

	assert(glGetError() == GL_NO_ERROR);

	glDisable(GL_TEXTURE_2D);
	glEnable(GL_DEPTH_TEST);

	glActiveTexture(GL_TEXTURE4);
	glDisable(GL_TEXTURE_CUBE_MAP);
	glBindTexture(GL_TEXTURE_CUBE_MAP, 0);
	glActiveTexture(GL_TEXTURE0);
	glUseProgramObjectARB(0);
}

void render_ambient()
{
	set_ortho_mode();

	glUseProgramObjectARB(ambientProgram);

	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, deferred->g1.tex[2]);
	setUniform(ambientProgram, "realColorMap", 0U);

	glEnable(GL_BLEND);
	glBlendFunc(GL_ONE, GL_ONE);
	glBegin(GL_QUADS);
	glVertex2f(0.0f, 0.0f);
	glVertex2f(1.0f, 0.0f);
	glVertex2f(1.0f, 1.0f);
	glVertex2f(0.0f, 1.0f);
	glEnd();
	glDisable(GL_BLEND);

	REPORT_TIME("ambient");
	
	glDisable(GL_TEXTURE_2D);
	glUseProgramObjectARB(0);
}

/*
 * Main display function
 */

static void display()
{
#ifdef WIN32
	if ((frame % 50) == 0)
		system("cls");
	COORD cpos = { 0, 0 };
	SetConsoleCursorPosition(GetStdHandle(STD_OUTPUT_HANDLE), cpos);
#endif

	timer_individuals = CONFI(timer_individuals);

	float delta;
	{
		static double last = 0.0f;
		double current = timer.time();
		delta = (float)(current - last);
		last = current;
	}

	delta *= 5.0f;

	if (keyPressed.find(GLUT_KEY_UP) != keyPressed.end())
		camera.pos += camera.dir * delta;

	if (keyPressed.find(GLUT_KEY_DOWN) != keyPressed.end())
		camera.pos += camera.dir * -delta;

	if (keyPressed.find(GLUT_KEY_LEFT) != keyPressed.end())
		camera.pos += camera.right * delta;

	if (keyPressed.find(GLUT_KEY_RIGHT) != keyPressed.end())
		camera.pos += camera.right * -delta;

	double frameStartTime = timer.time();
	prevTime = frameStartTime;

	float time = frameStartTime + timeOffset;
	//float time = timeOffset;

	printf("RENDERER: %s\tVERSION: %s\n", glGetString(GL_RENDERER), glGetString(GL_VERSION));

	glEnable(GL_CULL_FACE);

	if (CONFI(record_video))
		time = frame / 50.0f;
	//time = 0.3f;

	/* Set light. */

	light->setPosition(getLightPath(time + 0.01f));
	float rosp = time * CONFF(light_rotation_speed) + 0.01f;
	light->setDirection(normalize(m::Vector3(cosf(rosp * 0.3f), sinf(rosp * 0.2f) * 0.7f, sinf(rosp * 0.3f))));

	/* Set elukka. */

#if 1
	if (trimeshElukka)
	{
		m::Vector3 delta = normalize(getElukkaPath(time + 0.01f) - getElukkaPath(time));
		trimeshElukka->position = getElukkaPath(time);
		trimeshElukka->rotation.x = 0.0f;
		trimeshElukka->rotation.y = atan2f(delta.x, delta.z) / 3.14159265f * 180.0f - 0.0f;
		trimeshElukka->rotation.z = -acosf(delta.y) / 3.14159265f * 180.0f + 90.0f;
	}
#endif

	/* Render all light maps on first frame. */

	if (frame == 0)
	{
		light->updateSecondaryLights(rt, renderStaticSceneDepth, NULL, true);
	}

	/* Set camera. */

	switch (CONFI(camera_path))
	{
	case 1:
		{
			float t = fmodf(time / 30.0f, 1.0f);
			m::Vector3 a = m::Vector3(-18, 1, 3);
			m::Vector3 b = m::Vector3(-4, 2,  2);

			camera.pos = a + (b - a) * t;
		}
		break;

	case 2:
	case 3:
	case 4:
	case 5:
	case 6:
		getCameraPath(time, camera.pos, camera.dir);
		break;
	}

	/* Do lighting. */

	REPORT_TIME("hox");

	render_direct_light_shadow_map();
	render_gbuffer();
	render_direct_lighting();
	do_mosaic();
	update_secondary_lights();
	blend_indirect_lights();
	do_demosaic();
	filter_indirect_to_screen();

	if (CONFI(render_ambient))
		render_ambient();

	/* Set perspective mode for other visualizations. */

	set_perspective_mode();

	/* Render depth buffer if secondary lights are visualized. */

	if (visualizeSecondaryLights || visualizeLightPath)
	{
		if (CONFI(culling))
			glEnable(GL_CULL_FACE);
		glClear(GL_DEPTH_BUFFER_BIT);
		glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
		renderSceneDepth(NULL);
		glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	}

	assert(glGetError() == GL_NO_ERROR);

	/* Visualize secondary light sources. */

	if (visualizeSecondaryLights)
	{
		glUseProgramObjectARB(0);
		glDisable(GL_TEXTURE_2D);

		for (int i = 0; i < light->subLightCount; i++)
		{
			if (light->subLights[i].valid == false)
				continue;

			glPushMatrix();
			JC::VECTORMATRIX::Matrix4 mtx;
			memcpy(&mtx[0][0], light->subLights[i].shadowMap.transform, 16*4);
			mtx.invert();
			glMultMatrixf(&mtx[0][0]);

			if (i == depthMap)
				glColor3f(1.0f, 0.5f, 0.5f);
			else
				glColor3fv(&light->subLights[i].intensity.x);
			glutSolidSphere(0.15f, 16, 16);

			glLineWidth(2.0f);
			glColor3f(1.0f, 1.0f, 1.0f);
			glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
			glutSolidSphere(0.15f, 16, 16);
			glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

			glLineWidth(5.0f);
			glColor3f(1.0f, 1.0f, 0.6f);
			glBegin(GL_LINES);
			glVertex3f(0, 0, 0);
			glVertex3f(0, 0, -0.5f);
			glEnd();

			glPopMatrix();
		}
	}

	assert(glGetError() == GL_NO_ERROR);

	/* Visualize light path. */

	if (visualizeLightPath)
	{
		glUseProgramObjectARB(0);
		glDisable(GL_TEXTURE_2D);
		glColor3f(1, 1, 1);
		glLineWidth(2.0f);
		glBegin(GL_LINE_STRIP);
		for (int i = 0; i < 256; i++)
		{
			m::Vector3 v = getLightPath(i / 256.0f * 3.14159265 * 2.0f * 2.0f);
			glVertex3fv(&v.x);
		}
		glEnd();
	}

	assert(glGetError() == GL_NO_ERROR);

	/* Visualize depth map. */

	if (visualizeDepthMap)
	{
		glUseProgramObjectARB(0);
		//setUniform(mul16Program, "map", 0U);
		//setUniform(mul16Program, "map2", 1U);

		glDisable(GL_DEPTH_TEST);

		glActiveTexture(GL_TEXTURE0);

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		glOrtho(0, 2, 0, 2, -1, 1);

		glDisable(GL_LIGHTING);
		glEnable(GL_TEXTURE_2D);

		if (depthMap == -1 || depthMap >= light->subLightCount)
			glBindTexture(GL_TEXTURE_CUBE_MAP, light->shadowMap.texture);
		else
			glBindTexture(GL_TEXTURE_2D, light->subLights[depthMap].shadowMap.texture);

		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE_ARB, GL_NONE);

		//glBindTexture(GL_TEXTURE_2D, deferred->c2.tex);
		//glActiveTexture(GL_TEXTURE1);
		glBindTexture(GL_TEXTURE_2D, deferred->g1.tex[2]);

		glColor3f(1, 1, 1);

		glBegin(GL_QUADS);
		glTexCoord3f(0, 0, -1);
		glVertex2f(0, 0);
		glTexCoord3f(1, 0, -1);
		glVertex2f(1, 0);
		glTexCoord3f(1, 1, -1);
		glVertex2f(1, 1);
		glTexCoord3f(0, 1, -1);
		glVertex2f(0, 1);
		glEnd();

		glActiveTexture(GL_TEXTURE0);
		glDisable(GL_TEXTURE_2D);
		glEnable(GL_DEPTH_TEST);
	}

	assert(glGetError() == GL_NO_ERROR);

	/* Visualize 2d voronoi. */

	if (light->type == Light::SPOT && visualizeVoronoi)
	{
		glUseProgramObjectARB(0);
		glDisable(GL_DEPTH_TEST);

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		float asp = CONFF(screen_width) / CONFF(screen_height);
		glOrtho(-3 * asp, 1 * asp, -1, 3, -2, 2);
		glTranslatef(0.125 * asp, 0.125, 0);

		glDisable(GL_LIGHTING);
		glDisable(GL_TEXTURE_2D);

		glColor3f(0.1f, 0.1f, 0.4f);
		renderCircle(0.0f, 0.0f, 1.0f, GL_POLYGON);
		glColor3f(1.0f, 0.2f, 0.2f);
		glLineWidth(1.0f);
		renderCircle(0.0f, 0.0f, 1.0f, GL_LINE_LOOP);

		for (unsigned int i = 0; i < light->edges2.size(); i++)
		{
			Light::VoronoiEdge2& edge = light->edges2[i];

			glLineWidth(3.0f);
			glColor3f(0.2f, 1.0f, 0.2f);

			glBegin(GL_LINES);
			glVertex2fv(&edge.p[0].x);
			glVertex2fv(&edge.p[1].x);
			glEnd();
		}

		for (Triangulation2::Point_iterator iter = light->tri2.points_begin();
		     iter != light->tri2.points_end();
			 iter++)
		{
			glColor3f(1, 1, 1);
			glPointSize(5.0f);
			glBegin(GL_POINTS);
			glVertex2f(iter->x(), iter->y());
			glEnd();
		}

		glEnable(GL_DEPTH_TEST);
	}

	/* Visualize 3d voronoi. */

	if (light->type == Light::OMNI && visualizeVoronoi)
	{
		glUseProgramObjectARB(0);
		glDisable(GL_DEPTH_TEST);
		glClear(GL_DEPTH_BUFFER_BIT);

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();
		float asp = CONFF(screen_width) / CONFF(screen_height);
		glOrtho(-3 * asp, 1 * asp, -1, 3, -2, 2);
		glTranslatef(0.125 * asp, 0.125, 0);

		glDisable(GL_LIGHTING);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_CULL_FACE);

		glColor3f(0.1f, 0.1f, 0.4f);
		glLineWidth(1.0f);
		renderCircle(0.0f, 0.0f, 1.0f, GL_POLYGON);

		for (unsigned int i = 0; i < light->debugVoronoiVertices3.size(); i++)
		{
			glPointSize(5.0f);
			glColor3f(0.8f, 0.8f, 0.8f);
			glBegin(GL_POINTS);
			//glVertex3fv(&light->debugVoronoiVertices3[i].x);
			glEnd();
		}

		glEnable(GL_BLEND);
		glBlendFunc(GL_ONE, GL_ONE);

		for (unsigned int i = 0; i < light->debugVoronoiEdges3.size(); i++)
		{
			glLineWidth(3.0f);
			renderLineOnSphere(light->debugVoronoiEdges3[i].first, light->debugVoronoiEdges3[i].second);
		}

		glLineWidth(2.f);
		glBegin(GL_LINES);
		for (Triangulation3::Point_iterator iter = light->tri3.points_begin();
		     iter != light->tri3.points_end();
			 iter++)
		{
			float c = (iter->z()+1.0f)/2.0f;
			glColor3f(c, c, c);
			glVertex2f(iter->x(), iter->y());
			glVertex2f(iter->x()*1.1f, iter->y()*1.1f);
		}
		glEnd();
		glPointSize(5.0f);
		glBegin(GL_POINTS);
		for (Triangulation3::Point_iterator iter = light->tri3.points_begin();
		     iter != light->tri3.points_end();
			 iter++)
		{
			float c = (iter->z() + 1.0f) / 2.0f;
			glColor3f(c, c, c);
			glVertex2f(iter->x(), iter->y());
		}
		glEnd();

		glDisable(GL_BLEND);

		glDisable(GL_CULL_FACE);
		glDisable(GL_BLEND);
	}

	/* Swap buffers. */

	glFlush();
	glFinish();
	glutSwapBuffers();

	REPORT_TIME("swap");

	if (CONFI(record_video) && (frame & 1) == 1)
	{
		char filename[64];
		sprintf(filename, "frames/frame%03d.tga", frame/2);

		int w = CONFI(screen_width);
		int h = CONFI(screen_height);

		unsigned char* buf = (unsigned char*) malloc(w * h * 3);
		glReadPixels(0, 0, w, h, GL_RGB, GL_UNSIGNED_BYTE, buf);

		if (CONFI(downsample_frames))
		{
			// 2x2 aa
			unsigned char* buf2 = (unsigned char*) malloc((w/2) * (h/2) * 3);
			for (int i = 0; i < h/2; i++)
			{
				for (int j = 0; j < w/2; j++)
				{
					for (int k = 0; k < 3; k++)
						buf2[(i * (w/2) + j) * 3 + k] =
							(buf[((i*2  ) * w + j*2  ) * 3 + k] +
							 buf[((i*2+1) * w + j*2  ) * 3 + k] +
							 buf[((i*2  ) * w + j*2+1) * 3 + k] +
							 buf[((i*2+1) * w + j*2+1) * 3 + k]) / 4;
				}
			}
			export_targa(filename, w/2, h/2, buf2);
			free(buf2);
		} else
		{
			export_targa(filename, w, h, buf);
		}

		free(buf);
	}

	/* Show FPS. */

	static AverageTiming at;

	double frameEndTime = timer.time();
	at.insert(frameEndTime - frameStartTime);

	printf("FPS %.1f %.1f                      \n", 1.0 / (frameEndTime - frameStartTime), 1.0 / at.getAverage());

	frame++;
}

static void idle()
{
	glutPostRedisplay();
}

int main(int argc, char* argv[])
{
	config.setInt(		"screen_width",				640);
	config.setInt(		"screen_height",			480);
	config.setInt(		"fullscreen",				0);
	config.setInt(		"split_n",					4);
	config.setInt(		"split_m",					4);
	config.setString(	"obj_file",					"cornell.obj");
	config.setFloat(	"obj_scale",				0.035);
	config.setFloat(	"obj_trans_x",				-275);
	config.setFloat(	"obj_trans_y",				-235);
	config.setFloat(	"obj_trans_z",				-275);
	config.setInt(		"obj_swap_yz",				0);
	config.setInt(		"flat_normals",				1);
	config.setFloat(	"normal_threshold",			0.8);
	config.setFloat(	"position_threshold",		1.0);
	config.setInt(		"secondary_lights",			128);
	config.setInt(		"primary_shadowmap_size",	1024);
	config.setInt(		"secondary_shadowmap_size",	256);
	config.setInt(		"shadowmap_hack",			1);
	config.setInt(		"no_depth_cubemaps",		0);
	config.setFloat(	"indirect_boost",			1.5);

	config.setInt(		"render_direct",			1);
	config.setInt(		"render_indirect",			1);
	config.setInt(		"render_toekoet",			0);
	config.setInt(		"render_ambient",			0);
	config.setInt(		"render_voronoi",			0);

	config.setInt(		"timer_individuals",		0);
	config.setInt(		"recalc_lights_min",		4);
	config.setInt(		"recalc_lights_max",		10);
	config.setFloat(	"subdivide_edge",			2.0);
	config.setInt(		"spot",						1);
	config.setInt(		"reuse_secondary_lights",	1);
	config.setInt(		"record_video",				0);
	config.setInt(		"downsample_frames",		0);
	config.setInt(		"ao_samples",				32);

	config.setInt(		"camera_path",				0);
	config.setFloat(	"camera_path_scale",		1.0);

	config.setInt(		"culling",					1);

	config.setInt(		"cupid",					0);
	config.setFloat(	"cupid_x",					0.0);
	config.setFloat(	"cupid_y",					0.0);
	config.setFloat(	"cupid_z",					0.0);
	config.setFloat(	"cupid_scale",				1.0);

	config.setInt(		"elukka",					0);
	config.setFloat(	"elukka_scale",				1.0);

	config.setFloat(	"light_x",					0.0);
	config.setFloat(	"light_y",					6.0);
	config.setFloat(	"light_z",					0.0);

	config.setFloat(	"light_x_scale",			4.0);
	config.setFloat(	"light_y_scale",			1.0);
	config.setFloat(	"light_z_scale",			4.0);

	config.setFloat(	"light_speed_xz",			1.0);
	config.setFloat(	"light_speed_y",			1.0);
	config.setFloat(	"light_time_offset",		0.0);
	config.setFloat(	"light_rotation_speed",		1.0);

	config.setFloat(	"light_intensity",			50.0);

	config.setFloat(	"light_size",				1.0);

	if (argc >= 2)
	{
		if (config.load(argv[1]) == false)
			return EXIT_FAILURE;
	}
	// <HACKS>
	else
	{
		//config.load("data/cornell.txt");
		//config.load("data/sibenik-illu.txt");
		config.load("data/maze.txt");

		config.setInt(		"screen_width",				800);
		config.setInt(		"screen_height",			600);
		config.setInt(		"secondary_lights",			64);
		config.setInt(		"camera_path",				0);
		//config.setInt(		"timer_individuals",		0);
		//config.setInt(		"spot",						1);
		config.setInt(		"render_ambient",			0);
	}
	// </HACKS>

	for (int i = 2; i < argc; i++)
	{
		if (config.parseLine(argv[i]) == false)
		{
			fprintf(stderr, "parse error: %s\n", argv[i]);
			return EXIT_FAILURE;
		}
	}

#ifdef WIN32
	SetProcessAffinityMask(GetCurrentProcess(), 1);
#endif

	glutInit(&argc, argv);
	glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);

	glutInitWindowSize(CONFI(screen_width), CONFI(screen_height));

	glutCreateWindow("Incremental Instant Radiosity");

	glutReshapeFunc(reshape);
	glutKeyboardFunc(keyboard);
	glutSpecialFunc(special);
	glutSpecialUpFunc(specialUp);
	glutMouseFunc(mouse);
	glutMotionFunc(motion);
	glutDisplayFunc(display);
	glutIdleFunc(idle);

	if (CONFI(fullscreen))
		glutFullScreen();

	init();

	// Move console window next to GLUT window
#ifdef WIN32
	COORD cpos = { 0, 0 };
	SetConsoleCursorPosition(GetStdHandle(STD_OUTPUT_HANDLE), cpos);
	for (int i=0; i < 15; i++)
		printf("%75s\n", "");

	SetWindowPos(GetConsoleWindow(), 0, CONFI(screen_width) + 8, 0, 0, 0, SWP_NOSIZE | SWP_NOZORDER);
#endif

	glutMainLoop();

	return 0;
}
