#ifndef TIMER_H
#define TIMER_H

#ifdef WIN32
#	ifndef WIN32_LEAN_AND_MEAN
#		define WIN32_LEAN_AND_MEAN
#	endif
#	include <windows.h>
#else
#  include <sys/time.h>
#  include <stdlib.h>
#endif

/// The sys namespace contains various system related classes.
namespace sys
{

/// A timer class.
class Timer
{
public:
	/// Initialize the timer.
	/// Queries the timer frequency and resets the time.
	void init(void)
	{
#ifdef WIN32
		QueryPerformanceFrequency(&m_performanceFrequency);
		m_performanceReciprocal = 1.0 / (double)m_performanceFrequency.QuadPart;
#endif
	
		reset();
	}

	/// Reset the timer.
	/// Makes the moment this function is called the new "0 time".
	void reset(void)
	{
#ifdef WIN32
		QueryPerformanceCounter(&m_startTime);
#else
		gettimeofday(&start, NULL);
#endif
	}

	/// Get the amount of time passed since timer start.
	/// Returns the amount of seconds passed since the timer was reset.
	/// @return time in seconds
	double time(void)
	{
#ifdef WIN32
		LARGE_INTEGER endTime;

		QueryPerformanceCounter(&endTime);
		return ((double)(endTime.QuadPart - m_startTime.QuadPart) * m_performanceReciprocal);
#else
		struct timeval tv;
		gettimeofday(&tv, NULL);
		double d = (tv.tv_sec - start.tv_sec) * 1000000.0 + (double)tv.tv_usec - (double)start.tv_usec;
		return float(d / 1000000.0);
#endif
	}

	/// Get the timer resolution.
	/// Returns the timer frequency.
	/// @return number of ticks per second
	int timerResolution(void)
	{
#ifdef WIN32
		return (int)m_performanceFrequency.QuadPart;
#else
		// Actually probably 100
		return 1000;
#endif
	}

private:
#ifdef WIN32
	LARGE_INTEGER m_startTime;
	LARGE_INTEGER m_performanceFrequency;
	double m_performanceReciprocal;
#else
	struct timeval start;
#endif
};

}

#endif
