#include "paraboloidmap.h"
#include "glutil.h"
#include "configuration.h"
#include <assert.h>

static const char* paraboloidMapVertex =
"\
varying float alpha;\
\
void main()\
{\
	vec4 p = ftransform();\
	alpha = p.z;\
	float l = inversesqrt(dot(p.xyz, p.xyz));\
	p *= l;\
	p.z -= 1.0;\
	float pzinv = 1.0 / p.z;\
	p.x *= pzinv;\
	p.y *= pzinv;\
	p.z = (1.0/l - 0.1) * (1.0 / (30.0 - 0.1));\
	p.w = 1.0;\
\
	gl_Position = p;\
}\
";

static const char* paraboloidMapFrag =
"\
varying float alpha;\
\
void main()\
{\
	if (alpha >= 0.0)\
		discard;\
}\
";

static void checkFramebufferStatus(void)
{
	GLenum status;
	status = glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	if (status != GL_FRAMEBUFFER_COMPLETE_EXT)
		printf("FBO STATUS %x\n", status);
	assert(status == GL_FRAMEBUFFER_COMPLETE_EXT);
}

static int program = 0;

static GLuint lastFB;
static GLuint lastTexture;
static int counter;

ParaboloidMap::ParaboloidMap(int size) : size(size)
{
	glGenTextures(1, &texture);

	checkFramebufferStatus();

	if (!CONFI(shadowmap_hack) || counter == 0)
	{
		int s = CONFI(shadowmap_hack) ? size*2 : size;
		glBindTexture(GL_TEXTURE_2D, texture);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT24_ARB, s, s, 0, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, NULL);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

		glGenFramebuffersEXT(1, &fb);
		glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fb);
		glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_TEXTURE_2D, texture, 0);
		glDrawBuffer(GL_NONE);
		glReadBuffer(GL_NONE);

		lastTexture = texture;
		lastFB = fb;
		offsetX = 0;
		offsetY = 0;
	}
	else
	{
		texture = lastTexture;
		fb = lastFB;
		offsetX = (counter == 1 || counter == 3) ? size : 0;
		offsetY = (counter == 2 || counter == 3) ? size : 0;		
	}

	checkFramebufferStatus();

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	counter = (counter + 1) % 4;

	if (program == 0)
		program = MAKE_GLSL_PROGRAM(paraboloidMapFrag, paraboloidMapVertex);

	assert(glGetError() == GL_NO_ERROR);
}

ParaboloidMap::~ParaboloidMap()
{
	/* TODO: delete OpenGL stuff */
}

void ParaboloidMap::getMatrix(float* m)
{
	memcpy(m, transform, sizeof(transform));
}

void ParaboloidMap::setPosition(const m::Vector3& v)
{
	pos = v;
	updateMatrix();
}

void ParaboloidMap::setDirection(const m::Vector3& v)
{
	dir = v;
	updateMatrix();
}

void ParaboloidMap::updateMatrix()
{
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glRotatef(rand() / (float)RAND_MAX * 360.0f, 0, 0, 1);
	lookat(pos, dir);
	glGetFloatv(GL_MODELVIEW_MATRIX, transform);
	glPopMatrix();
}

void ParaboloidMap::render(void (*render)(void*), void* user)
{
	prepareRendering();
	render(user);
	doneRendering();
}

void ParaboloidMap::prepareRendering()
{
	assert(glGetError() == GL_NO_ERROR);

	glEnable(GL_DEPTH_TEST);

	glGetIntegerv(GL_VIEWPORT, viewport);
	glViewport(offsetX, offsetY, size, size);

	glBindTexture(GL_TEXTURE_2D, texture);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE_ARB, GL_COMPARE_R_TO_TEXTURE_ARB);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_COMPARE_FUNC_ARB, GL_LEQUAL);

	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, fb);
	assert(glGetError() == GL_NO_ERROR);

	//checkFramebufferStatus();

	if (CONFI(shadowmap_hack))
		glEnable(GL_SCISSOR_TEST);
	glScissor(offsetX, offsetY, size, size);
	glClear(GL_DEPTH_BUFFER_BIT);
	glDisable(GL_SCISSOR_TEST);
	assert(glGetError() == GL_NO_ERROR);

	/* TODO: test perforamnce by setting color masks to false */
             
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	//glLoadIdentity();
	/*gluLookAt(pos.x, pos.y, pos.z,
		pos.x + dir.x, pos.y + dir.y, pos.z + dir.z,
		0, 1, 0);*/
	glLoadMatrixf(transform);

	glUseProgramObjectARB(program);
	assert(glGetError() == GL_NO_ERROR);
}

void ParaboloidMap::doneRendering()
{
	glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
	glUseProgramObjectARB(0);

	glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);

	glMatrixMode(GL_PROJECTION);
	glPopMatrix();

	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	assert(glGetError() == GL_NO_ERROR);
}
