#include "configuration.h"

Configuration config;

Configuration::Configuration()
{
}

Configuration::~Configuration()
{
}

void Configuration::setInt(const char* n, int v)
{
	Value val;
	val.type = INTEGER;
	val.i = v;
	values[n] = val;
}

void Configuration::setFloat(const char* n, double v)
{
	Value val;
	val.type = FLOAT;
	val.f = v;
	values[n] = val;
}

void Configuration::setString(const char* n, const char* v)
{
	Value val;
	val.type = STRING;
	val.s = v;
	values[n] = val;
}

int Configuration::getInt(const char* n) const
{
	std::map<std::string, Value>::const_iterator iter;

	iter = values.find(n);
	if (iter == values.end())
	{
		fprintf(stderr, "Configuration '%s' (int) not defined.\n", n);
		exit(EXIT_FAILURE);
	}

	if (iter->second.type != INTEGER)
	{
		fprintf(stderr, "Configuration '%s' is not int.\n", n);
		exit(EXIT_FAILURE);
	}

	return iter->second.i;
}

double Configuration::getFloat(const char* n) const
{
	std::map<std::string, Value>::const_iterator iter;

	iter = values.find(n);
	if (iter == values.end())
	{
		fprintf(stderr, "Configuration '%s' (float) not defined.\n", n);
		exit(EXIT_FAILURE);
	}

	if (iter->second.type != FLOAT && iter->second.type != INTEGER)
	{
		fprintf(stderr, "Configuration '%s' is not float nor int.\n", n);
		exit(EXIT_FAILURE);
	}

	if (iter->second.type == INTEGER)
		return (double)iter->second.i;

	return iter->second.f;
}

const char* Configuration::getString(const char* n) const
{
	std::map<std::string, Value>::const_iterator iter;

	iter = values.find(n);
	if (iter == values.end())
	{
		fprintf(stderr, "Configuration '%s' (string) not defined.\n", n);
		exit(EXIT_FAILURE);
	}

	if (iter->second.type != STRING)
	{
		fprintf(stderr, "Configuration '%s' is not string.\n", n);
		exit(EXIT_FAILURE);
	}

	return iter->second.s.c_str();
}

static char* strip_line(char* p)
{
	if (strchr(p, '#'))
		*strchr(p, '#') = '\0';
	if (strchr(p, '\r'))
		*strchr(p, '\r') = '\0';
	if (strchr(p, '\n'))
		*strchr(p, '\n') = '\0';

	while (isspace(*p))
		p++;

	return p;
}

bool Configuration::parseLine(const char* l)
{
	char buf[2048];
	strncpy(buf, l, sizeof(buf));

	char* p = strip_line(buf);

	if (*p == '\0')
		return true;

	if (!isalpha(*p) && *p != '_')
		return false;

	const char* ns = p;
	while (isalnum(*p) || *p == '_')
		p++;

	if (*p == '\0')
		return false;

	char *ne = p;

	while (isspace(*p))
		p++;

	if (*p != '=')
		return false;
	p++;

	while (isspace(*p))
		p++;

	Value val;
	if (isdigit(*p) || *p == '-' || *p == '+' || *p == '.')
	{
		const char* vs = p;

		if (*p == '+' || *p == '-')
			p++;

		while (isdigit(*p))
			p++;

		if (*p == '.')
		{
			while (isdigit(*p))
				p++;
			val.type = FLOAT;
			val.f = strtod(vs, NULL);
		}
		else
		{

			val.type = INTEGER;
			val.i = strtol(vs, NULL, 0);
		}
	}
	else
	{
		val.type = STRING;
		val.s = p;
	}

	*ne = '\0';
	if (values.find(ns) == values.end())
		return false;
	values[ns] = val;

	return true;
}

bool Configuration::load(const char* fn)
{
	FILE* fp = fopen(fn, "r");
	if (!fp)
	{
		fprintf(stderr, "Can't open '%s'\n", fn);
		return false;
	}

	int lineno = 0;

	for (;;)
	{
		char buf[2048];
		if (fgets(buf, sizeof(buf), fp) == NULL)
			break;

		lineno++;
		if (!parseLine(buf))
		{
			fclose(fp);
			fprintf(stderr, "%s:%d: bad configuration\n", fn, lineno);
			return false;
		}
	}

	fclose(fp);

	return true;
}
