function [p,Rs,ts,err]=calibrate_demo(ms,xs,pinit,filename)
% [p,Rs,ts]=calibrate(ms,xs)
%
% CALIBRATE computes the optimal camera parameters. Without any
% input the calibration data is read from the files supplied by
% CALIBIMAGES 
% 
% optional input:
%   ms = the measured control point positions in each image, 
%        cell array of matrices containing two columns
%   xs = the control point positions in the calibration plane,
%        cell array of matrices containing two columns
%
% output:
%   p = the internal camera parameters, whose number depends on the
%       camera model (6, 9 or 23)
%   Rs = cell array, Rs{j} is the rotation matrix for view j
%   ts = cell array, ts{j} is the translation vector for view j
%
% See also CALIBCONFIG, CALIBIMAGES
%

% Copyright (C) 2004-2005 Juho Kannala
%
% This software is distributed under the GNU General Public
% Licence (version 2 or later); please refer to the file
% Licence.txt, included with the software, for details.

disp('Give the name of your configuration (same as in file calibconfig.m): ');
file='configname.mat';
load(file,'configname');
%configname=input('Give the name of your configuration (the name given in file calibconfig.m): ','s');

sys=calibconfig(configname);

if nargin<2 | isempty(xs) | isempty(ms)
  [ms,xs]=readdata(sys);
end
if nargin<3
  pinit=[];
end

if isempty(pinit)
  [pinit,thetamax]=initialiseinternalp(sys);
else
  thetamax=sys.viewfield/2*pi/180;
end
pinit=pinit

p0=pinit;

[Hs,err]=compHs(p0,ms,xs,thetamax);
err=err

K=eye(3);

[Rs0,ts0]=initialiseexternalp(Hs,K,0);
%save p0 p0
%save Rs0 Rs0
%save ts0 ts0

p0=p0
[err,meanerr]=projerrs(ms,xs,p0,Rs0,ts0,sys.blobradius);

%keyboard;

disp('Mean distance between measured and modelled centroids before nonlinear minimization')
fprintf(1,' %.4f pixels\n',meanerr);


% first just the basic model  
[p,Rs,ts]=minimiseprojerrs(ms,xs,p0,Rs0,ts0,'basic',0);
p=p
% if the initial guess is bad the general radially symmetric
% projection may give negative values for r and we need the
% following correction
if p(1)<0 | (p(1)==0 & p(2)<0)
  disp('In calibcomp: Radius must be positive');
  p(1)=-p(1); p(2)=-p(2);
  S=[-1 0 0; 0 -1 0; 0 0 1];
  for i=1:length(Rs)
    R=Rs{i};
    Rs{i}=S*Rs{i};
    ts{i}=S*ts{i};
  end
end

% if a more complex model is assumed or the control points are circular
if ~strcmp(sys.model,'basic') | sys.blobradius~=0
  [err,meanerr]=projerrs(ms,xs,p,Rs,ts,sys.blobradius);
  disp('Mean distance between measured and modelled centroids before fitting the full model')
  fprintf(1,' %.4f pixels\n\n',meanerr);
  disp('If the control points are circular the computation may take time. Computing ...');
  [p,Rs,ts]=minimiseprojerrs(ms,xs,p,Rs,ts,sys.model,sys.blobradius);
end

[err,meanerr,mederr,su,sv,rmserr]=projerrs(ms,xs,p,Rs,ts,sys.blobradius);

if nargin<4 | isempty(filename)
switch lower(sys.model)
 case 'basic'
  if sys.blobradius==0
    filename='calibrate_basic_noncirc.mat';
  else
    filename='calibrate_basic.mat';
  end  
 case 'radial'
  if sys.blobradius==0
    filename='calibrate_radial_noncirc.mat';
  else
    filename='calibrate_radial.mat';
  end
 case 'extended'
  if sys.blobradius==0
    filename='calibrate_extended_noncirc.mat';
  else  
    filename='calibrate_extended.mat';
  end
 otherwise
  disp('Unknown model');
end
end

  
disp('Mean distance between measured and modelled centroids:')
fprintf(1,' %.4f pixels\n',meanerr);

disp('RMS distance between measured and modelled centroids:')
fprintf(1,' %.4f pixels\n',rmserr);

disp('Standard deviation of the residuals:');
fprintf(1,'sigma_u: %.4f pixels\n',su);
fprintf(1,'sigma_v: %.4f pixels\n',sv);

[maxd,meand]=testbackproject(p,thetamax);

save(filename,'ms','xs','p0','Rs0','ts0','p','Rs','ts','err','meanerr','mederr','su','sv','maxd','meand');

disp('The backward model error:');
fprintf(1,'maximum reprojection error: %e pixels\n',maxd);
fprintf(1,'mean reprojection error: %e pixels\n',meand);
