#!/usr/bin/env python

# A simple script for dynamically creating fluxbox menu entries for
# volume management
#
# Copyright (c) 2008 Lauri Peltonen
# Copyright (c) 2008 Emil Karlson

version = "0.10"

import dbus
import time
import gobject
import os
import re
import sys
import getopt

from configobj import ConfigObj
from os.path import join, expanduser

if getattr(dbus, "version", (0,0,0)) >= (0,41,0):
    import dbus.glib


# print usage
def usage():
    print "fbstorage version", version
    print
    print "usage:"
    print sys.argv[0]
    print sys.argv[0] + " -h,--help"
    print sys.argv[0] + " -v,--verbose"
    print sys.argv[0] + " -q,--quiet"
    print sys.argv[0] + " -c configfile,--conf=configfile"
    print sys.argv[0] + " -n,--no-autorun"
    print sys.argv[0] + " -V,--version"
    return


# set a few defaults
verbose = 0
configfile = ""
autorun = True

# parse options
try: 
    opts, args = getopt.getopt(sys.argv[1:], "hc:vVqna", ["help", "conf=", "verbose", "quiet", "no-autorun", "all-removable", "version"])
except getopt.GetoptError:
    usage()
    sys.exit(2)

for opt, arg in opts:
    if opt in ("-h", "--help"):
        usage()
        sys.exit()
    elif opt in ("-v", "--verbose"):
        verbose = verbose + 1
    elif opt in ("-q", "--quiet"):
        verbose = -1
    elif opt in ("-n", "--no-autorun"):
        autorun = False
    elif opt in ("-c", "--conf"):
        configfile = expanduser(arg)
        print "Forced configfile to ", configfile
    elif opt in ("-V", "--version"):
        print "fbstorage version", version
        sys.exit()


# set default settings
if not configfile:
    configfile = expanduser("~/.fluxbox/fbstorage.conf")
    if not os.path.exists(configfile):
        if 0 <= verbose: print "userconfig not found, using global configfile"
        configfile = "/etc/fbstorage.conf"

update_delay = 2000
menu_file = expanduser("~/.fluxbox/storage")
touch_file = expanduser("~/.fluxbox/menu")
defconfig = True

# parse config file
if os.path.exists(configfile):
    conf = ConfigObj(configfile)

    if conf.has_key("update_delay"):
        update_delay = int(conf["update_delay"])
    if conf.has_key("menu_file"):
        menu_file = expanduser(conf["menu_file"])
    if conf.has_key("touch_file"):
        touch_file = expanduser(conf["touch_file"])

    config = []
    for entry in conf:
        try:
            conf[entry]["title"]
        except TypeError:
            ""
        else:
            config.append(conf[entry])
            defconfig = False

# load defaults if no sections are found
if defconfig:
    if 0 <=  verbose: print "Loading default config"
    config = [
         {  "title":"%d: %l",
            "subtitle":"%m",
            "unmounted":{ "mount":{ "title":"Mount", "command":"gnome-mount -d %d" } },
            "mounted":{ "unmount":{"title":"Unmount", "command":"gnome-umount -d  %d" } }
         }
         ]

if 1 <= verbose: print config
# config code ends


# actual working code begins
class HalManager:
    volumes = {}
    is_updated = False

    def __init__(self):
        if 1 <= verbose: print "Preparing halmanager..."

        self.current_hal = True

        try:
            self.bus = dbus.SystemBus()
            self.hal_manager_obj = self.bus.get_object("org.freedesktop.Hal", "/org/freedesktop/Hal/Manager")
            self.hal_manager = dbus.Interface(self.hal_manager_obj, "org.freedesktop.Hal.Manager")
        except:
            print "Hal not found or version not supported"
            print "Please make sure that hald is running, and check the version"
            print "Currently only 0.5.5.1 is tested to work."
            exit

        self.__get_items__()

        self.hal_manager.connect_to_signal("DeviceAdded", self.__device_added_callback__)
        self.hal_manager.connect_to_signal("DeviceRemoved", self.__device_removed_callback__)
        self.hal_manager.connect_to_signal("NewCapability", self.__device_capability_callback__)

        self.is_updated = True
        if 1 <= verbose: print "Halmanager ready"
        return

    def __get_items__(self): 

       # Get all items that work as volume (Hard drive partitions, cdroms, usb-storages etc)
        for item in self.volumes:
            if 2 <= verbose: print "Removing receiver..."
            self.bus.remove_signal_receiver(None,
                                            "PropertyModified",
                                            "org.freedesktop.Hal.Device",
                                            "org.freedesktop.Hal",
                                            item)
            if 2 <= verbose: print "Done"
        self.volumes = { }

        udi_list = self.hal_manager.FindDeviceByCapability("volume")
        for udi in udi_list:
            data = self.__udi_to_infotuple__(udi)
            if data:
                self.volumes[udi] = data
                if 1 <= verbose: print "Adding receiver..."
                self.bus.add_signal_receiver(self.__property_modified_callback__,
                                     "PropertyModified",
                                     "org.freedesktop.Hal.Device",
                                     "org.freedesktop.Hal",
                                     udi)
                if 1 <= verbose: print "Done"

                # Do not want to autorun already plugged in devices
                self.volumes[udi]["autorun"] = True

        self.is_updated = True


    def __device_added_callback__(self, udi):
        if 1 <= verbose: print "Device with udi %s was added" % (udi)

        if self.volumes.has_key(udi):
            if 2 <= verbose: print "  Device already exists in database, overwriting"
            del self.volumes[udi]
            self.is_updated = True
        else:
            self.bus.add_signal_receiver(self.__property_modified_callback__,
                                     "PropertyModified",
                                     "org.freedesktop.Hal.Device",
                                     "org.freedesktop.Hal",
                                     udi)

    
        data = self.__udi_to_infotuple__(udi)
        if data:
            if 1 <= verbose: print "  Device is a valid volume, adding to database"
            self.volumes[udi] = data
            self.is_updated = True

    def __device_removed_callback__(self, udi):
        if 1 <= verbose: print "Device with udi %s was removed" % (udi)
        if self.volumes.has_key(udi):
            if 1 <= verbose: print "  Device is in database, removing..."
            del self.volumes[udi]
            self.is_updated = True
            self.bus.remove_signal_receiver(None,
                                            "PropertyModified",
                                            "org.freedesktop.Hal.Device",
                                            "org.freedesktop.Hal",
                                            udi)


    def __device_capability_callback__(self, udi, capability):
        if 1 <= verbose: print "Device with udi %s added capability %s" % (udi, capability)
        if self.volumes.has_key(udi):
            if 2 <= verbose: print "  Device found in database, overwriting with new capabilities"
            del self.volumes[udi]
            data = self.__udi_to_infotuple__(udi)
            self.is_updated = True
            if data:
                self.volumes[udi] = data

    def __property_modified_callback__(self, changes, changelist):
        # I didn't get the "udi" -thing to work (to assign an udi on calltime),
        # So we must probe all devices for changes if one gets mounted or so...
        # It shouldn't take too long to notice or anything, but it is an
        # ugly solution :P
        if 1 <= verbose: print "Device property was modified, probing..."
        if 2 <= verbose: print "  Changelist (%s : %s)" % (changes, changelist)
        self.__get_items__()

    def __udi_to_infotuple__(self, udi):
        device_obj = self.bus.get_object("org.freedesktop.Hal", udi)
        volume = dbus.Interface(device_obj, "org.freedesktop.Hal.Device")

        try:
            parent = volume.GetProperty("block.storage_device")
            parent_obj = self.bus.get_object("org.freedesktop.Hal", parent)
            storage = dbus.Interface(parent_obj, "org.freedesktop.Hal.Device")
        except:
            # This was not a storage device!
            return None

        # Check if the item is removable. Those are the only we want to show.
        if storage.GetProperty("storage.removable"):

            # We are only interested in self.volumes as only they are mountable
            if volume.GetProperty("block.is_volume"):

                # Some sane default values for device information
                type_major = ""
                type_minor = ""
                type_extended = ""
                label = "Unknown"
                product = "Unknown"
                mounted = False
                mountpoint = "None"
                udi = "None"
                dev = "None"
                bus = "None"
                drivetype = "None"
                automount = False

                # Read basic information, these are mandatory (all devices have these)
                # If these should throw, then something is horribly wrong!
                try:
                    mounted = volume.GetProperty("volume.is_mounted")
                    mountpoint = volume.GetProperty("volume.mount_point")
                    label = volume.GetProperty("volume.label")
                    udi = volume.GetProperty("info.udi")
                    dev = volume.GetProperty("block.device")

                    # Parent is the "storage" node, so these are from it
                    bus = storage.GetProperty("storage.bus")
                    drivetype = storage.GetProperty("storage.drive_type")
                    automount = storage.GetProperty("storage.automount_enabled_hint")
                except:
                    print "Something went horribly wrong when retrieving device information."
                    print "I will try to continue, but this device might not show up properly."

                # This is not mandatory property
                try: product = volume.GetProperty("info.product")
                except: product = "None"

                # CDs, DVDs and other discs
                if volume.GetProperty("volume.is_disc"):
                    audio = volume.GetProperty("volume.disc.has_audio")
                    data = volume.GetProperty("volume.disc.has_data")
                    type = volume.GetProperty("volume.disc.type")

                    # Possible major types are currently "dvd", "cd" and just an unknown "disc"
                    # Extended type tells if cd is "ROM", "R", or "RW" and dvd is "ROM", "RAM", "-R", "-RW", "+R" or "+RW"
                    if type == "cd_rom":
                        type_major = "cd"
                        type_extended = "rom"
                    elif type == "cd_r":
                        type_major = "cd"
                        type_extended = "r"
                    elif type == "cd_rw":
                        type_major = "cd"
                        type_extended = "rw"

                    elif type == "dvd_rom":
                        type_major = "dvd"
                        type_extended = "rom"
                    elif type == "dvd_ram":
                        type_major = "dvd"
                        type_extended = "ram"
                    elif type == "dvd_r":
                        type_major = "dvd"
                        type_extended = "-r"
                    elif type == "dvd_rw":
                        type_major = "dvd"
                        type_extended = "-rw"
                    elif type == "dvd_plus_r":
                        type_major = "dvd"
                        type_extended = "+r"
                    elif type == "dvd_plus_rw":
                        type_major = "dvd"
                        type_extended = "+rw"

                    elif type == "bd_rom":
                        type_major = "bd"
                        type_extended = "rom"
                    elif type == "bd_r":
                        type_major = "bd"
                        type_extended = "r"
                    elif type == "bd_re":
                        type_major = "bd"
                        type_extended = "rw"

                    elif type == "hddvd_rom":
                        type_major = "hddvd"
                        type_extended = "rom"
                    elif type == "hddvd_r":
                        type_major = "hddvd"
                        type_extended = "r"
                    elif type == "hddvd_rw":
                        type_major = "hddvd"
                        type_extended = "rw"

                    else: 
                        type_major = "disc"


                    # Minor type means disc is datadisc, audiodisc or mixed disc
                    # It can also be a videocd, supervideocd, videodvd or blank
                    # PLEASE TELL ME IF VCD OR SVCD OR VIDEODVD IS NOT DETECTED CORRECTLY!
                    
                    if self.current_hal:
                        try:
                            if volume.GetProperty("volume.disc.is_videodvd"): type_minor = "videodvd"
                            elif volume.GetProperty("volume.disc.is_vcd"): type_minor = "vcd"
                            elif volume.GetProperty("volume.disc.is_svcd"): type_minor = "svcd"
                        except:
                            print "Your version of HAL does not support svcd, vcd or videodvd detection."
                            print "Please try to upgrade to 0.5.5.3 or newer."
                            print "Program will continue, but those ^ are not detected"
                            print "NOTE: This is also on issue in HAL with audio cds, and blank cds and maybe"
                            print "on some others, so everything may be correctly detected."
                            print "This message will not repeat."
                            self.current_hal = False

                    if type_minor == "":
                        if volume.GetProperty("volume.disc.is_blank"): type_minor = "blank"
                        elif audio and data: type_minor = "mixed"
                        elif audio and not data: type_minor = "audio"
                        elif data and not audio: type_minor = "data"

                # Other device types?

                dev_data = {
                         "major":type_major, "minor":type_minor, "extended":type_extended,
                         "label":label, 
                         "product":product,
                         "device":dev,
                         "udi":udi,
                         "mounted":mounted,
                         "mountpoint":mountpoint,
                         "drivetype": drivetype,
                         "bus":bus,
                         "automount":automount
                       }
                if 3 <= verbose: print dev_data

                return dev_data

        return None

    def updated(self):
        retval = self.is_updated
        self.is_updated = False
        return retval

    def get_volumes(self):
        return self.volumes


def item_matches(dict, key, item):
    if dict.has_key(key) and not dict[key] == "":
        for type in dict[key].split(","):
            if item == type: return True
        return False
    return True


def print_item(items, device):
    output = ""
    for title in items:
        item = items[title]

        if item.has_key("title") and item.has_key("command"):
            output = output + "    [exec] (" + escape_chars(item["title"], device) + ") {" + escape_chars(item["command"], device) + "}"
        if item.has_key("icon"):
            output = output + " <" + escape_chars(item["icon"], device) + ">"

        output = output + "\r\n"
    return output


def print_menu(halmanager):
    volumes = halmanager.get_volumes()

    try:
        cFile = open(menu_file, "w")
    except:
        print "Error opening file for writing: ", menu_file
        return

    for dev in volumes:
        device = volumes[dev]
        for match in config:
            if not match.has_key("title"): continue

            if not item_matches(match, "type", device["major"]): continue
            if not item_matches(match, "minor", device["minor"]): continue
            if not item_matches(match, "extra", device["extended"]): continue
            if not item_matches(match, "product", device["product"]): continue
            if not item_matches(match, "label", device["label"]): continue
            if not item_matches(match, "bus", device["bus"]): continue
            if not item_matches(match, "drive", device["drivetype"]): continue


            # Item passed all cases => it is a correct item. Let"s print the commands out
            cFile.write("[submenu] (" + escape_chars(match["title"], device) + ")")
            if match.has_key("subtitle"): cFile.write(" {"+ escape_chars(match["subtitle"], device) + "}")
            if match.has_key("icon"): cFile.write(" <" + escape_chars(match["icon"], device) + ">")
            cFile.write("\r\n")

            if match.has_key("mounted") and device["mounted"]:  cFile.write(print_item(match["mounted"], device))
            elif match.has_key("unmounted") and not device["mounted"]: cFile.write(print_item(match["unmounted"], device))

            if match.has_key("always"):  cFile.write(print_item(match["always"], device))

            cFile.write("[end]\r\n")


            # Autorun if enabled
            if autorun and match.has_key("autorun"):

                # Test if this has not yet been executed
                if not device.has_key("autorun"):
                    device["autorun"] = True
                    command = escape_chars(match["autorun"], device)
                    if 2 <= verbose: print "Autorun executing command '%s'" % command
                    os.system(command)

            # Do not try to match any other items anymore for this :)
            break

    cFile.close()

    # Touch the main menu for updating the menu
    if touch_file:
        os.system("touch " + touch_file)

    return

def escape_chars(line, device):
    line = line.replace("%%", "%")
    line = line.replace("%s", device["mountpoint"])
    line = line.replace("%u", device["udi"])
    line = line.replace("%d", device["device"])
    line = line.replace("%l", device["label"])
    line = line.replace("%p", device["product"])

    if device["mounted"] == True:
        line = line.replace("%m", "Mounted")
    else:
        line = line.replace("%m", "Unmounted")

    return line

def update_menu(halmanager):
    if halmanager.updated():
        print_menu(halmanager)
    return True




# Main program starts here
# Initialise and launch halmanager
halmanager = HalManager()
update_menu(halmanager)


# Add a timer to check for mounting and so
timer = gobject.timeout_add (update_delay, update_menu, halmanager)
mainloop = gobject.MainLoop()
mainloop.run()
gobject.source_remove(timer)
