import * as vscode from 'vscode';

/* respect https://github.com/Gimly/vscode-fortran/blob/229cddce53a2ea0b93032619efeef26376cd0d2c/src/documentSymbolProvider.ts
           https://github.com/Microsoft/vscode/blob/34ba2e2fbfd196e2d6db5a4db0e42d03a97c655e/extensions/markdown-language-features/src/features/documentLinkProvider.ts
 */
export class ShellScriptFunctionSymbolProvider implements vscode.DocumentSymbolProvider {

    public provideDocumentSymbols(
        document: vscode.TextDocument,
        token: vscode.CancellationToken): vscode.SymbolInformation[] {

        const tokenToKind = this.tokenToKind;
        const text = document.getText();
        const matchedList = this.matchAll(this.pattern, text);

        return matchedList.map((matched) => {
            const type = matched[5];
            const name = matched[3];
            const kind = tokenToKind[type];
            const position = document.positionAt(matched.index || 0);
            return new vscode.SymbolInformation(name, kind, '', new vscode.Location(document.uri, position));
        });
    }

    private get tokenToKind(): { [name: string]: vscode.SymbolKind; } {
        return {
            '{': vscode.SymbolKind.Module,
            '(': vscode.SymbolKind.Interface
        };
    }

    private get pattern() {
        return /^([ \t]*)(function[ \t]+)?([_A-Za-z][_A-Za-z0-9]+)[ \t]*(\(\))?[ \t*]*([{(])[^(]?/gm;
    }

    private matchAll(pattern: RegExp, text: string): Array<RegExpMatchArray> {
        const out: RegExpMatchArray[] = [];
        pattern.lastIndex = 0;
        let match: RegExpMatchArray | null;
        while ((match = pattern.exec(text))) {
            if (match[2] || match[4]) {
                if (match[1]) match[3] += ' ⇥ ' + match[1].length + '';
                out.push(match);
            }
        }
        return out;
    }
}
