function fbmmlpwrite(logfile,net,noiseSigmas,noiseHyper,use_ard,reverse_byte_order)
%FBMMLPWRITE Write network parameter guess into FBM logfile
%
%   FBMMLPWRITE(LOG,NET,NOISESIGMAS,NOISEHYPER,USE_ARD,REVERSE_BYTE_ORDER)
%   estimates the hyperparameters and writes given (nnet toolbox
%   3.0 -form) MLP-network and hyperparameters into the log file.
%   Written data replaces the last network weights and sigmas
%   previoudly written into the log file. If USE_ARD is non-zero
%   separate estimates are calculated for each of the hyperpameters
%   and input sigmas (default is to use ARD). Non-zero REVERSE_BYTE_ORDER
%   argument specifies that Intel (BE) byte order should be used.
%
%   Arguments NOISESIGMAS and NOISEHYPER are the estimates for
%   standard deviations and hyperparameters of noise in outputs.
%   Column vector NOISESIGMAS should contain one column for each
%   output. NOISEHYPER should be scalar.
%
%   For classification models NOISESIGMAS and NOISEHYPER should be NaN.
%
%   Quite nice guesses for these sigmas can be determined
%   as follows:
%
%     If "guesses" contains output of a network for each
%     of the test cases (as rows) and "answers" are the right
%     answers, the you can estimate them as follows:
%
%     >> noiseSigmas = std(guesses-answers);
%     >> noiseHyper  = std(noiseSigmas);
%
%     And if no ard is used, the following will do fine:
%     >> noiseHyper = mean(std(guesses-answers));
%     >> noiseSigmas = repmat(noiseHyper,1,size(guesses,2));
%
%   See also
%     FBMMLPREAD

% Copyright (C) 1999 Simo Srkk
% Copyright (C) 2000-2003 Aki Vehtari
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

if nargin < 3
  noiseSigmas = [];
end
if nargin < 4
  noiseHyper = [];
end
if nargin < 5
  use_ard = [];
end
if isempty(use_ard)
  use_ard = 1;
end
if nargin < 6
  reverse_byte_order = 0;
end
if (reverse_byte_order~=0)
  machine_fmt = 'ieee-be.l64';
  double_size = 8;
else
  machine_fmt = 'ieee-le.l64';
  double_size = 8;
end

% Read the log file
[fbm,arch] = fbmmlpread(logfile,reverse_byte_order);

% Get weights and hyperparameters
if strcmp(class(net),'network') | isfield(net,'IW')
  % Matlab neural network toolbox network
  
  IW = net.IW{1};
  IW = reshape(IW,1,prod(size(IW)));
  IB = net.b{1};
  IB = reshape(IB,1,prod(size(IB)));
  LW = cell(1,net.numLayers-1);
  LB = cell(1,net.numLayers-1);
  for i=1:net.numLayers-1
    lw = net.LW{i+1};
    lw = reshape(lw,1,prod(size(lw)));
    LW{i} = lw;
    ib = net.b{i+1};
    ib = reshape(ib,1,prod(size(ib)));
    LB{i} = ib;
  end
  % Hyperparameters not available, so estimate them
  [IH_h,IH,HB,HO_h,HO,OB,V_h,V] = hyperest(IW,IB,LW,LB,arch,use_ard);
  if ~use_ard
    IH_h = mean(IH);
    IH   = repmat(IH_h,size(IH));
    tmp  = mean(HB);
    HB   = repmat(tmp,size(HB));
    HO_h = mean(HO);
    HO   = repmat(HO_h,size(HO));
    tmp  = mean(OB);
    OB   = repmat(tmp,size(OB));
    V_h  = mean(V);
    V    = repmat(V_h,size(V));
  end
  % Check if noiseSigmas given in arguments
  if isnan(noiseSigmas)
    V=[];
  elseif ~isempty(noiseSigmas)
    V = noiseSigmas;
  end
  % Check if noiseHyper given in arguments
  if isnan(noiseHyper)
    V_h=[];
  elseif ~isempty(noiseHyper)
    V_h = noiseHyper;
  end
elseif isfield(net,'inputWeights')
  % similar to fbmtools network
  
  IW = net.inputWeights;
  IB = net.inputBiases;
  LW = net.layerWeights;
  LB = net.layerBiases;
  if ~isempty(net.inputHiddenHyper)
    IH_h=net.inputHiddenHyper;
    IH=net.inputHiddenSigmas;
  else
    IH_h=net.inputHiddenSigmas;
    IH=repmat(net.inputHiddenSigmas,1,net.numInputs);
  end
  HB=net.hiddenBiasSigma;
  if ~isempty(net.hiddenOutputHyper)
    HO_h=net.hiddenOutputHyper;
    HO=net.hiddenOutputSigmas;
  else
    HO_h=net.hiddenOutputSigmas;
    HO=repmat(net.hiddenOutputSigmas,1,net.numHidden);
  end
  OB=net.outputBiasSigma;
  V=[];V_h=[];
  if isfield(net,'noiseSigmas')
    if isfield(net,'noiseHyper')
      V_h=net.noiseHyper;
    end
    V=net.noiseSigmas;
  end
else
  error('Unknown network type.');
end

% Pack sigmas and weights
S = join_s(IH_h,IH,HB,HO_h,HO,OB,V_h,V,arch);
W = join_w(IW,IB,LW,LB,arch);

% Find indices of last records in file

% Magic number that starts a record
log_header_magic = hex2dec('F41A');

file = fopen(logfile,'r+',machine_fmt);
if file < 0
  error(['Cannot open file ' filename ' for r/w']);
end

spos = -1;
wpos = -1;
while ~feof(file) & (fread(file,1,'ushort')==log_header_magic)
  % Record type
  rec.type = char(fread(file,1,'short'));
  % Record index
  rec.index = fread(file,1,'int');
  li = rec.index;
  % Read record size
  rec.size = fread(file,1,'short');
  % Skip reserved fields
  fseek(file,6,0);
  % Where is the next record
  datapos = ftell(file);
  nextrec = datapos + rec.size + 16;

  switch rec.type
    case 'S', % Sigmas
      spos = datapos;
    case 'W', % Weights
      wpos = datapos;
    otherwise
      ;
  end

  fseek(file,nextrec,-1);
end

if spos < 0
  error('No sigma records found in log file');
end
if wpos < 0
  error('No weights record found in log file');
end

% Write weights and sigmas into the file
fseek(file,wpos,-1);
fwrite(file,W(end,:),'double');
fseek(file,spos,-1);
fwrite(file,S(end,:),'double');

fclose(file);

function [IH_h,IH,HB,HO_h,HO,OB,V_h,V] = hyperest(IW,IB,LW,LB,arch,use_ard)
%HYPEREST Estimate hyperparameters of Bayesian MLP
%
%   [IH_h,IH,HB,HO_h,HO,OB,V_h,V] = hyperest(IW,IB,LW,LB,ARCH,USE_ARD)
%   returns estimates of hyperparameters for each row in
%   given network matrices IW,IB,LW,LB. Variable outputs
%   should contain Network architecture is given in "arch".
%
%   Note that output sigmas cannot really be estimated using
%   network weights only (they are guesses to be same as
%   output bias variances).
%
%   Better estimate can be determined as follows:
%
%     If "guesses" contains output of a network for each
%     of the test cases and "answers" are the right answers,
%     the you can estimate them as follows:
%
%     >> V = std(guesses-answers);
%     >> V_h = std(V);

% Copyright (C) 1999 Simo Srkk
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.


  if use_ard
    IH=std(reshape(IW,arch.numHidden,arch.numInputs));
  else
    IH=std(IW(:));
  end
  IH(IH==0)=rand;
  IH_h = mean(IH);

  HB = std(IB,0,2);
  HB(find(HB==0)) = rand;

  LW = LW{1};
  HO_h = std(LW,0,2);
  HO_h(find(HO_h==0)) = rand;
  HO = repmat(HO_h,1,arch.numHidden);

  LB = LB{1};
  OB = std(LB,0,2);
  OB(find(OB==0)) = rand;
  V_h = OB;
  V = repmat(V_h,arch.numOutputs,1);

function S = join_s(IH_h,IH,HB,HO_h,HO,OB,V_h,V,arch)
%JOIN_S Join sigmas into FBM's sigma structure
%
%   S = JOIN_S(IH_h,IH,HB,HO_h,HO,OB,V_h,V,ARCH) joins the given
%   input-hidden hyperparameters (IH_h), input-hidden sigmas (IH),
%   hidden bias sigmas (HB), hidden-output hyperparameters (HO_h),
%   hidden-output sigmas (HO), output biases (OB), output
%   variance hyperparameters (V_h) and output variances (V)
%   into form used in FBM log files.
%
%   Architecture is checked to match the architecture given
%   in variable "ARCH".

% Copyright (C) 1999 Simo Srkk
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

% 2000-04-04  Aki Vehtari  <Aki.Vehtari@hut.fi>
%             Added check for variances as classification
%             (logit/softmax) models do not have variances.

S = [];
if size(IH_h,2)~=1
  error('Wrong number of inputs-hidden hyperparameters');
end
S = [S IH_h];
if size(IH,2)~=arch.numInputs
  error('Wrong number of inputs-hidden sigmas');
end
S = [S IH];
if size(HB,2)~=1
  error('Wrong number of hidden-bias sigmas');
end
S = [S HB];
if size(HO_h,2)~=1
  error('Wrong number of hidden-output hyperparameters');
end
S = [S HO_h];
if size(HO,2)~=arch.numHidden
  error('Wrong number of hidden-output sigmas');
end
S = [S HO];
if size(OB,2)~=1
  error('Wrong number of output bias sigmas');
end
S = [S OB];
if ~isempty(V_h)
  if size(V_h,2)~=1
    error('Wrong number of variance hyperparameters');
  end
  S = [S V_h];
end
if ~isempty(V)
  if size(V,2)~=arch.numOutputs
    error('Wrong number of output variances');
  end
  S = [S V];
end

function W = join_w(IW,IB,LW,LB,arch)
%JOIN_W Join network weights for FBM's weight structure
%
%   W = JOIN_W(IW,IB,LW,LB,ARCH) composes inputs weights (IW),
%   input biases (IB), layer weights (LW) and layer biases (LB)
%   into form used in FBM log file. Arguments are checked to match
%   architecture given in arch.
%
%   Function returns single matrix consisting of network
%   weights on each row. The weights are in the order that
%   can be directly written to FBM log file.
%
%   "ARCH" is a struture and it should contain all the fields
%   from FBM's net_arch structure (see net.h). And the following
%   fields:
%
%     numLayers = number of layers
%     numHidden = array of hidden layer sizes
%     numInputs = number of inputs
%     numOutputs = number of outputs


% Copyright (C) 1999 Simo Srkk
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

W = [];

% Check that layers match
if (size(LW,1)~=arch.numLayers) | (size(LB,1)~=arch.numLayers)
  error('Wrong number of layers');
end

% Input offsets
if arch.has_ti
  error('Input offsets cannot be used');
end

% Layer weights
for i=find(arch.has_hh)
  count = arch.numHidden(i+1) * arch.numHidden(i);
  if size(LW{i},2)~=count
    error(['Wrong number of hidden units on layer ' i]);
  end
  W = [W LW{i}];
end

% Input weights
for i=1:10
  if arch.has_ih(i)
    count = arch.numInputs * arch.numHidden(i);
    if size(IW,2)~=count
      error('Wrong number of input weights');
    end
    W = [W IW];
  end
end

% Layer biases (except output)
for i=find(arch.has_bh)
  if i==1
    if size(IB,2)~=arch.numHidden(i)
      error(['Wrong number of biases on layer ' i]);
    end
    W = [W IB];
  else
    if size(LB{i},2)~=arch.numHidden(i)
      error(['Wrong number of biases on layer ' i]);
    end
    W = [W LB{i}];
  end
end

% Layer offsets
if any(arch.has_th)
  error('Layer offsets cannot be used');
end

% Hidden to output weights
if any(arch.has_ho(1:arch.numLayers-1) ~= 0)
  error('Only rightmost layer can be connected to output');
end
if arch.has_ho(arch.numLayers)
  count = arch.numOutputs * arch.numHidden(arch.numLayers);
  if size(LW{arch.numLayers},2)~=count
    error('Wrong number of output weights');
  end
  W = [W LW{arch.numLayers}];
end

% Inputs to output weights
if arch.has_io
  error('Input to output weights cannot be used');
end

% Output biases
if arch.has_bo
  if size(LB{arch.numLayers},2)~=arch.numOutputs
    error('Wrong number of output biases');
  end
  W = [W LB{arch.numLayers}];
end  

