function [fbm,arch] = fbmmlpread(filename,nburn,nthin,nlast,reverse_byte_order,verb)
%FBMMLPREAD Read MLP networks from FBM log file
%
%   [FBMMLP,ARCH]=FBMMLPREAD(LOG,NBURN,NTHIN,NLAST,REVERSE_BYTE_ORDER,VERB)
%   reads simulated MLP-networks from log file of "Flexible Bayesian
%   Models" and returns them as a structure. 
%
%   Optional arguments NBURN, NTHIN and NLAST specify burn-in, thinning
%   and last index. Only samples (1+NBURN):NTHIN:NLAST are returned.
%   Default values are 0, 1 and Inf.
%
%   Non-zero REVERSE_BYTE_ORDER argument specifies that Intel (BE)
%   byte order should be used. Nonzero VERB argument means that
%   so information on status should be printed during reading.
%
%   Example:
%   >> fbmmlp = fbmmlpread('sin.mlp.log');
%   >> y = fbmmlppred(fbmmlp,x);
%
%   See also
%     FBMMLPPRED
%     THIN
%     "Flexible Bayesian Models" Software

% Copyright (C) 1999 Simo Srkk
% Copyright (C) 2003 Aki Vehtari
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

if nargin < 2
  nburn = [];
end
if nargin < 3
  nthin = [];
end
if nargin < 4
  nlast = [];
end
if nargin < 5
  reverse_byte_order=[];
end
if nargin < 6
  verb = [];
end

if isempty(nburn)
  nburn = 0;
end
if isempty(nthin)
  nthin = 1;
end
if isempty(nlast)
  nlast = Inf;
end
if isempty(verb)
  verb = 0;
end

% double type is big-endian in Intel and
% little-endian in almost all other archs.
if ~isempty(reverse_byte_order) & (reverse_byte_order~=0)
  machine_fmt = 'ieee-be.l64';
  double_size = 8;
else
  machine_fmt = 'ieee-le.l64';
  double_size = 8;
end

% Magic number that starts a record
log_header_magic = hex2dec('F41A');

file = fopen(filename,'r',machine_fmt);
if file < 0
  error(['Cannot open file ' filename ' for reading']);
end

fbm = struct(...
    'type','mlp',...
    'numInputs',0,...
    'numLayers',0,...
    'numHidden',0,...
    'numOutputs',0,...
    'inputHiddenHyper',[],...
    'inputHiddenSigmas',[],...
    'hiddenOutputHyper',[],...
    'hiddenOutputSigmas',[],...
    'hiddenBiasSigmas',[],...
    'outputBiasSigmas',[],...
    'noiseHyper',[],...
    'noiseSigmas',[],...
    'inputWeights',[],...
    'layerWeights',{cell(0)},...
    'inputBiases',[],...
    'layerBiases',{cell(0)},...
    'temperature',[],...
    'rejects',[]);

W = [];           % Weights
S = [];           % Sigmas
li = -1;          % Last index seen
status = 'A';     % What record to expect
counter =0;       % 

while ~feof(file) & (fread(file,1,'ushort')==log_header_magic) & (li<nlast)
  % Record type
  rec.type = char(fread(file,1,'short'));
  % Record index
  rec.index = fread(file,1,'int');
  li=rec.index;
  % Read record size
  rec.size = fread(file,1,'ushort');
  % Skip reserved fields
  fseek(file,6,0);
  % Where is the next record
  nextrec = ftell(file) + rec.size + 16;

  if li <0 || (li>=nburn && ~mod(li-nburn,nthin))
    
    % Read the record data
    switch rec.type
     case 'A'
      % Read basic architecture
      fbm.numInputs = fread(file,1,'int');
      fbm.numLayers = fread(file,1,'int');
      place = ftell(file);
      fbm.numHidden = [];
      for i=1:fbm.numLayers
	fbm.numHidden(i) = fread(file,1,'int');
      end
      fseek(file,place+4*10,-1);
      fbm.numOutputs = fread(file,1,'int');    
      
      arch.numInputs = fbm.numInputs;
      arch.numLayers = fbm.numLayers;
      arch.numHidden = fbm.numHidden;
      arch.numOutputs = fbm.numOutputs;
      
      % Do we have input offsets?
      arch.has_ti = fread(file,1,'int');
      
      % Hidden-hidden offsets?
      arch.has_hh = [];
      for i=1:9
	arch.has_hh(i) = fread(file,1,'int');
      end
      
      % Inputs to hidden offsets?
      arch.has_ih = [];
      for i=1:10
	arch.has_ih(i) = fread(file,1,'int');
      end
      
      % Biases for hidden units?
      arch.has_bh = [];
      for i=1:10
	arch.has_bh(i) = fread(file,1,'int');
      end
      
      % Offsets for hidden units?
      arch.has_th = [];
      for i=1:10
	arch.has_th(i) = fread(file,1,'int');
      end
      
      % Hidden to output weights?
      arch.has_ho = [];
      for i=1:10
	arch.has_ho(i) = fread(file,1,'int');
      end
      
      % Input to output weights?
      arch.has_io = fread(file,1,'int');
      
      % Biases for output units?
      arch.has_bo = fread(file,1,'int');
      
      % Do hidden layers have adjustments?
      arch.has_ah = [];
      for i=1:10
	arch.has_ah(i) = fread(file,1,'int');
      end
      
      % Does output layer have adjustments?
      arch.has_ao = fread(file,1,'int');
      
      % Next expected record
      status = 'S';
      
     case 'S', % Sigmas
      counter=counter+1;
      % Did we miss a status?
      if status=='i'
        fbm.rejects = [fbm.rejects; NaN];
        fbm.temperature = [fbm.temperature; NaN];
      elseif status~='S'
        error('Sigma record not found');
      end
      % Read the data
      S(counter,:) = fread(file,rec.size/double_size,'double')';

      % Next expected record
      status = 'W';
      
     case 'W', % Weights
      if status~='W'
        error('Weight record not found');
      end
      % Read the data
      %W = [W; fread(file,rec.size/double_size,'double')'];
      W(counter,:) = fread(file,rec.size/double_size,'double')';
      if verb
        fprintf('Read record %d\n',rec.index);
      end

      % Next expected record
      status = 'i';
      
     case 'i', % Information on MCMC
      
      % Temperature used during this iteration
      temperature = fread(file,1,'float')';
      fbm.temperature(counter,1) = temperature;
      
      % Heatbath decay used during this iteration
      decay = fread(file,1,'float')';
      
      % Skip approx_order
      fseek(file,100,0);
      
      % Factor last used to adjust stepsizes
      stepsize_factor = fread(file,1,'float')';
      
      % Change in total energy for last state proposed
      delta = fread(file,1,'double');
      
      move_point = fread(file,1,'int');
      window_offset = fread(file,1,'int');
      rejects = fread(file,1,'int');
      proposals = fread(file,1,'int');
      time = fread(file,1,'int');
      
      % Rejection rate
      if proposals ~= 0
        fbm.rejects(counter,1) = rejects/proposals;
      else
        fbm.rejects(counter,1) = NaN;
      end
      
      status = 'S';
      
      %    case 'P', % Priors
      %    case 'p', % Momentum
      %    case 'M',
      %    case 'D',
      %    case 'r', % Rand seed
      %    case 'o',
      
     otherwise,
      ;
    end
  end  
  % Find the next record
  fseek(file,nextrec,-1);
  
end

% Split weights
[IW,IB,LW,LB] = split_w(W,arch);
fbm.inputWeights = IW;
fbm.inputBiases  = IB;
fbm.layerWeights = LW;
fbm.layerBiases  = LB;

% Split sigmas
[IH_h,IH,HB,HO_h,HO,OB,V_h,V] = split_s(S,arch);
fbm.inputHiddenHyper = IH_h;
fbm.inputHiddenSigmas = IH;
fbm.hiddenOutputHyper = HO_h;
fbm.hiddenOutputSigmas = HO;
fbm.hiddenBiasSigmas = HB;
fbm.outputBiasSigmas = OB;
fbm.noiseHyper = V_h;
fbm.noiseSigmas = V;

if verb & ~feof(file) & ~isinf(nlast)
  warning(['File ' filename ' was not entirely read.'])
end
fclose(file);

if verb
  fprintf(1,'Read %d iterations from file %s\n',...
      size(fbm.inputWeights,1),filename);
end

function [IH_h,IH,HB,HO_h,HO,OB,V_h,V] = split_s(S,arch)
%SPLIT_S Split FBM's sigma structure (vector) into components
%
%   [IH_h,IH,HB,HO_h,HO,OB,V_h,V] = split_s(S,arch) returns
%   input-hidden hyperparameters (IH_h), input-hidden sigmas (IH),
%   hidden bias sigmas (HB), hidden-output hyperparameters (HO_h),
%   hidden-output sigmas (HO), output biases (OB), output
%   variance hyperparameters (V_h) and output variances (V).
%   
%   The sigmas and hyperparameters are extracted from vector S
%   according to given network architecture. Each row in S
%   corresponds to one network. Rows in output matrices
%   correspond to rows in S.
%
%   "arch" is a struture and it should contain all the fields
%   from FBM's net_arch structure (see net.h). And the following
%   fields:
%
%     numLayers = number of layers
%     numHidden = array of hidden layer sizes
%     numInputs = number of inputs
%     numOutputs = number of outputs

% Copyright (C) 1999 Simo Srkk
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

% 2000-04-04  Aki Vehtari  <Aki.Vehtari@hut.fi>
%             Added check for variances as classification
%             (logit/softmax) models do not have variances.

index = 1;
IH_h = S(:,index);
index = index + 1;
IH = S(:,index:(index-1 + arch.numInputs));
index = index + arch.numInputs;
HB = S(:,index);
index = index + 1;
HO_h = S(:,index);
index = index + 1;
HO = S(:,index:(index-1 + arch.numHidden));
index = index + arch.numHidden;
OB = S(:,index);
% classification (logit/softmax) models do not have variances
if size(S,2) > index
  index = index + 1;
  V_h = S(:,index);
  index = index + 1;
  V = S(:,index:(index-1 + arch.numOutputs));
else
  V_h=[];
  V=[];
end

function [IW,IB,LW,LB] = split_w(W,arch)
%SPLIT_W Split FBM's weight structure to components
%
%   [IW,IB,LW,LB] = split_w(W,arch) returns input weights
%   (IW), input biases (IB), layer weights (LW) and layer biases (LB)
%   which are extracted from vector W according to given network
%   architecture. All the input related things are column vectors
%   and all the layer related are cell arrays of row vectors.
%
%   Each row in W corresponds to one network. Rows in output matrices
%   correspond to rows in S.
%
%   "arch" is a struture and it should contain all the fields
%   from FBM's net_arch structure (see net.h). And the following
%   fields:
%
%     numLayers = number of layers
%     numHidden = array of hidden layer sizes
%     numInputs = number of inputs
%     numOutputs = number of outputs

% Copyright (C) 1999 Simo Srkk
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.


% Initialize outputs
IW = [];
IB = [];
LW = cell(1,arch.numLayers);
LB = cell(1,arch.numLayers);

% Input offsets
if arch.has_ti
  warning(['Input offsets were ignored.' ...
	' Network cannot be simulated correctly.']);
  data = W(1:arch.numInputs);
  W = W(:,arch.numInputs+1:end);
end
  
% Layer weights (except output)
for i=find(arch.has_hh)
  count = arch.numHidden(i+1) * arch.numHidden(i);
  LW{i} = W(:,1:count);
  W = W(:,count+1:end);
end
  
% Input weights
for i=1:10
  if arch.has_ih(i)
    count = arch.numInputs * arch.numHidden(i);
    IW = W(:,1:count);
    W = W(:,count+1:end);
  end
end
  
% Layer biases (except output)
for i=find(arch.has_bh)
  data = W(:,1:arch.numHidden(i));
  if i==1
    IB = data;
  else
    LB{i} = data;
  end
  W = W(:,arch.numHidden(i)+1:end);
end
  
% Layer offsets
for i=1:10
  if arch.has_th(i)
    warning(['Layer offsets were ignored.' ...
	  ' Network cannot be simulated correctly.']);
    data = W(:,1:arch.numHidden(i));
    W = W(:,arch.numHidden(i)+1:end);
  end
end
  
% Hidden to output weights
if any(arch.has_ho(1:arch.numLayers-1) ~= 0)
  warning(['Some hidden to output weights were ignored.' ...
	' Network cannot be simulated correctly.']);
end
if arch.has_ho(arch.numLayers)
  count = arch.numOutputs * arch.numHidden(arch.numLayers);
  LW{arch.numLayers} = W(:,1:count);
  W = W(:,count+1:end);
end
  
% Inputs to output weights
if arch.has_io
  warning(['Inputs to output weights were ignored.' ...
	' Network cannot be simulated correctly.']);
  data = W(:,1:arch.numInputs);
  W = W(:,arch.numInputs+1:end);
end
  
% Output biases
if arch.has_bo
  LB{arch.numLayers} = W(:,1:arch.numOutputs);
  W = W(:,arch.numOutputs+1:end);
end  

