function Y = fbmmlppred(fbmmlp,X,nrnd,varargin)
%FBMMLPPRED Calculate predictive distribution of Bayesian MLP
%
%   Y = FBMMLPPRED(FBMMLP,X,NRND)
%   returns responses of the MCMC-simulated networks in structure
%   FBMMLP. Input values are given as rows of X. NRND samples taken
%   from each target distribution. If NRND is zero only the
%   distribution means are returned. Default value for NRND is 0.
%
%   Return value Y is 3D matrix containing NRND*N 3rd dimensions
%   where N is number of simulated MLP-networks. Every row of Y
%   is a response of one network to the input value on the
%   corresponding row of X.
%   
%   BUGS: t-distributed noise model is not yet supported
%         Only one hidden layer models supported
%
%   Example:
%   >> fbmmlp = fbmmlpread('sin.log');
%   >> Y = fbmmlppred(fbmmlp,x,20);
%   >> mean(Y)
%   >> std(Y)
%
%   See also
%     FBMMLPREAD

% Copyright (C) 1999 Simo Srkk
% Copyright (C) 2000 Aki Vehtari
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

if nargin < 2
  error('Requires at least 2 arguments');
end
if nargin < 3
  nrnd = [];
end
if isempty(nrnd)
  nrnd = 0;
end

% Determine number of inputs/hidden/outputs
nin  = fbmmlp.numInputs;
nhid = fbmmlp.numHidden;
nout = fbmmlp.numOutputs;

% Simulate the networks
nmc=size(fbmmlp.inputWeights,1);
Y = zeros(size(X,1),nout,min(1,nrnd)*nmc);
% Avoid using structures in loops because structures are slooow.
IW=fbmmlp.inputWeights;
IB=fbmmlp.inputBiases;
LW=fbmmlp.layerWeights;
LB=fbmmlp.layerBiases;
if nrnd
  S =fbmmlp.noiseSigmas;
end
nlayers=fbmmlp.numLayers;
if nlayers>1
  error('Only one hidden layer MLPs supported');
end
LW=LW{1};
LB=LB{1};
count = 1;
for i=1:nmc
  net.w1=reshape(IW(i,:),nhid,nin)';
  net.b1=IB(i,:);
  net.w2=reshape(LW(i,:),nout,nhid)';
  net.b2=LB(i,:);
  y=mlp2fwd(net,X);
  if ~nrnd
    % Store the mean values only
    Y(:,:,count) = y;
    count = count+1;
  else
    % Add some noise to outputs
    for k=1:nrnd	 
      Y(:,:,count) = y + S(i,:) .* randn(1,nout);
      count = count+1;
    end
  end
end

function [y, z] = mlp2fwd(net, x)
%MLP2FWD Forward propagation through 2-layer network.
ndata = size(x, 1);
z = tanh(x*net.w1 + ones(ndata, 1)*net.b1);
y = z*net.w2 + ones(ndata, 1)*net.b2;
