function [Y,S] = fbmgppred(fbmgp,TX,TY,X,nrnd)
%FBMGPPRED Calculate predictive distribution of Bayesian GP
%
%   Y = FBMGPPRED(FBMGP,TX,TY,X,NRND) returns
%   responses of the MCMC-simulated Gaussian process in structure
%   FBMGP. Training inputs and targets are given as rows of TX and
%   TY respectively. Input values are given as rows of X. NRND
%   samples taken from each target distribution. If NRND is zero
%   only the distribution means are returned. Default value for
%   NRND is 0.
%
%   Return value Y is 3D matrix containing NRND*N 3rd dimensions
%   where N is number of hyperparameter samples of GP. Every row of
%   Y is a response of GP with one hyperparameters to the input
%   value on the corresponding row of X.
%   
%   BUGS: only exp2 covariance function is supported
%         only 1 output allowed
%
%   Example:
%   >> fbmgp = fbmgpread('sin.log');
%   >> Y = fbmgppred(fbmgp,tx,ty,x,20);
%   >> mean(Y)
%   >> std(Y)
%
%   See also
%     FBMGPREAD

% Copyright (C) 1999 Simo Srkk
% Copyright (C) 2000-2003 Aki Vehtari
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

if nargin < 4
  error('Requires at least 4 arguments');
end
if nargin < 5
  nrnd = [];
end
if isempty(nrnd)
  nrnd = 0;
end

%
if isfield(fbmgp,'mtype')
  mtype=fbmgp.mtype;
else
  mtype='R';
end

% Determine number of inputs/outputs
nin  = fbmgp.numInputs;
nout = fbmgp.numOutputs;

gp.type = 'gp2r';
gp.nin = nin;
gp.nout = 1;

nmc=size(fbmgp.rejects,1);
if isfield(fbmgp,'constSigmas');
  constSigmas=fbmgp.constSigmas;else;constSigmas=[];
end
if isfield(fbmgp,'linearSigmas');
  linearSigmas=fbmgp.linearSigmas;else;linearSigmas=[];
end
if isfield(fbmgp,'linearii');
  gp.linearii=fbmgp.linearii;
end
if isfield(fbmgp,'expScale');
  expScale=fbmgp.expScale;else;expScale=[];
end
if isfield(fbmgp,'expSigmas');
  expSigmas=fbmgp.expSigmas;else;expSigmas=[];
end
if isfield(fbmgp,'expii');
  gp.expii=fbmgp.expii;else;expii=[];
end
if isfield(fbmgp,'jitterSigmas');
  jitterSigmas=fbmgp.jitterSigmas;else;jitterSigmas=[];
end
if isfield(fbmgp,'noiseSigmas');
  noiseSigmas=fbmgp.noiseSigmas;else;noiseSigmas=[];
end
if isfield(fbmgp,'noiseVariances');
  noiseVariances=fbmgp.noiseVariances;else;noiseVariances=[];
end
n=size(TX,1);
n2=size(X,1);
Y = zeros(size(X,1),nout,min(1,nrnd)*nmc);
count = 1;
gp.constSigmas=[];
gp.linearSigmas=[];
gp.expScale=[];
gp.expSigmas=[];
gp.jitterSigmas=[];
gp.noiseSigmas=[];
gp.noiseVariances=[];
for i1=1:nmc
  
  if ~isempty(constSigmas); gp.constSigmas=constSigmas(i1); end
  if ~isempty(linearSigmas); gp.linearSigmas=linearSigmas(i1,:); end
  if ~isempty(expScale); gp.expScale=expScale(i1); end
  if ~isempty(expSigmas); gp.expSigmas=expSigmas(i1,:); end
  if ~isempty(expii); gp.expii=expii(i1,:); end
  if ~isempty(jitterSigmas); gp.jitterSigmas=jitterSigmas(i1,:); end
  if ~isempty(noiseSigmas); gp.noiseSigmas=noiseSigmas(i1,:); end
  if ~isempty(noiseVariances); gp.noiseVariances=noiseVariances(i1,:); end
  C=gptrcov(gp, TX);
  
  K=gpcov(gp, TX, X);
  
  if mtype=='B'
    y=K'*(C\TY(:,i1));
    V=gptrcov(gp, X);

    v=diag(V-K'*(C\K));
    s=sqrt(v);
    nr=1000;
    vr=gplus(y,gtimes(s,randn(n2,nr)));
    for i2=1:n2
      n0=sum(vr(i2,:)<0)+0.1;
      av0=sum(logsig(vr(i2,vr(i2,:)<0)))/n0;
      n1=sum(vr(i2,:)>0)+0.1;
      av1=(sum(logsig(vr(i2,vr(i2,:)>0)))+0.1)/n1;
      pr0=normcdf(-y(i2)/s(i2));
      y(i2)=pr0*av0+(1-pr0)*av1;
    end
  else
    y=K'*(C\TY);
    if nargout >1
      n=size(X,1);
      V=gpcov(gp, X, X);
      V=V+diag(repmat(gp.jitterSigmas.^2,1,n));
      v=diag(V-K'*(C\K));
      S(:,count)=sqrt(v);
    end
  end
  if ~nrnd
    % Store the mean values only
    Y(:,:,count) = y;
    count = count+1;
  else
    % Add some noise to outputs
    NS = fbmgp.noiseSigmas(i1,:);
    for k=1:nrnd	 
      Y(:,:,count) = y + NS .* randn(1,nout);
      count = count+1;
    end
  end
end

function C=gpcov(gp, x1, x2)
if isempty(x2)
  x2=x1;
end
C=0;
if ~isempty(gp.constSigmas)
  C=C+gp.constSigmas.^2;
end
if ~isempty(gp.linearSigmas)
  C=C+sum(gtimes(permute(gtimes(gp.linearSigmas(ii).^2,x1),[1 3 2]),...
		 permute(x2,[3 1 2])),3);
end
if ~isempty(gp.expScale)
  C=C+gp.expScale.^2.*exp(-sum(gtimes(permute(gp.expSigmas.^2,[3 1 2]),...
                                      gminus(permute(x1,[1 3 2]),...
                                             permute(x2,[3 1 2])).^2),3));
end
C(C<eps)=0;

function [C,Cl]=gptrcov(gp, tx)
n=size(tx,1);n1=n+1;
C=gpcov(gp, tx, tx);
C(1:n1:end)=C(1:n1:end)+gp.jitterSigmas.^2;
if nargout > 1
  Cl=C;
end
if ~isempty(gp.noiseVariances)
  C(1:n1:end)=C(1:n1:end)+gp.noiseVariances;
elseif ~isempty(gp.noiseSigmas)
  C(1:n1:end)=C(1:n1:end)+gp.noiseSigmas.^2;
end
