function [fbm,arch] = fbmgpread(filename,nburn,nthin,nlast,reverse_byte_order)
%FBMGPREAD Read  GP models from FBM log file
%
%   [FBMGP,ARCH] = FBMGPREAD(LOG,NBURN,NTHIN,NLAST,REVERSE_BYTE_ORDER,VERB)
%   reads simulated Gaussian Process (GP) parameters from log file of
%   "Flexible Bayesian Models" and returns them as a structure.
%
%   Optional arguments NBURN, NTHIN and NLAST specify burn-in, thinning
%   and last index. Only samples (1+NBURN):NTHIN:NLAST are returned.
%   Default values are 0, 1 and Inf.
%
%   Non-zero REVERSE_BYTE_ORDER argument specifies that Intel (BE)
%   byte order should be used. Nonzero VERB argument means that
%   so information on status should be printed during reading.
%
%   Example:
%   >> fbmgp = fbmgpread('sin.gp.log');
%   >> y = fbmgppred(fbmgp,tx,ty,x);
%
%   See also
%     FBMGPPRED
%     THIN
%     "Flexible Bayesian Models" Software

% Copyright (C) 1999 Simo Srkk
% Copyright (C) 2000-2003 Aki Vehtari
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

if nargin < 2
  nburn = [];
end
if nargin < 3
  nthin = [];
end
if nargin < 4
  nlast = [];
end
if nargin < 5
  reverse_byte_order=[];
end
if nargin < 6
  verb = [];
end

if isempty(nburn)
  nburn = 0;
end
if isempty(nthin)
  nthin = 1;
end
if isempty(nlast)
  nlast = Inf;
end
if isempty(verb)
  verb = 0;
end

% double type is big-endian in Intel and
% little-endian in almost all other archs.
if ~isempty(reverse_byte_order) & (reverse_byte_order~=0)
  machine_fmt = 'ieee-be.l64';
  double_size = 8;
else
  machine_fmt = 'ieee-le.l64';
  double_size = 8;
end

% Magic number that starts a record
log_header_magic = hex2dec('F41A');

file = fopen(filename,'r',machine_fmt);
if file < 0
  error(['Cannot open file ' filename ' for reading']);
end

fbm = struct(...
    'type','gp',...
    'numInputs',0,...
    'numOutputs',0,...
    'constSigmas',[],...
    'linearHyper',[],...
    'linearSigmas',[],...
    'jitterSigmas',[],...
    'expHyper',[],...
    'expSigmas',[],...
    'expScale',[],...
    'noiseHyper',[],...
    'noiseSigmas',[],...
    'noiseVariances',[],...
    'latentValues',[],...
    'temperature',[],...
    'rejects',[]);

arch = [];
S = [];  % Sigmas
L = [];  % Latent values
N = [];  % Noise levels
li = -1;  % Last index seen
counter =0;  % 

while ~feof(file) & (fread(file,1,'ushort')==log_header_magic) & (li<nlast)
  % Record type
  rec.type = char(fread(file,1,'short'));
  % Record index
  rec.index = fread(file,1,'int');
  li=rec.index;
  % Read record size
  rec.size = fread(file,1,'ushort');
  % Skip reserved fields
  fseek(file,6,0);
  % Where is the next record
  nextrec = ftell(file) + rec.size + 16;

  if li <0 || (li>=nburn && ~mod(li-nburn,nthin))
  
    % Read the data
    switch rec.type
    
     case 'P', % Priors
      fbm.numInputs = fread(file,1,'int');
      fbm.numOutputs = fread(file,1,'int');

      arch.numInputs = fbm.numInputs;
      arch.numOutputs = fbm.numOutputs;

      % Read priors for const/linear/jitter
      tmp = cell(3,1);
      flg = [];
      for i=1:3
        flg(i) = fread(file,1,'int'); % Has const/linear/jitter?
                                      % Skip unknown field
        fseek(file,4,0);
        % Read prior
        tmp{i} = [];
        tmp{i}.scale    = fread(file,1,'int');
        tmp{i}.twopoint = fread(file,1,'int');
        tmp{i}.width    = fread(file,1,'double');
        tmp{i}.alpha    = fread(file,3,'double');
      end
      
      arch.hasConstant = flg(1);
      arch.hasLinear   = flg(2);
      arch.hasJitter   = flg(3);
      arch.constPrior   = tmp{1};
      arch.linearPrior  = tmp{2};
      arch.jitterPrior  = tmp{3};

      % Read priors for exponentials
      arch.numExps = fread(file,1,'int');
      arch.powers  = [];
      arch.scalePriors = cell(arch.numExps,1);
      arch.relevancePriors =  cell(arch.numExps,1);

      % Read one exponential
      arch.powers(1) = fread(file,1,'double');
      fseek(file,4,0); % Skip unknown field
      arch.scalePriors{1}.scale    = fread(file,1,'int');
      arch.scalePriors{1}.twopoint = fread(file,1,'int');
      arch.scalePriors{1}.width    = fread(file,1,'double');
      arch.scalePriors{1}.alpha    = fread(file,3,'double');
      arch.relevancePriors{1}.scale    = fread(file,1,'int');
      arch.relevancePriors{1}.twopoint = fread(file,1,'int');
      arch.relevancePriors{1}.width    = fread(file,1,'double');
      arch.relevancePriors{1}.alpha    = fread(file,3,'double');
      
      if arch.numExps > 1
        error('Sorry, cannot read more than one exponential');
      end
      
     case 'S', % Sigmas
      % Read the data
      if li >= 0
        counter=counter+1;
        S(counter,:) = fread(file,rec.size/double_size,'double')';
      end
      
     case 'F', % Latent values
      % Read the data
      if li >= 0
        tmp=fread(file,rec.size/double_size,'double')';
        L(counter,:,:) = ...
            reshape(tmp,1,arch.numOutputs,length(tmp)/arch.numOutputs);
      end
      
     case 'N', % Noise levels
      % Read the data
      if li >= 0
        N(counter,:) = fread(file,rec.size/double_size,'double')';
      end

     case 'i', % Information on MCMC

      if li >= 0
        % Temperature used during this iteration
        temperature = fread(file,1,'float')';
        fbm.temperature(counter,1) = temperature;
        
        % Heatbath decay used during this iteration
        decay = fread(file,1,'float')';
        
        % Skip approx_order
        fseek(file,100,0);
        
        % Factor last used to adjust stepsizes
        stepsize_factor = fread(file,1,'float')';
        
        % Change in total energy for last state proposed
        delta = fread(file,1,'double');
        
        move_point = fread(file,1,'int');
        window_offset = fread(file,1,'int');
        rejects = fread(file,1,'int');
        proposals = fread(file,1,'int');
        time = fread(file,1,'int');
        
        % Rejection rate
        if proposals ~= 0
          fbm.rejects(counter,1) = rejects/proposals;
        else
          fbm.rejects(counter,1) = NaN;
        end
        
      end
     otherwise,
      ;
    end
  end

  % Find the next record
  fseek(file,nextrec,-1);
  
end

% Save the simulations
S = exp(S);
count = 1;
if arch.hasConstant
  if arch.constPrior.alpha(1)~=0
    fbm.constSigmas = S(:,count);
    count=count+1;
  else
    fbm.constSigmas = repmat(arch.constPrior.width,size(S,1),1);
  end
end
if arch.hasLinear
  if arch.linearPrior.alpha(1)~=0
    fbm.linearHyper = S(:,count);
    count=count+1;
    fbm.linearSigmas = S(:,count:(count+arch.numInputs-1));
    count = count+arch.numInputs;
  else
    fbm.linearSigmas = repmat(arch.linearPrior.width,size(S,1),fbm.numInputs);
  end
end
if arch.hasJitter
  if arch.jitterPrior.alpha(1)~=0
    error('Hierachical jitter terms not supported');
  else
    fbm.jitterSigmas = repmat(arch.jitterPrior.width,size(S,1),1);
  end
end
fbm.expScale = S(:,count);
count = count+1;
fbm.expHyper = S(:,count);
count=count+1;
fbm.expSigmas = S(:,count:(count+arch.numInputs-1));
count = count+arch.numInputs;
if count<=size(S,2)
  fbm.noiseSigmas = S(:,count:end);
end

fbm.latentValues = L;
fbm.noiseVariances = N;

if verb & ~feof(file) & ~isinf(nlast)
  warning(['File ' filename ' was not entirely read.'])
end
fclose(file);

if verb
  fprintf(1,'Read %d iterations from file %s\n',...
          size(fbm.expSigmas,1),filename);
end
