##### utils_dimer.py
##### Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
##### This module includes functions needed in dimer and GP-dimer calculations.

import numpy as np


##### This auxiliary function calculates energy and gradient for given images 'R'
##### according to a GP model specified by the remaining input variables.
#####
##### Input:   R         coordinates of the images (ndarray of shape 'N_im' x 'D')
#####          gp_model  GPy model structure
#####
##### Output:  E_R       energy for the images (ndarray of shape 'N_im' x 1)
#####          G_R       gradient for the images (ndarray of shape 'N_im' x 'D')
def potential_gp(R,gp_model):
    N_im = R.shape[0]
    D = R.shape[1]
    EG_R = gp_model.predict_noiseless([R]*(D+1))[0]
    E_R = EG_R[:N_im,:].copy()
    G_R = np.reshape(EG_R[N_im:,:],(D,N_im)).T.copy()
    return E_R, G_R


##### This function substracts a bias from a potential.
#####
##### Input:   pot_general  original potential energy function
#####          R            coordinates of images (ndarray of shape 'n' x 'D')
#####          Elevel       energy level to be subtracted
#####
##### Output:  E_R          energy of images (ndarray of shape 'n' x 1)
#####          G_R          energy of images (ndarray of shape 'n' x 'D')
def subtract_Elevel(pot_general,R,Elevel):
    [E_R,G_R] = pot_general(R)
    E_R = E_R - Elevel
    return E_R, G_R


##### This is an auxiliary function for the dimer method ('dimer.py').
#####
##### Input:   G01          gradient vectors at the middle point and image 1 of the dimer (ndarray of shape 2 x 'D')
#####          orient       unit vector along the direction of the dimer (ndarray of shape 1 x 'D')
#####          dimer_sep    dimer separation (distance from the middle point of the dimer to the two images)
#####   
##### Output:  F_rot        rotational force acting on image 1 of the dimer (ndarray of shape 1 x 'D')
def force_rot(G01,orient,dimer_sep):
    F1 = -G01[1,:][np.newaxis]
    F2 = -2*G01[0,:][np.newaxis]+G01[1,:][np.newaxis]
    F1_rot = F1-np.dot(F1[0,:],orient[0,:])*orient
    F2_rot = F2-np.dot(F2[0,:],orient[0,:])*orient
    F_rot = (F1_rot-F2_rot)/dimer_sep
    return F_rot


##### This is an auxiliary function for the dimer method ('dimer.py').
##### Given a rotation plane spanned by 'orient' and 'F_rot',
##### the dimer is rotated one step towards its minimum energy orientation.
##### The rotation angle is optimized according to the modified Newton method
##### based on a finite difference estimation of the scalar rotational force
##### along the direction of the rotation.
#####
##### Input:   R               coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient          unit vector along the direction of the dimer (before rotation) (ndarray of shape 1 x 'D')
#####          G01             gradient at the middle point and image 1 of the dimer (before rotation) (ndarray of shape 2 x 'D')
#####          F_rot           rotational force acting on image 1 of the dimer (ndarray of shape 1 x 'D')
#####          potential       potential and gradient function
#####          dimer_sep       dimer separation (distance from the middle point of the dimer to the two images)
#####          T_anglerot      convergence threshold for the rotation angle
#####          estim_Curv      if 1, an estimate for the curvature along the direction of the dimer after the rotation is calculated
#####          estim_G1        if 1, an estimate for the gradient at image 1 after the rotation is calculated
#####   
##### Output:  orient_new      unit vector along the direction of the dimer after optimal rotation (ndarray of shape 1 x 'D')
#####          orient_rot_new  unit vector perpendicular to 'orient_new' within the rotation plane of this iteration (ndarray of shape 1 x 'D')
#####          Curv            estimate for the curvature along the direction of the dimer after the rotation (None if estim_Curv = 0)
#####          G1              estimate for the gradient at image 1 after the rotation (empty if estim_G1 = 0)
#####          R1_dtheta       coordinates of image 1 of the dimer after a small test rotation (ndarray of shape 1 x 'D')
#####          E1_dtheta       energy at image 1 of the dimer after a small test rotation (ndarray of shape 1 x 1)
#####          G1_dtheta       gradient at image 1 of the dimer after a small test rotation (ndarray of shape 1 x 'D')
def rotate_dimer(R,orient,G01,F_rot,potential,dimer_sep,T_anglerot,estim_Curv,estim_G1):
    F_0 = np.sqrt(np.sum(np.square(F_rot)))
    C_0 = np.dot(-G01[0,:]+G01[1,:],orient[0,:])/dimer_sep
    dtheta = 0.5*np.arctan(0.5*F_0/np.abs(C_0))
    D = R.shape[1]
    if dtheta < T_anglerot:
        orient_new = orient.copy()
        orient_rot_new = np.zeros((1,D))
        Curv = None
        G1 = np.ndarray(shape=(0,D))
        R1_dtheta = np.ndarray(shape=(0,D))
        E1_dtheta = np.ndarray(shape=(0,1))
        G1_dtheta = np.ndarray(shape=(0,D))
    else:
        orient_rot = F_rot/F_0
        orient_dtheta = np.cos(dtheta)*orient + np.sin(dtheta)*orient_rot
        orient_rot_dtheta = -np.sin(dtheta)*orient + np.cos(dtheta)*orient_rot
        R1_dtheta = R + dimer_sep*orient_dtheta
        E1_dtheta, G1_dtheta = potential(R1_dtheta)
        F_rot_dtheta = force_rot(np.vstack((G01[0,:],G1_dtheta[0,:])),orient_dtheta,dimer_sep)
        F_dtheta = np.dot(F_rot_dtheta[0,:],orient_rot_dtheta[0,:])
        a1 = (F_dtheta-F_0*np.cos(2*dtheta))/(2*np.sin(2*dtheta))
        b1 = -0.5*F_0
        angle_rot = 0.5*np.arctan(b1/a1)
        if angle_rot < 0:
            angle_rot = np.pi/2 + angle_rot
        orient_new = np.cos(angle_rot)*orient + np.sin(angle_rot)*orient_rot
        orient_new = orient_new/np.sqrt(np.sum(np.square(orient_new)))
        orient_rot_new = -np.sin(angle_rot)*orient + np.cos(angle_rot)*orient_rot
        orient_rot_new = orient_rot_new/np.sqrt(np.sum(np.square(orient_rot_new)))
        if estim_Curv > 0:
            Curv = C_0 + a1*(np.cos(2*angle_rot)-1) + b1*np.sin(2*angle_rot)
        else:
            Curv = None
        if estim_G1 > 0:
            G1 = np.sin(dtheta-angle_rot)/np.sin(dtheta)*G01[1,:][np.newaxis]+np.sin(angle_rot)/np.sin(dtheta)*G1_dtheta+(1-np.cos(angle_rot)-np.sin(angle_rot)*np.tan(dtheta/2))*G01[0,:][np.newaxis]
        else:
            G1 = np.ndarray(shape=(0,D))
    return orient_new, orient_rot_new, Curv, G1, R1_dtheta, E1_dtheta, G1_dtheta


##### This is an auxiliary function for the dimer method ('dimer.py').
##### The dimer is rotated one step towards its minimum energy orientation
##### according to the modified Newton method.
#####
##### Input:   R            coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient       unit vector along the direction of the dimer (before rotation) (ndarray of shape 1 x 'D')
#####          G01          gradient at the middle point and image 1 of the dimer (before rotation) (ndarray of shape 2 x 'D')
#####          potential    potential and gradient function
#####          dimer_sep    dimer separation (distance from the middle point of the dimer to the two images)
#####          T_anglerot   convergence threshold for the rotation angle
#####          estim_Curv   if 1, an estimate for the curvature along the direction of the dimer after the rotation is calculated
#####          rotinfo      dictionary including necessary input information for the rotation method [irrelevant for this method]
#####   
##### Output:  orient_new   unit vector along the direction of the dimer after optimal rotation (ndarray of shape 1 x 'D')
#####          Curv         estimate for the curvature along the direction of the dimer after the rotation (None if estim_Curv = 0)
#####          R_obs        coordinates of the new observed location (ndarray of shape 1 x 'D')
#####          E_obs        energy at the new observed location (ndarray of shape 1 x 1)
#####          G_obs        gradient at the new observed location (ndarray of shape 1 x 'D')
def rot_iter_mn(R,orient,G01,potential,dimer_sep,T_anglerot,estim_Curv,rotinfo):
    F_rot = force_rot(G01,orient,dimer_sep)
    orient_new, orient_rot_new, Curv, G1, R_obs, E_obs, G_obs = rotate_dimer(R,orient,G01,F_rot,potential,dimer_sep,T_anglerot,estim_Curv,0)
    return orient_new, Curv, R_obs, E_obs, G_obs


##### This is an auxiliary function for the dimer method ('dimer.py').
##### The dimer is rotated one step towards its minimum energy orientation
##### according to the modified Newton method on a rotation plane chosen
##### based on a conjugate gradient method.
#####
##### Input:   R            coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient       unit vector along the direction of the dimer (before rotation) (ndarray of shape 1 x 'D')
#####          G01          gradient at the middle point and image 1 of the dimer (before rotation) (ndarray of shape 2 x 'D')
#####          potential    potential and gradient function
#####          dimer_sep    dimer separation (distance from the middle point of the dimer to the two images)
#####          T_anglerot   convergence threshold for the rotation angle
#####          estim_Curv   if 1, an estimate for the curvature along the direction of the dimer after the rotation is calculated
#####          rotinfo      dictionary including necessary input information for the rotation method
#####                       - rotinfo['F_rot_old']: rotational force of the previous rotation iteration (ndarray of shape 1 x 'D')
#####                       - rotinfo['F_modrot_old']: modified rotational force of the previous rotation iteration (ndarray of shape 1 x 'D')
#####                       - rotinfo['orient_rot_oldplane']: unit vector perpendicular to 'orient' within the rotation plane of the previous rotation iteration (ndarray of shape 1 x 'D')
#####                       - rotinfo['cgiter_rot']: number of conjugated rotation iterations
#####                       - rotinfo['num_cgiter_rot']: maximum number of conjugated rotation iterations before resetting the conjugate directions
#####   
##### Output:  orient_new   unit vector along the direction of the dimer after optimal rotation (ndarray of shape 1 x 'D')
#####          Curv         estimate for the curvature along the direction of the dimer after the rotation (None if estim_Curv = 0 or no rotation)
#####          R_obs        coordinates of the new observed location (ndarray of shape 1 x 'D')
#####          E_obs        energy at the new observed location (ndarray of shape 1 x 1)
#####          G_obs        gradient at the new observed location (ndarray of shape 1 x 'D')
def rot_iter_cg(R,orient,G01,potential,dimer_sep,T_anglerot,estim_Curv,rotinfo):
    F_rot_old = rotinfo['F_rot_old'].copy()
    F_modrot_old = rotinfo['F_modrot_old'].copy()
    orient_rot_oldplane = rotinfo['orient_rot_oldplane'].copy()
    cgiter_rot = rotinfo['cgiter_rot']
    num_cgiter_rot = rotinfo['num_cgiter_rot']
    D = R.shape[1]
    if cgiter_rot >= num_cgiter_rot:
        F_rot_old = np.zeros((1,D))
        F_modrot_old = np.zeros((1,D))
        orient_rot_oldplane = np.zeros((1,D))
        cgiter_rot = 1
    else:
        cgiter_rot = cgiter_rot + 1
    F_rot = force_rot(G01,orient,dimer_sep)
    F_rot_old2 = np.dot(F_rot_old[0,:],F_rot_old[0,:])
    if F_rot_old2 == 0:
        F_modrot = F_rot.copy()
    else:
        gamma = np.dot(F_rot[0,:]-F_rot_old[0,:],F_rot[0,:])/F_rot_old2
        if gamma < 0 or np.sum(np.square(gamma*F_modrot_old)) > np.sum(np.square(F_rot)):
            F_modrot = F_rot.copy()
            cgiter_rot = 1
        else:
            F_modrot = F_rot + gamma*np.sqrt(np.sum(np.square(rotinfo['F_modrot_old'])))*orient_rot_oldplane
    orient_rot = F_modrot-np.dot(orient[0,:],F_modrot[0,:])*orient
    orient_rot = orient_rot/np.sqrt(np.sum(np.square(orient_rot)))
    F_rot_oriented = np.dot(F_rot[0,:],orient_rot[0,:])*orient_rot
    orient_new, orient_rot_new, Curv, G1, R_obs, E_obs, G_obs = rotate_dimer(R,orient,G01,F_rot_oriented,potential,dimer_sep,T_anglerot,estim_Curv,0)
    if R_obs.shape[1] < 1:
        F_rot = np.zeros((1,D))
        F_modrot = np.zeros((1,D))
        cgiter_rot = 0
    rotinfo['F_rot_old'] = F_rot.copy()
    rotinfo['F_modrot_old'] = F_modrot.copy()
    rotinfo['orient_rot_oldplane'] = orient_rot_new.copy()
    rotinfo['cgiter_rot'] = cgiter_rot
    return orient_new, Curv, R_obs, E_obs, G_obs


##### This is an auxiliary function for the dimer method ('dimer.py').
##### The dimer is rotated one step towards its minimum energy orientation
##### according to the modified Newton method on a rotation plane chosen
##### based on the L-BFGS method (this version does not give the estimated
##### gradient at image 1 after the rotation as an output but instead lets
##### it to be evaluated in the next iteration).
#####
##### Input:   R            coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient       unit vector along the direction of the dimer (before rotation) (ndarray of shape 1 x 'D')
#####          G01          gradient at the middle point and image 1 of the dimer (before rotation) (ndarray of shape 2 x 'D')
#####          potential    potential and gradient function
#####          dimer_sep    dimer separation (distance from the middle point of the dimer to the two images)
#####          T_anglerot   convergence threshold for the rotation angle
#####          estim_Curv   if 1, an estimate for the curvature along the direction of the dimer after the rotation is calculated
#####          rotinfo      dictionary including necessary input information for the rotation method
#####                       - rotinfo['F_rot_old']: rotational force of the previous rotation iteration (ndarray of shape 1 x 'D')
#####                       - rotinfo['deltaR_mem']: change of orientation in 'm' previous rotation iterations (ndarray of shape 'm' x 'D')
#####                       - rotinfo['deltaF_mem']: change of rotational force in 'm' previous rotation iterations excluding the last one (ndarray of shape ('m'-1) x 'D')
#####                       - rotinfo['num_lbfgsiter_rot']: maximum number of previous rotation iterations kept in memory
#####   
##### Output:  orient_new   unit vector along the direction of the dimer after optimal rotation (ndarray of shape 1 x 'D')
#####          Curv         estimate for the curvature along the direction of the dimer after the rotation (None if estim_Curv = 0 or no rotation)
#####          R_obs        coordinates of the new observed location (ndarray of shape 1 x 'D')
#####          E_obs        energy at the new observed location (ndarray of shape 1 x 1)
#####          G_obs        gradient at the new observed location (ndarray of shape 1 x 'D')
def rot_iter_lbfgs(R,orient,G01,potential,dimer_sep,T_anglerot,estim_Curv,rotinfo):
    D = R.shape[1]
    F_rot_old = rotinfo['F_rot_old'].copy()
    deltaR_mem = rotinfo['deltaR_mem'].copy()
    deltaF_mem = rotinfo['deltaF_mem'].copy()
    num_lbfgsiter_rot = rotinfo['num_lbfgsiter_rot']
    m = deltaR_mem.shape[0]
    F_rot = force_rot(G01,orient,dimer_sep)
    if m > 0:
        deltaF_mem = np.vstack((deltaF_mem,F_rot-F_rot_old))
    q = -F_rot[0,:]
    a_mem = np.zeros((m,1))        
    for k in range(m):
        s = deltaR_mem[m-1-k,:]
        y = -deltaF_mem[m-1-k,:]
        rho = 1/np.dot(y,s)
        a = rho*np.dot(s,q)
        a_mem[m-1-k,0] = a
        q = q - a*y
    if m > 0:
        s = deltaR_mem[m-1,:]
        y = -deltaF_mem[m-1,:]
        scaling = np.dot(s,y)/np.dot(y,y)
    else:
        scaling = 0.01
    r = scaling*q
    for k in range(1,m+1):
        s = deltaR_mem[k-1,:]
        y = -deltaF_mem[k-1,:]
        rho = 1/np.dot(y,s)
        b = rho*np.dot(y,r)
        r = r + s*(a_mem[k-1,0]-b)
    orient_rot = r[np.newaxis]-np.dot(orient[0,:],r)*orient
    orient_rot = orient_rot/np.sqrt(np.sum(np.square(orient_rot)))
    F_rot_oriented = np.dot(F_rot[0,:],orient_rot[0,:])*orient_rot
    orient_new, orient_rot_new, Curv, G1, R_obs, E_obs, G_obs = rotate_dimer(R,orient,G01,F_rot_oriented,potential,dimer_sep,T_anglerot,estim_Curv,0)
    if R_obs.shape[1] < 1:
        F_rot = np.zeros((1,D))
        deltaR_mem = np.ndarray(shape=(0,D))
        deltaF_mem = np.ndarray(shape=(0,D))
    else:
        deltaR_mem = np.vstack((deltaR_mem,orient_new-orient))
        if m >= num_lbfgsiter_rot:
            deltaR_mem = np.delete(deltaR_mem,0,0)
            deltaF_mem = np.delete(deltaF_mem,0,0)
    rotinfo['F_rot_old'] = F_rot.copy()
    rotinfo['deltaR_mem'] = deltaR_mem.copy()
    rotinfo['deltaF_mem'] = deltaF_mem.copy()
    return orient_new, Curv, R_obs, E_obs, G_obs


##### This is an auxiliary function for the dimer method ('dimer.py').
##### The dimer is rotated one step towards its minimum energy orientation
##### according to the modified Newton method on a rotation plane chosen
##### based on the L-BFGS method (this version gives the estimated gradient
##### at image 1 after the rotation as an output to be used in the next
##### iteration instead of evaluating it).
#####
##### Input:   R            coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient       unit vector along the direction of the dimer (before rotation) (ndarray of shape 1 x 'D')
#####          G01          gradient at the middle point and image 1 of the dimer (before rotation) (ndarray of shape 2 x 'D')
#####          potential    potential and gradient function
#####          dimer_sep    dimer separation (distance from the middle point of the dimer to the two images)
#####          T_anglerot   convergence threshold for the rotation angle
#####          estim_Curv   if 1, an estimate for the curvature along the direction of the dimer after the rotation is calculated
#####          rotinfo      dictionary including necessary input information for the rotation method
#####                       - rotinfo['F_rot_old']: rotational force of the previous rotation iteration (ndarray of shape 1 x 'D')
#####                       - rotinfo['deltaR_mem']: change of orientation in 'm' previous rotation iterations (ndarray of shape 'm' x 'D')
#####                       - rotinfo['deltaF_mem']: change of rotational force in 'm' previous rotation iterations excluding the last one (ndarray of shape ('m'-1) x 'D')
#####                       - rotinfo['num_lbfgsiter_rot']: maximum number of previous rotation iterations kept in memory
#####   
##### Output:  orient_new   unit vector along the direction of the dimer after optimal rotation (ndarray of shape 1 x 'D')
#####          Curv         estimate for the curvature along the direction of the dimer after the rotation (None if estim_Curv = 0 or no rotation)
#####          R_obs        coordinates of the new observed location (ndarray of shape 1 x 'D')
#####          E_obs        energy at the new observed location (ndarray of shape 1 x 1)
#####          G_obs        gradient at the new observed location (ndarray of shape 1 x 'D')
def rot_iter_lbfgsext(R,orient,G01,potential,dimer_sep,T_anglerot,estim_Curv,rotinfo):
    D = R.shape[1]
    F_rot_old = rotinfo['F_rot_old'].copy()
    deltaR_mem = rotinfo['deltaR_mem'].copy()
    deltaF_mem = rotinfo['deltaF_mem'].copy()
    num_lbfgsiter_rot = rotinfo['num_lbfgsiter_rot']
    m = deltaR_mem.shape[0]
    F_rot = force_rot(G01,orient,dimer_sep)
    if m > 0:
        deltaF_mem = np.vstack((deltaF_mem,F_rot-F_rot_old))
    q = -F_rot[0,:]
    a_mem = np.zeros((m,1))        
    for k in range(m):
        s = deltaR_mem[m-1-k,:]
        y = -deltaF_mem[m-1-k,:]
        rho = 1/np.dot(y,s)
        a = rho*np.dot(s,q)
        a_mem[m-1-k,0] = a
        q = q - a*y
    if m > 0:
        s = deltaR_mem[m-1,:]
        y = -deltaF_mem[m-1,:]
        scaling = np.dot(s,y)/np.dot(y,y)
    else:
        scaling = 0.01
    r = scaling*q
    for k in range(1,m+1):
        s = deltaR_mem[k-1,:]
        y = -deltaF_mem[k-1,:]
        rho = 1/np.dot(y,s)
        b = rho*np.dot(y,r)
        r = r + s*(a_mem[k-1,0]-b)
    orient_rot = r[np.newaxis]-np.dot(orient[0,:],r)*orient
    orient_rot = orient_rot/np.sqrt(np.sum(np.square(orient_rot)))
    F_rot_oriented = np.dot(F_rot[0,:],orient_rot[0,:])*orient_rot
    orient_new, orient_rot_new, Curv, G1, R_obs, E_obs, G_obs = rotate_dimer(R,orient,G01,F_rot_oriented,potential,dimer_sep,T_anglerot,estim_Curv,0)
    if R_obs.shape[1] < 1:
        F_rot = np.zeros((1,D))
        deltaR_mem = np.ndarray(shape=(0,D))
        deltaF_mem = np.ndarray(shape=(0,D))
    else:
        deltaR_mem = np.vstack((deltaR_mem,orient_new-orient))
        if m >= num_lbfgsiter_rot:
            deltaR_mem = np.delete(deltaR_mem,0,0)
            deltaF_mem = np.delete(deltaF_mem,0,0)
    rotinfo['F_rot_old'] = F_rot.copy()
    rotinfo['deltaR_mem'] = deltaR_mem.copy()
    rotinfo['deltaF_mem'] = deltaF_mem.copy()
    rotinfo['G1'] = G1.copy()
    return orient_new, Curv, R_obs, E_obs, G_obs


##### This is an auxiliary function for the dimer method ('dimer.py').
##### The dimer is translated one step towards saddle point according to the
##### Newton method.
#####
##### Input:   R            coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient       unit vector along the direction of the dimer (ndarray of shape 1 x 'D')
#####          F_R          force at the middle point of the dimer (ndarray of shape 1 x 'D')
#####          Curv         curvature of energy along the direction of the dimer
#####          param_trans  [[predefined step length for convex regions, maximum step length]] (ndarray of shape 1 x 2)
#####          transinfo    dictionary including necessary input information for the translation method
#####                       - transinfo['potential']: potential and gradient function
#####
##### Output:  R_new        coordinates of the new middle point of the dimer (ndarray of shape 1 x 'D')
#####          R_obs        coordinates of the new observed location (ndarray of shape 1 x 'D')
#####          E_obs        energy at the new observed location (ndarray of shape 1 x 1)
#####          G_obs        gradient at the new observe location (ndarray of shape 1 x 'D')
def trans_iter(R,orient,F_R,Curv,param_trans,transinfo):
    potential = transinfo['potential']
    steplength_convex = param_trans[0,0]
    max_steplength = param_trans[0,1]
    if Curv < 0:
        F_trans = F_R - 2*np.dot(F_R[0,:],orient[0,:])*orient
        F_0 = np.sqrt(np.sum(np.square(F_trans)))
        orient_search = F_trans/F_0
        dr = 0.5*F_0/np.abs(Curv)
        R_dr = R + dr*orient_search
        E_R_dr, G_R_dr = potential(R_dr)
        F_trans_dr = -G_R_dr + 2*np.dot(G_R_dr[0,:],orient[0,:])*orient
        F_dr = np.dot(F_trans_dr[0,:],orient_search[0,:])
        steplength = -F_0*dr/(F_dr-F_0)
        if steplength < 0:
            steplength = steplength_convex
        elif steplength > max_steplength:
            steplength = max_steplength
        R_obs = R_dr.copy()
        E_obs = E_R_dr.copy()
        G_obs = G_R_dr.copy()
    else:
        F_trans = -np.dot(F_R[0,:],orient[0,:])*orient
        orient_search = F_trans/np.sqrt(np.sum(np.square(F_trans)))
        steplength = steplength_convex
        R_obs = np.ndarray(shape=(0,D))
        E_obs = np.ndarray(shape=(0,1))
        G_obs = np.ndarray(shape=(0,D))
    R_new = R + steplength*orient_search
    return R_new, R_obs, E_obs, G_obs


##### This is an auxiliary function for the dimer method ('dimer.py').
##### The dimer is translated one step towards saddle point according to the
##### conjugate gradient method.
#####
##### Input:   R            coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient       unit vector along the direction of the dimer (ndarray of shape 1 x 'D')
#####          F_R          force at the middle point of the dimer (ndarray of shape 1 x 'D')
#####          Curv         curvature of energy along the direction of the dimer
#####          param_trans  [[step length for a small test displacement, predefined step length for convex regions, maximum step length]] (ndarray of shape 1 x 3)    
#####          transinfo    dictionary including necessary input information for the translation method
#####                       - transinfo['potential']: potential and gradient function
#####                       - transinfo['F_trans_old']: translational force of the previous translation iteration (ndarray of shape 1 x 'D')
#####                       - transinfo['F_modtrans_old']: modified translational force of the previous translation iteration (ndarray of shape 1 x 'D')
#####                       - transinfo['cgiter_trans']: number of conjugated transition iterations
#####                       - transinfo['num_cgiter_trans']: maximum number of conjugated transition iterations before resetting the conjugate directions
#####
##### Output:  R_new        coordinates of the new middle point of the dimer (ndarray of shape 1 x 'D')
#####          R_obs        coordinates of the new observed location (ndarray of shape 1 x 'D')
#####          E_obs        energy at the new observed location (ndarray of shape 1 x 1)
#####          G_obs        gradient at the new observe location (ndarray of shape 1 x 'D')
def trans_iter_cg(R,orient,F_R,Curv,param_trans,transinfo):
    potential = transinfo['potential']
    F_trans_old = transinfo['F_trans_old'].copy()
    F_modtrans_old = transinfo['F_modtrans_old'].copy()
    cgiter_trans = transinfo['cgiter_trans']
    num_cgiter_trans = transinfo['num_cgiter_trans']
    steplength_convex = param_trans[0,0]
    max_steplength = param_trans[0,1]
    D = R.shape[1]
    if cgiter_trans >= num_cgiter_trans:
        F_trans_old = np.zeros((1,D))
        F_modtrans_old = np.zeros((1,D))
        cgiter_trans = 1
    else:
        cgiter_trans = cgiter_trans + 1
    if Curv < 0:
        F_trans = F_R - 2*np.dot(F_R[0,:],orient[0,:])*orient
        F_trans_old2 = np.dot(F_trans_old[0,:],F_trans_old[0,:])
        if F_trans_old2 == 0:
            gamma = 0.0
        else:
            gamma = np.dot(F_trans[0,:]-F_trans_old[0,:],F_trans[0,:])/F_trans_old2
            if gamma < 0 or np.sum(np.square(gamma*F_modtrans_old)) > np.sum(np.square(F_trans)):
                gamma = 0.0
                cgiter_trans = 1
        F_modtrans = F_trans + gamma*F_modtrans_old
        orient_search = F_modtrans/np.sqrt(np.sum(np.square(F_modtrans)))
        F_0 = np.dot(F_trans[0,:],orient_search[0,:])
        dr = 0.5*F_0/np.abs(Curv)
        R_dr = R + dr*orient_search
        E_R_dr, G_R_dr = potential(R_dr)
        F_trans_dr = -G_R_dr + 2*np.dot(G_R_dr[0,:],orient[0,:])*orient     
        F_dr = np.dot(F_trans_dr[0,:],orient_search[0,:])
        steplength = -F_0*dr/(F_dr-F_0)
        if steplength < 0:
            steplength = steplength_convex
            F_trans = np.zeros((1,D))
            F_modtrans = np.zeros((1,D))
            cgiter_trans = 0
        elif steplength > max_steplength:
            steplength = max_steplength
            F_trans = np.zeros((1,D))
            F_modtrans = np.zeros((1,D))
            cgiter_trans = 0
        R_obs = R_dr.copy()
        E_obs = E_R_dr.copy()
        G_obs = G_R_dr.copy()
    else:
        F_trans = -np.dot(F_R[0,:],orient[0,:])*orient
        orient_search = F_trans/np.sqrt(np.sum(np.square(F_trans)))
        steplength = steplength_convex
        F_trans = np.zeros((1,D))
        F_modtrans = np.zeros((1,D))
        cgiter_trans = 0
        R_obs = np.ndarray(shape=(0,D))
        E_obs = np.ndarray(shape=(0,1))
        G_obs = np.ndarray(shape=(0,D))
    R_new = R + steplength*orient_search
    transinfo['F_trans_old'] = F_trans.copy()
    transinfo['F_modtrans_old'] = F_modtrans.copy()
    transinfo['cgiter_trans'] = cgiter_trans
    return R_new, R_obs, E_obs, G_obs


##### This is an auxiliary function for the dimer method ('dimer.py').
##### The dimer is translated one step towards saddle point according to the
##### L-BFGS method.
#####
##### Input:   R            coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient       unit vector along the direction of the dimer (ndarray of shape 1 x 'D')
#####          F_R          force at the middle point of the dimer (ndarray of shape 1 x 'D')
#####          Curv         curvature of energy along the direction of the dimer
#####          param_trans  [[predefined step length for convex regions, maximum step length]] (ndarray of shape 1 x 2)    
#####          transinfo    dictionary including necessary input information for the translation method
#####                       - transinfo['F_trans_old']: translational force of the previous translation iteration (ndarray of shape 1 x 'D')
#####                       - transinfo['deltaR_mem']: change of location in 'm' previous translation iterations (ndarray of shape 'm' x 'D')
#####                       - transinfo['deltaF_mem']: change of translational force in 'm' previous translation iterations excluding the last one (ndarray of shape ('m'-1) x 'D')
#####                       - transinfo['num_lbfgsiter_trans']: maximum number of previous translation iterations kept in memory
#####
##### Output:  R_new        coordinates of the new middle point of the dimer (ndarray of shape 1 x 'D')
#####          R_obs        coordinates of the new observed location [empty for this translation method]
#####          E_obs        energy at the new observed location [empty for this translation method]
#####          G_obs        gradient at the new observe location [empty for this translation method]
def trans_iter_lbfgs(R,orient,F_R,Curv,param_trans,transinfo):
    D = R.shape[1]
    F_trans_old = transinfo['F_trans_old'].copy()
    deltaR_mem = transinfo['deltaR_mem'].copy()
    deltaF_mem = transinfo['deltaF_mem'].copy()
    num_lbfgsiter_trans = transinfo['num_lbfgsiter_trans']
    steplength_convex = param_trans[0,0]
    max_steplength = param_trans[0,1]
    m = deltaR_mem.shape[0]
    if Curv < 0:
        F_trans = F_R - 2*np.dot(F_R[0,:],orient[0,:])*orient
        if m > 0:
            deltaF_mem = np.vstack((deltaF_mem,F_trans-F_trans_old))
        q = -F_trans[0,:]
        a_mem = np.zeros((m,1))        
        for k in range(m):
            s = deltaR_mem[m-1-k,:]
            y = -deltaF_mem[m-1-k,:]
            rho = 1/np.dot(y,s)
            a = rho*np.dot(s,q)
            a_mem[m-1-k,0] = a
            q = q - a*y
        if m > 0:
            s = deltaR_mem[m-1,:]
            y = -deltaF_mem[m-1,:]
            scaling = np.dot(s,y)/np.dot(y,y)
        else:
            scaling = 0.01
        r = scaling*q
        for k in range(1,m+1):
            s = deltaR_mem[k-1,:]
            y = -deltaF_mem[k-1,:]
            rho = 1/np.dot(y,s)
            b = rho*np.dot(y,r)
            r = r + s*(a_mem[k-1,0]-b)
        steplength = np.sqrt(np.sum(np.square(r)))
        if steplength > max_steplength:
            r = max_steplength/steplength*r
            transinfo['F_trans_old'] = np.zeros((1,D))
            transinfo['deltaR_mem'] = np.ndarray(shape=(0,D))
            transinfo['deltaF_mem'] = np.ndarray(shape=(0,D))
        else:
            deltaR_mem = np.vstack((deltaR_mem,-r[np.newaxis]))
            if m >= num_lbfgsiter_trans:
                deltaR_mem = np.delete(deltaR_mem,0,0)
                deltaF_mem = np.delete(deltaF_mem,0,0)
            transinfo['F_trans_old'] = F_trans.copy()
            transinfo['deltaR_mem'] = deltaR_mem.copy()
            transinfo['deltaF_mem'] = deltaF_mem.copy()
        R_new = R - r[np.newaxis]
    else:
        F_trans = -np.dot(F_R[0,:],orient[0,:])*orient
        orient_search = F_trans/np.sqrt(np.sum(np.square(F_trans)))
        R_new = R + steplength_convex*orient_search
        transinfo['F_trans_old'] = np.zeros((1,D))
        transinfo['deltaR_mem'] = np.ndarray(shape=(0,D))
        transinfo['deltaF_mem'] = np.ndarray(shape=(0,D))
    R_obs = np.ndarray(shape=(0,D))
    E_obs = np.ndarray(shape=(0,1))
    G_obs = np.ndarray(shape=(0,D))
    return R_new, R_obs, E_obs, G_obs


##### This is an auxiliary function for the dimer method ('dimer.py').
##### The dimer is translated one step towards saddle point according to the
##### quick-min optimizer implemented using the velocity Verlet algorithm.
#####
##### Input:   R             coordinates of the middle point of the dimer (ndarray of shape 1 x 'D')
#####          orient        unit vector along the direction of the dimer (ndarray of shape 1 x 'D')
#####          F_R           force at the middle point of the dimer (ndarray of shape 1 x 'D')
#####          Curv          curvature of energy along the direction of the dimer
#####          param_trans   [[time step dt, maximum step length]] (ndarray of shape 1 x 2)
#####          transinfo     dictionary including necessary input information for the translation method
#####                        - transinfo['F_trans_old']: translational force of the previous translation iteration (ndarray of shape 1 x 'D')
#####                        - transinfo['V_old']: velocity of the dimer in the previous translation iteration (ndarray of shape 1 x 'D')
#####                        - transinfo['zeroV']: indicator if zero velocity used (for the first iteration)
#####
##### Output:
#####          R_new         coordinates of the new middle point of the dimer (ndarray of shape 1 x 'D')
#####          R_obs         coordinates of new observed locations [empty for this translation method]
#####          E_obs         energy at new observed locations [empty for this translation method]
#####          G_obs         gradient at new observed locations [empty for this translation method]
def trans_iter_qmvv(R,orient,F_R,Curv,param_trans,transinfo):
    F_trans_old = transinfo['F_trans_old'].copy()
    V_old = transinfo['V_old'].copy()
    zeroV = transinfo['zeroV']
    dt = param_trans[0,0]
    max_steplength = param_trans[0,1]
    D = R.shape[1]
    if Curv < 0:
        F_trans = F_R - 2*np.dot(F_R[0,:],orient[0,:])*orient
    else:
        F_trans = -np.dot(F_R[0,:],orient[0,:])*orient
    if zeroV > 0:
        V = np.zeros((1,D))
    else:
        V = V_old + dt/2*(F_trans_old + F_trans)
    normF_trans = np.sqrt(np.sum(np.square(F_trans)))
    P = np.dot(V[0,:],F_trans[0,:])/normF_trans
    V = P*F_trans/normF_trans
    if P < 0:
        V = np.zeros((1,D))
    step = dt*V+np.square(dt)/2*F_trans
    steplength = np.sqrt(np.sum(np.square(step)))
    if steplength > max_steplength:
        R_new = R + max_steplength/steplength*step
        F_trans = np.zeros((1,D))
        V = np.zeros((1,D))
    else:
        R_new = R + step
    R_obs = np.ndarray(shape=(0,D))
    E_obs = np.ndarray(shape=(0,1))
    G_obs = np.ndarray(shape=(0,D))
    transinfo['F_trans_old'] = F_trans.copy()
    transinfo['V_old'] = V.copy()
    transinfo['zeroV'] = 0
    return R_new, R_obs, E_obs, G_obs
