##### utils_atomic.py
##### Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2019
##### This module includes functions needed in the atomic version of GP-sNEB.

import numpy as np

 
##### This function gives the distance between two atomic configurations as defined in
##### the special GPy covariance function 'RBF_atomic'.
##### The distance between configurations C and C' is based on the changes of
##### the inter-atomic distances:
#####
##### dist(C,C') = sqrt(SUM_ij{[(1/r_ij-1/r_ij')/l_ij]^2}), where r_ij and
##### r_ij' are the distances between atoms i and j in configurations C and
##### C', respectively, and l_ij is the lengthscale of the corresponding
##### atom pair type.
#####
##### The input matrices x are assumed to be ndarrays of shape n x 3*'N_mov',
##### where each row represents one configuration including the coordinates of the moving atoms:
##### x_1,y_1,z_1,x_2,y_2,z_2,...
#####
##### The parameter 'conf_info' is a dictionary including necessary information about the configurations:
##### conf_info['conf_fro']: coordinates of active frozen atoms (ndarray of shape 'N_fro' x 3)
##### conf_info['atomtype_mov']: atomtype indices for moving atoms (ndarray of shape 'N_mov')
##### conf_info['atomtype_fro']: atomtype indices for active frozen atoms (ndarray of shape 'N_fro')
##### Atomtypes must be indexed as 0,1,2,...,'n_at'-1 (may include also inactive atomtypes).
##### conf_info['pairtype']: pairtype indices for pairs of atomtypes (ndarray of shape 'n_at' x 'n_at')
##### conf_info['n_pt']: number of active pairtypes
##### Active pairtypes are indexed as 0,1,2,...,'n_pt'-1. Inactive pairtypes are given index -1.
#####
##### Input:   x1           coordinates of the moving atoms in configurations C (ndarray of shape 'n1' x 3*'N_mov')
#####          x2           coordinates of the moving atoms in configurations C' (ndarray of shape 'n2' x 3*'N_mov')
#####          conf_info    dictionary including information about the configurations necessary for the GP model
#####          lengthscale  lengthscales for different atom pair types (ndarray of shape 'n_pt')
#####
##### Output:  dist         matrix including the distances between configurations C and C' (ndarray of shape 'n1' x 'n2')
def dist_at(x1,x2,conf_info,lengthscale):
    conf_fro = conf_info['conf_fro']
    atomtype_mov = conf_info['atomtype_mov']
    atomtype_fro = conf_info['atomtype_fro']
    pairtype = conf_info['pairtype']
    n1 = x1.shape[0]
    n2 = x2.shape[0]
    N_mov = atomtype_mov.shape[0]
    N_fro = atomtype_fro.shape[0]
    s2 = 1.0/lengthscale**2
    if s2.shape[0] == 1:
        s2 = s2*np.ones(n_pt)
    dist2 = np.zeros((n1,n2))
    # distances between moving atoms
    if N_mov > 1:
        for i in range(0, N_mov-1):
            for j in range(i+1, N_mov):
                invr_ij_1 = 1.0/np.sqrt(np.sum((x1[:,(i*3):(i*3+3)]-x1[:,(j*3):(j*3+3)])**2,1))[:,None]
                invr_ij_2 = 1.0/np.sqrt(np.sum((x2[:,(i*3):(i*3+3)]-x2[:,(j*3):(j*3+3)])**2,1))[None,:]
                dist2 = dist2 + s2[pairtype[atomtype_mov[i],atomtype_mov[j]]]*(invr_ij_1-invr_ij_2)**2
    # distances from moving atoms to frozen atoms
    if N_fro > 0:
        for i in range(0, N_mov):
            for j in range(0, N_fro):
                invr_ij_1 = 1.0/np.sqrt(np.sum((x1[:,(i*3):(i*3+3)]-conf_fro[j,0:3][None,:])**2,1))[:,None]
                invr_ij_2 = 1.0/np.sqrt(np.sum((x2[:,(i*3):(i*3+3)]-conf_fro[j,0:3][None,:])**2,1))[None,:]
                dist2 = dist2 + s2[pairtype[atomtype_mov[i],atomtype_fro[j]]]*(invr_ij_1-invr_ij_2)**2
    dist = np.sqrt(dist2)
    return dist


##### This function gives the maximum difference in the logarithmic inter-atomic distances
##### between atomic configurations C and C'.
#####
##### dist(C,C') = MAX_ij{|log(r_ij')-log(r_ij)|} = MAX_ij{|log(r_ij'/r_ij)|}, where r_ij and
##### r_ij' are the distances between atoms i and j in configurations C and C', respectively.
#####
##### The input matrices x are assumed to be ndarrays of shape n x 3*'N_mov',
##### where each row represents one configuration including the coordinates of the moving atoms:
##### x_1,y_1,z_1,x_2,y_2,z_2,...
#####
##### Input:   x1           coordinates of the moving atoms in configurations C (ndarray of shape 'n1' x 3*'N_mov')
#####          x2           coordinates of the moving atoms in configurations C' (ndarray of shape 'n2' x 3*'N_mov')
#####          conf_info    dictionary including information about the configurations necessary for the GP model
#####                       - conf_info['conf_fro']: coordinates of active frozen atoms (ndarray of shape 'N_fro' x 3)
#####                       - conf_info['atomtype_mov']: atomtype indices for moving atoms (ndarray of shape 'N_mov')
#####                       - conf_info['atomtype_fro']: pairtype indices for active frozen atoms (ndarray of shape 'N_fro')
#####                       - conf_info['pairtype']: pairtype indices for pairs of atomtypes (ndarray of shape 'n_at' x 'n_at')
#####                       - conf_info['n_pt']: number of active pairtypes
#####
##### Output:  dist         matrix including the "distances" from configurations C to configurations C' (ndarray of shape 'n1' x 'n2')
def dist_max1Dlog(x1,x2,conf_info):
    conf_fro = conf_info['conf_fro']
    n1,N_mov = x1.shape
    N_mov = int(N_mov/3)
    N_fro = conf_fro.shape[0]
    n2 = x2.shape[0]
    dist = np.zeros((n1,n2))
    # distances between moving atoms
    if N_mov > 1:
        for i in range(0,N_mov-1):
            for j in range((i+1),N_mov):
                r_ij_1 = np.sqrt(np.sum((x1[:,(i*3):(i*3+3)]-x1[:,(j*3):(j*3+3)])**2,1))[:,None]
                r_ij_2 = np.sqrt(np.sum((x2[:,(i*3):(i*3+3)]-x2[:,(j*3):(j*3+3)])**2,1))[None,:]
                dist = np.max((dist,abs(np.log(r_ij_2/r_ij_1))),axis=0)
    # distances from moving atoms to active frozen atoms
    if N_fro > 0:
        for i in range(0,N_mov):
            for j in range(0,N_fro):
                r_ij_1 = np.sqrt(np.sum((x1[:,(i*3):(i*3+3)]-conf_fro[j,0:3])**2,1))[:,None]
                r_ij_2 = np.sqrt(np.sum((x2[:,(i*3):(i*3+3)]-conf_fro[j,0:3])**2,1))[None,:]
                dist = np.max((dist,abs(np.log(r_ij_2/r_ij_1))),axis=0)
    return dist


##### This function gives the maximum of the following ratio over the moving atoms:
##### length of the change of the position of the moving atom between configurations C and C',
##### relative to the distance from the atom to its nearest neighbour atom in C or C'
##### (depending which is smaller):
#####
##### dist(C,C') = MAX_i{sqrt((x_i-x_i')^2+(y_i-y_i')^2+(z_i-z_i')^2)/MIN{mindist_interatomic_i(C),mindist_interatomic_i(C')}},
##### where (x_i,y_i,z_i) and (x_i',y_i',z_i') are the coordinates of atoms i in configurations C and C', respectively.
#####
##### The input matrices x are assumed to be ndarrays of shape n x 3*'N_mov',
##### where each row represents one configuration including the coordinates of the moving atoms:
##### x_1,y_1,z_1,x_2,y_2,z_2,...
#####
##### Input:   x1           coordinates of the moving atoms in configurations C (ndarray of shape 'n1' x 3*'N_mov')
#####          x2           coordinates of the moving atoms in configurations C' (ndarray of shape 'n2' x 3*'N_mov')
#####          conf_info    dictionary including information about the configurations necessary for the GP model
#####                       - conf_info['conf_fro']: coordinates of active frozen atoms (ndarray of shape 'N_fro' x 3)
#####                       - conf_info['atomtype_mov']: atomtype indices for moving atoms (ndarray of shape 'N_mov')
#####                       - conf_info['atomtype_fro']: pairtype indices for active frozen atoms (ndarray of shape 'N_fro')
#####                       - conf_info['pairtype']: pairtype indices for pairs of atomtypes (ndarray of shape 'n_at' x 'n_at')
#####                       - conf_info['n_pt']: number of active pairtypes
#####
##### Output:  dist         matrix including the "distances" from configurations C to configurations C' (ndarray of shape 'n1' x 'n2')
def dist_maxrel_atomwise3(x1,x2,conf_info):   
    n1 = x1.shape[0]
    n2 = x2.shape[0]
    dist = np.zeros((n1,n2))
    mindist_interatomic_x1 = mindist_interatomic(x1,conf_info)
    mindist_interatomic_x2 = mindist_interatomic(x2,conf_info)
    for ind_x1 in range(0,n1):
        for ind_x2 in range(0,n2):
            dist_atomwise = np.sqrt((x1[ind_x1,0::3]-x2[ind_x2,0::3])**2+(x1[ind_x1,1::3]-x2[ind_x2,1::3])**2+(x1[ind_x1,2::3]-x2[ind_x2,2::3])**2)
            dist_rel_atomwise = dist_atomwise/np.min((mindist_interatomic_x1[ind_x1,:],mindist_interatomic_x2[ind_x2,:]),0)
            dist[ind_x1,ind_x2] = np.max(dist_rel_atomwise)
    return dist


##### This function gives the maximum of the following ratio over the moving atoms:
##### length of the change of the position of the moving atom between configurations C and C',
##### relative to the distance from the atom to its nearest neighbour atom in C'.
##### Thus, the function is not symmetric as dist(C,C') is not equal to dist(C',C):
#####
##### dist(C,C') = MAX_i{sqrt((x_i-x_i')^2+(y_i-y_i')^2+(z_i-z_i')^2)/mindist_interatomic_i(C')},
##### where (x_i,y_i,z_i) and (x_i',y_i',z_i') are the coordinates of atoms i in configurations C and C', respectively.
#####
##### The input matrices x are assumed to be ndarrays of shape n x 3*'N_mov',
##### where each row represents one configuration including the coordinates of the moving atoms:
##### x_1,y_1,z_1,x_2,y_2,z_2,...
#####
##### Input:   x1           coordinates of the moving atoms in configurations C (ndarray of shape 'n1' x 3*'N_mov')
#####          x2           coordinates of the moving atoms in configurations C' (ndarray of shape 'n2' x 3*'N_mov')
#####          conf_info    dictionary including information about the configurations necessary for the GP model
#####                       - conf_info['conf_fro']: coordinates of active frozen atoms (ndarray of shape 'N_fro' x 3)
#####                       - conf_info['atomtype_mov']: atomtype indices for moving atoms (ndarray of shape 'N_mov')
#####                       - conf_info['atomtype_fro']: pairtype indices for active frozen atoms (ndarray of shape 'N_fro')
#####                       - conf_info['pairtype']: pairtype indices for pairs of atomtypes (ndarray of shape 'n_at' x 'n_at')
#####                       - conf_info['n_pt']: number of active pairtypes
#####
##### Output:  dist         matrix including the "distances" from configurations C to configurations C' (ndarray of shape 'n1' x 'n2')
def dist_maxrel_atomwise2(x1,x2,conf_info):   
    n1 = x1.shape[0]
    n2 = x2.shape[0]
    dist = np.zeros((n1,n2))
    mindist_interatomic_x2 = mindist_interatomic(x2,conf_info)
    for ind_x1 in range(0,n1):
        dist_atomwise_ind_x1 = np.sqrt((x1[ind_x1:(ind_x1+1),0::3]-x2[:,0::3])**2+(x1[ind_x1:(ind_x1+1),1::3]-x2[:,1::3])**2+(x1[ind_x1:(ind_x1+1),2::3]-x2[:,2::3])**2)
        dist_rel_atomwise_ind_x1 = dist_atomwise_ind_x1/mindist_interatomic_x2
        dist[ind_x1,:] = np.max(dist_rel_atomwise_ind_x1,1)
    return dist


##### This function gives the distance from each moving atom to its nearest neighbour atom in configuration C.
#####
##### The input matrix 'x' is assumed to be ndarray of shape 'n' x 3*'N_mov',
##### where each row represents one configuration including the coordinates of the moving atoms:
##### x_1,y_1,z_1,x_2,y_2,z_2,...
#####
##### Input:   x            coordinates of the moving atoms in configurations C (ndarray of shape 'n' x 3*'N_mov')
#####          conf_info    disctionary including information about the configurations necessary for the GP model
#####                       - conf_info['conf_fro']: coordinates of active frozen atoms (ndarray of shape 'N_fro' x 3)
#####                       - conf_info['atomtype_mov']: atomtype indices for moving atoms (ndarray of shape 'N_mov')
#####                       - conf_info['atomtype_fro']: pairtype indices for active frozen atoms (ndarray of shape 'N_fro')
#####                       - conf_info['pairtype']: pairtype indices for pairs of atomtypes (ndarray of shape 'n_at' x 'n_at')
#####                       - conf_info['n_pt']: number of active pairtypes
#####
##### Output:  dist         matrix including the minimum interatomic distances for each moving atom in configurations C (ndarray of shape 'n' x 'N_mov')
def mindist_interatomic(x,conf_info):
    conf_fro = conf_info['conf_fro']
    n,N_mov = x.shape
    N_mov = int(N_mov/3)
    N_fro = conf_fro.shape[0]
    dist = np.inf*np.ones((n,N_mov))
    # distances between moving atoms
    if N_mov > 1:
        for i in range(0,N_mov-1):
            for j in range((i+1),N_mov):
                r_ij = np.sqrt(np.sum((x[:,(i*3):(i*3+3)]-x[:,(j*3):(j*3+3)])**2,1))
                dist[:,i] = np.min((dist[:,i],r_ij),axis=0)
                dist[:,j] = np.min((dist[:,j],r_ij),axis=0)
    # distances from moving atoms to active frozen atoms
    if N_fro > 0:
        for i in range(0,N_mov):
            for j in range(0,N_fro):
                r_ij = np.sqrt(np.sum((x[:,(i*3):(i*3+3)]-conf_fro[j,0:3])**2,1))
                dist[:,i] = np.min((dist[:,i],r_ij),axis=0)
    return dist


##### This function sets pairtype indices for moving+moving atom pairs and
##### gives the updated number of active pairtypes.
#####
##### Input:
#####   atomtype_mov  atomtype indices for moving atoms (ndarray of shape 'N_mov')
#####   pairtype      pairtype indices for pairs of atomtypes before the update (ndarray of shape 'n_at' x 'n_at')
#####   n_pt          number of active pairtypes before the update
#####
##### Output:
#####   pairtype      indices for pairs of atomtypes after the update (ndarray of shape 'n_at' x 'n_at') 
#####   n_pt          number of active pairtypes after the update
def set_pairtype_mov(atomtype_mov,pairtype,n_pt):
    pairtype = pairtype.copy()
    N_mov = atomtype_mov.shape[0]
    for i in range(0,N_mov-1):
        at_i = atomtype_mov[i]
        for j in range(i+1,N_mov):
            at_j = atomtype_mov[j]
            if pairtype[at_i,at_j] == -1:
                pairtype[at_i,at_j] = n_pt
                pairtype[at_j,at_i] = n_pt
                n_pt = n_pt + 1
    return pairtype, n_pt


##### This function activates inactive frozen atoms within the radius of
##### 'actdist_fro' from some moving atom in configurations 'R_new'.
##### These frozen atoms are then taken into account in the covariance function.
##### 
##### When a frozen atom is activated, its coordinates and atomtype index are added
##### to 'conf_info['conf_fro']' and 'conf_info['atomtype_fro']', respectively,
##### and removed from 'conf_info_inactive['conf_fro']' and 'conf_info_inactive['atomtype_fro']'.
##### If the new frozen atom activates new pairtypes, also 'conf_info['pairtype']'
##### and 'conf_info['n_pt']' are updated. 
#####
##### Input:
#####   conf_info           dictionary including information about the configurations necessary for the GP model
#####                       - conf_info['conf_fro']: coordinates of active frozen atoms (ndarray of shape 'N_fro' x 3)
#####                       - conf_info['atomtype_mov']: atomtype indices for moving atoms (ndarray of shape 'N_mov')
#####                       - conf_info['atomtype_fro']: pairtype indices for active frozen atoms (ndarray of shape 'N_fro')
#####                       - conf_info['pairtype']: pairtype indices for pairs of atomtypes (ndarray of shape 'n_at' x 'n_at')
#####                       - conf_info['n_pt']: number of active pairtypes
#####   conf_info_inactive  dictionary including information about inactive frozen atoms
#####                       - conf_info_inactive['conf_ifro']: coordinates of inactive frozen atoms (ndarray of shape 'N_ifro' x 3)
#####                       - conf_info_inactive['atomtype_ifro']: atomtype indices for inactive frozen atoms (ndarray of shape 'N_ifro')
#####   R_new               coordinates of moving atoms in the new observations (ndarray of shape 'N_obs' x 3*'N_mov')
#####   actdist_fro         activation distance for moving+frozen atom pairs (inf when all active)
#####
##### Output:
#####   new_act             1 if new active frozen atoms included, 0 if not
def update_active_fro(conf_info,conf_info_inactive,R_new,actdist_fro):
    new_act = 0 # indicator of occurrence of new active frozen atoms
    atomtype_active_fro_new = np.array([], dtype='int64') # atomtypes of new active frozen atoms    
    if conf_info_inactive['conf_ifro'].shape[0] > 0:
        if actdist_fro < np.inf:
            # Activate inactive frozen atoms within the radius of 'actdist_fro' from some moving atom:
            for i in range(0,R_new.shape[0]):
                dist_ifro_i = np.sqrt((conf_info_inactive['conf_ifro'][:,0:1]-R_new[i,0::3])**2+(conf_info_inactive['conf_ifro'][:,1:2]-R_new[i,1::3])**2+(conf_info_inactive['conf_ifro'][:,2:3]-R_new[i,2::3])**2)
                active_fro_new_i = np.nonzero(np.any(dist_ifro_i<=actdist_fro,1))[0]
                if active_fro_new_i.shape[0] > 0:
                    new_act = 1
                    conf_info['conf_fro'] = np.vstack((conf_info['conf_fro'],conf_info_inactive['conf_ifro'][active_fro_new_i,:]))
                    conf_info['atomtype_fro'] = np.hstack((conf_info['atomtype_fro'],conf_info_inactive['atomtype_ifro'][active_fro_new_i]))
                    atomtype_active_fro_new = np.unique(np.hstack((atomtype_active_fro_new,conf_info_inactive['atomtype_ifro'][active_fro_new_i])))
                    conf_info_inactive['conf_ifro'] = np.delete(conf_info_inactive['conf_ifro'],active_fro_new_i,0)
                    conf_info_inactive['atomtype_ifro'] = np.delete(conf_info_inactive['atomtype_ifro'],active_fro_new_i)
        else:
            # Activate all inactive frozen atoms:
            new_act = 1
            conf_info['conf_fro'] = np.vstack((conf_info['conf_fro'],conf_info_inactive['conf_ifro']))
            conf_info['atomtype_fro'] = np.hstack((conf_info.atomtype_fro,conf_info_inactive['atomtype_ifro']))
            atomtype_active_fro_new = np.unique(conf_info_inactive['atomtype_ifro'])
            conf_info_inactive['conf_ifro'] = np.empty((0,3))
            conf_info_inactive['atomtype_ifro'] = np.empty(0)
        # Activate new pairtypes if necessary:
        if new_act > 0:
            for at_i in np.unique(conf_info['atomtype_mov']):
                for at_j in atomtype_active_fro_new:
                    if conf_info['pairtype'][at_i,at_j] == -1:
                        conf_info['pairtype'][at_i,at_j] = conf_info['n_pt']
                        conf_info['pairtype'][at_j,at_i] = conf_info['n_pt']
                        conf_info['n_pt'] = conf_info['n_pt'] + 1
    return new_act
