##### utils.py
##### Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
##### This module includes functions needed in NEB and GP-NEB calculations.

import numpy as np


##### This function calculates the tangent of the path at the intermediate images.
##### 
##### Input:
#####   R    coordinates of the images (ndarray of shape 'N_im' x 'D')
#####   E_R  energy at the images (ndarray of shape 'N_im' x 1)
#####
##### Output: 
#####   Z    normalized tangent vectors of the path at the intermediate images (ndarray of shape 'N_im'-2 x 'D')
#####
def tangent(R, E_R):
    N_im = R.shape[0]
    D = R.shape[1]
    Z = np.zeros((N_im-2,D))
    for i in range(1,N_im-1):
        if E_R[i-1,0] < E_R[i,0] and E_R[i,0] < E_R[i+1,0]:
            z = R[i+1,:]-R[i,:]
        elif E_R[i-1,0] > E_R[i,0] and E_R[i,0] > E_R[i+1,0]:
            z = R[i,:]-R[i-1,:]
        else:
            zplus = R[i+1,:]-R[i,:]
            zminus = R[i,:]-R[i-1,:]
            dEmax = np.max((np.abs(E_R[i-1,0]-E_R[i,0]),np.abs(E_R[i+1,0]-E_R[i,0])))
            dEmin = np.min((np.abs(E_R[i-1,0]-E_R[i,0]),np.abs(E_R[i+1,0]-E_R[i,0])))
            if E_R[i-1,0] < E_R[i+1,0]:
                z = dEmax*zplus + dEmin*zminus
            else:
                z = dEmin*zplus + dEmax*zminus
        Z[i-1,:] = z/np.sqrt(np.sum(np.square(z)))
    return Z


##### This function gives the NEB force for each image on the path according to the sNEB method.
#####
##### Input:  
#####   R        coordinates of the 'N_im' images on the path (ndarray of shape 'N_im' x 'D')
#####   E_R      energy at the 'N_im' images on the path (ndarray of shape 'N_im' x 1)
#####   G_R      gradient at the 'N_im' images on the path (ndarray of shape 'N_im' x D)
#####   k_par    spring constant for the parallel spring force
#####   k_perp   spring constant for the perpendicular spring force
#####   CI_on    1 if the climbing image option is in use, 0 if not
#####
##### Output:  
#####   F_R      NEB force acting on the 'N_im'-2 intermediate images (ndarray of shape 'N_im'-2 x 'D')
#####   normFCI  norm of the NEB force acting on the climbing image (zero if CI is off)
#####   i_CI     index of the climbing image (among the 'N_im'-2 intermediate images)
#####
def force_sNEB(R, E_R, G_R, k_par=1.0, k_perp=1.0, CI_on=0):
    N_im = R.shape[0]
    Z_R = tangent(R,E_R)
    G_R_perp = G_R[1:-1,:] - np.sum(G_R[1:-1,:]*Z_R,1)[np.newaxis].T*Z_R
    int_R_next = R[2:,:] - R[1:-1,:]
    int_R_prev = R[1:-1,:] - R[:-2,:]
    d_R_next = np.sqrt(np.sum(np.square(int_R_next),1)[np.newaxis].T)
    d_R_prev = np.sqrt(np.sum(np.square(int_R_prev),1)[np.newaxis].T)
    cosphi_R = np.sum(int_R_next*int_R_prev,1)[np.newaxis].T/d_R_next/d_R_prev
    F_R_spring2 = k_perp*0.5*(1.0+np.cos(np.pi*cosphi_R))*(int_R_next-int_R_prev)
    F_R_spring_perp = F_R_spring2 - np.sum(F_R_spring2*Z_R,1)[np.newaxis].T*Z_R
    Di = np.zeros((N_im-2,1))
    DN = 0.0
    for i in range(N_im-2):
        DN = DN + d_R_prev[i,0]
        Di[i,0] = DN
    DN = DN + d_R_next[N_im-3,0]
    if CI_on > 0:
        E_CI = np.max(E_R[1:-1,0])
        i_CI = np.argmax(E_R[1:-1,0])+1
        d_ave1 = Di[i_CI-1,0]/i_CI
        d_ave2 = (DN-Di[i_CI-1,0])/(N_im-1-i_CI)
        Di_ideal1 = np.array([range(1,i_CI)]).T*d_ave1
        Di_ideal2 = Di[i_CI-1,0] + np.array([range(1,N_im-1-i_CI)]).T*d_ave2
        F_R_spring_par = k_par*np.vstack(((Di_ideal1-Di[:(i_CI-1),:])/(2.0*d_ave1),0,(Di_ideal2-Di[i_CI:,:])/(2*d_ave2)))*Z_R
    else:
        d_ave = DN/(N_im-1)
        Di_ideal = np.array([range(1,N_im-1)]).T*d_ave
        F_R_spring_par = k_par*(Di_ideal-Di)/(2*d_ave)*Z_R
        normFCI = 0
        i_CI = 0
    F_R = -G_R_perp + F_R_spring_par + F_R_spring_perp
    if CI_on > 0:
        F_R[i_CI-1,:] = -G_R[i_CI,:] + 2*G_R[i_CI,:].dot(Z_R[i_CI-1,:])*Z_R[i_CI-1,:]
        normFCI = np.sqrt(np.sum(np.square(F_R[i_CI-1,:])))
    return F_R, normFCI, i_CI


##### This function gives the NEB force for each image on the path according to the sNEB method.
#####
##### Input:  
#####   R            coordinates of the 'N_im' images on the path (ndarray of shape 'N_im' x 'D')
#####   E_R          energy at the 'N_im' images on the path (ndarray of shape 'N_im' x 1)
#####   G_R          gradient at the 'N_im' images on the path (ndarray of shape 'N_im' x D)
#####   param_force  [[parallel spring constant, perpendicular spring constant]] (ndarray of shape 1 x 2)
#####   CI_on        1 if the climbing image option is in use, 0 if not
#####
##### Output:  
#####   F_R          NEB force acting on the 'N_im'-2 intermediate images (ndarray of shape 'N_im'-2 x 'D')
#####   maxG_R_perp  maximum component of the gradient perpendicular to the path tangent at the 'N_im'-2 intermediate images (ndarray of shape 'N_im'-2 x 1)
#####   maxG_CI      maximum component of the gradient at the climbing image (zero if CI is off)
#####   i_CI         index of the climbing image (among the 'N_im'-2 intermediate images)
#####
def force_sNEB2(R, E_R, G_R, param_force = np.array([[1.0,1.0]]), CI_on=0):
    k_par = param_force[0,0]
    k_perp = param_force[0,1]
    N_im = R.shape[0]
    Z_R = tangent(R,E_R)
    G_R_perp = G_R[1:-1,:] - np.sum(G_R[1:-1,:]*Z_R,1)[np.newaxis].T*Z_R
    maxG_R_perp = np.max(np.abs(G_R_perp),1)[np.newaxis].T
    int_R_next = R[2:,:] - R[1:-1,:]
    int_R_prev = R[1:-1,:] - R[:-2,:]
    d_R_next = np.sqrt(np.sum(np.square(int_R_next),1)[np.newaxis].T)
    d_R_prev = np.sqrt(np.sum(np.square(int_R_prev),1)[np.newaxis].T)
    cosphi_R = np.sum(int_R_next*int_R_prev,1)[np.newaxis].T/d_R_next/d_R_prev
    F_R_spring2 = k_perp*0.5*(1.0+np.cos(np.pi*cosphi_R))*(int_R_next-int_R_prev)
    F_R_spring_perp = F_R_spring2 - np.sum(F_R_spring2*Z_R,1)[np.newaxis].T*Z_R
    Di = np.zeros((N_im-2,1))
    DN = 0.0
    for i in range(N_im-2):
        DN = DN + d_R_prev[i,0]
        Di[i,0] = DN
    DN = DN + d_R_next[N_im-3,0]
    if CI_on > 0:
        i_CI = np.argmax(E_R[1:-1,0])+1
        d_ave1 = Di[i_CI-1,0]/i_CI
        d_ave2 = (DN-Di[i_CI-1,0])/(N_im-1-i_CI)
        Di_ideal1 = np.array([range(1,i_CI)]).T*d_ave1
        Di_ideal2 = Di[i_CI-1,0] + np.array([range(1,N_im-1-i_CI)]).T*d_ave2
        F_R_spring_par = k_par*np.vstack(((Di_ideal1-Di[:(i_CI-1),:])/(2.0*d_ave1),0,(Di_ideal2-Di[i_CI:,:])/(2*d_ave2)))*Z_R
    else:
        d_ave = DN/(N_im-1)
        Di_ideal = np.array([range(1,N_im-1)]).T*d_ave
        F_R_spring_par = k_par*(Di_ideal-Di)/(2*d_ave)*Z_R
        maxG_CI = 0
        i_CI = 0
    F_R = -G_R_perp + F_R_spring_par + F_R_spring_perp
    if CI_on > 0:
        F_R[i_CI-1,:] = -G_R[i_CI,:] + 2*G_R[i_CI,:].dot(Z_R[i_CI-1,:])*Z_R[i_CI-1,:]
        normFCI = np.sqrt(np.sum(np.square(F_R[i_CI-1,:])))
        maxG_CI = np.max(np.abs(G_R[i_CI,:]))
    return F_R, maxG_R_perp, maxG_CI, i_CI


##### This function gives the NEB force for each image on the path according to the regular NEB method.
#####
##### Input:  
#####   R        coordinates of the 'N_im' images on the path (ndarray of shape 'N_im' x 'D')
#####   E_R      energy at the 'N_im' images on the path (ndarray of shape 'N_im' x 1)
#####   G_R      gradient at the 'N_im' images on the path (ndarray of shape 'N_im' x D)
#####   k_par    spring constant for the parallel spring force
#####   CI_on    1 if the climbing image option is in use, 0 if not
#####
##### Output:  
#####   F_R      NEB force acting on the 'N_im'-2 intermediate images (ndarray of shape 'N_im'-2 x 'D')
#####   normFCI  norm of the NEB force acting on the climbing image (zero if CI is off)
#####   i_CI     index of the climbing image (among the 'N_im'-2 intermediate images)
#####
def force_NEB(R, E_R, G_R, k_par=1.0, CI_on=0):
    N_im = R.shape[0]
    Z_R = tangent(R,E_R)
    G_R_perp = G_R[1:-1,:] - np.sum(G_R[1:-1,:]*Z_R,1)[np.newaxis].T*Z_R
    int_R_next = R[2:,:] - R[1:-1,:]
    int_R_prev = R[1:-1,:] - R[:-2,:]
    d_R_next = np.sqrt(np.sum(np.square(int_R_next),1)[np.newaxis].T)
    d_R_prev = np.sqrt(np.sum(np.square(int_R_prev),1)[np.newaxis].T)
    if CI_on > 0:
        E_CI = np.max(E_R[1:-1,0])
        i_CI = np.argmax(E_R[1:-1,0])+1
    else:
        normFCI = 0
        i_CI = 0
    F_R_spring_par = k_par*(d_R_next-d_R_prev)*Z_R
    F_R = -G_R_perp + F_R_spring_par
    if CI_on > 0:
        F_R[i_CI-1,:] = -G_R[i_CI,:] + 2*G_R[i_CI,:].dot(Z_R[i_CI-1,:])*Z_R[i_CI-1,:]
        normFCI = np.sqrt(np.sum(np.square(F_R[i_CI-1,:])))
    return F_R, normFCI, i_CI


##### This function gives the NEB force for each image on the path according to the regular NEB method.
#####
##### Input:  
#####   R            coordinates of the 'N_im' images on the path (ndarray of shape 'N_im' x 'D')
#####   E_R          energy at the 'N_im' images on the path (ndarray of shape 'N_im' x 1)
#####   G_R          gradient at the 'N_im' images on the path (ndarray of shape 'N_im' x D)
#####   param_force  parallel spring constant
#####   CI_on        1 if the climbing image option is in use, 0 if not
#####
##### Output:  
#####   F_R          NEB force acting on the 'N_im'-2 intermediate images (ndarray of shape 'N_im'-2 x 'D')
#####   maxG_R_perp  maximum component of the gradient perpendicular to the path tangent at the 'N_im'-2 intermediate images (ndarray of shape 'N_im'-2 x 1)
#####   maxG_CI      maximum component of the gradient at the climbing image (zero if CI is off)
#####   i_CI         index of the climbing image (among the 'N_im'-2 intermediate images)
#####
def force_NEB2(R, E_R, G_R, param_force=1.0, CI_on=0):
    k_par = param_force
    N_im = R.shape[0]
    Z_R = tangent(R,E_R)
    G_R_perp = G_R[1:-1,:] - np.sum(G_R[1:-1,:]*Z_R,1)[np.newaxis].T*Z_R
    maxG_R_perp = np.max(np.abs(G_R_perp),1)[np.newaxis].T
    int_R_next = R[2:,:] - R[1:-1,:]
    int_R_prev = R[1:-1,:] - R[:-2,:]
    d_R_next = np.sqrt(np.sum(np.square(int_R_next),1)[np.newaxis].T)
    d_R_prev = np.sqrt(np.sum(np.square(int_R_prev),1)[np.newaxis].T)
    if CI_on > 0:
        i_CI = np.argmax(E_R[1:-1,0])+1
    else:
        maxG_CI = 0
        i_CI = 0
    F_R_spring_par = k_par*(d_R_next-d_R_prev)*Z_R
    F_R = -G_R_perp + F_R_spring_par
    if CI_on > 0:
        F_R[i_CI-1,:] = -G_R[i_CI,:] + 2*G_R[i_CI,:].dot(Z_R[i_CI-1,:])*Z_R[i_CI-1,:]
        maxG_CI = np.max(np.abs(G_R[i_CI,:]))
    return F_R, maxG_R_perp, maxG_CI, i_CI


##### This function initializes a linear path 'R_line' of 'N_im' images
##### between two minimum points 'min1' and 'min2'.
#####
##### Input:
#####   min1    coordinates for minimum point 1 (ndarray of shape 1 x 'D')
#####   min2    coordinates for minimum point 2 (ndarray of shape 1 x 'D')
#####   N_im    number of images on the path (scalar)
#####
##### Output:
#####   R_line  coordinates for the images on the linear path (ndarray of shape 'N_im' x 'D')
#####
def initialize_path_linear(min1, min2, N_im):
    D = min1.shape[1]
    R_line = np.zeros((N_im,D))
    R_line[0,:] = min1
    R_line[-1,:] = min2
    interval = (min2-min1)/(N_im-1)
    for i in range(1,N_im-1):
        R_line[i,:] = min1 + interval*i
    return R_line


##### This is function moves the path simply one step along the NEB force.
#####
##### Input:
#####   R           coordinates for the images on the current path (ndarray of shape 'N_im' x 'D')
#####   F_R         NEB force on the intermediate images on the current path (ndarray of shape 'N_im'-2 x 'D')
#####   param_step  step size coefficient K_step (scalar)
#####   F_R_old     NEB force on the intermediate images on the previous path (irrelevant for this method)
#####   V_old       velocity of the intermediate images (irrelevant for this method)
#####   zeroV       indicator if zero velocity used (irrelevant for this method)
#####
##### Output:
#####   R_new       coordinates for the images on the moved path (ndarray of shape 'N_im' x 'D')
#####   V           velocity of the intermediate images (irrelevant for this method -> zeros)
#####
def step_simple(R, F_R, param_step=0.01, F_R_old=0, V_old=0, zeroV=1):
    K_step = param_step
    N_im = R.shape[0]
    D = R.shape[1]
    R_new = np.zeros((N_im,D))
    R_new[0,:] = R[0,:]
    R_new[-1,:] = R[-1,:]
    R_new[1:-1,:] = R[1:-1,:] + K_step*F_R
    V = np.zeros((N_im-2,D))
    return R_new, V


##### This is function moves the path one step along the NEB force according
##### to the quick-min optimizer implemented using the Velocity Verlet algorithm.
#####
##### Input:
#####   R           coordinates for the images on the current path (ndarray of shape 'N_im' x 'D')
#####   F_R         NEB force on the intermediate images on the current path (ndarray of shape 'N_im'-2 x 'D')
#####   param_step  time step dt (scalar)
#####   F_R_old     NEB force on the intermediate images on the previous path (ndarray of shape 'N_im'-2 x 'D')
#####   V_old       velocity of the intermediate images (given as output of the previous step) (ndarray of shape 'N_im'-2 x 'D')
#####   zeroV       indicator if zero velocity used (for the first iteration)
#####
##### Output:
#####   R_new       coordinates for the images on the moved path (ndarray of shape 'N_im' x 'D')
#####   V           velocity of the intermediate images (ndarray of shape 'N_im'-2 x 'D')
#####
def step_QMVelocityVerlet(R, F_R, param_step=0.01, F_R_old=0, V_old=0, zeroV=1):
    dt = param_step
    N_im = R.shape[0]
    D = R.shape[1]
    if zeroV > 0:
        V = np.zeros((N_im-2,D))
    else:
        V = V_old + dt/2*(F_R_old + F_R)
    for im in range(N_im-2):
        normF_R = np.sqrt(np.sum(np.square(F_R),1))
        P = V[im,:].dot(F_R[im,:])/normF_R[im]
        V[im,:] = P*F_R[im,:]/normF_R[im]
        if P < 0:
            V[im,:] = 0
    R_new = np.zeros((N_im,D))
    R_new[0,:] = R[0,:]
    R_new[-1,:] = R[-1,:]
    R_new[1:-1,:] = R[1:-1,:] + dt*V + np.square(dt)/2*F_R
    return R_new, V


##### This function defines the "virtual Hessian" points, i.e., additional
##### observation coordinates around both minimum points. The points are
##### located at a distance 'epsilon' from the minimum point towards each
##### coordinate axis.
#####
##### Input:   R        coordinates of the images on the path (ndarray of shape 'N_im' x 'D')
#####          epsilon  distance from the minimum points (scalar)
#####
##### Output:  Rh       coordinates of the "virtual Hessian" points (ndarray of shape 2*'D' x 'D')
def get_hessian_points(R,epsilon):
    N_im = R.shape[0]
    D = R.shape[1]
    Rh = np.zeros((2*D,D))
    for d in range(D):
        Rh[d,:] = R[0,:]
        Rh[D+d,:] = R[-1,:]
        Rh[d,d] = Rh[d,d] + epsilon
        Rh[D+d,d] = Rh[D+d,d] + epsilon
    return Rh


##### QUATERNION FEATURE: NOT IMPLEMENTED !!!
#####
##### Input:   atoms
#####          target
#####
##### Output:  new_atoms
#####          new_target
def doRotation(atoms,target):
    print('NOTICE: Quaternion feature not implemented!\n')
    new_atoms = atoms
    new_target = target
    return new_atoms,new_target


##### This function gives the distance between two vectors.
#####
##### Input:   x1           coordinates of the moving atoms in configurations C (ndarray of shape 'n1' x 'D')
#####          x2           coordinates of the moving atoms in configurations C' (ndarray of shape 'n2' x 'D')
#####
##### Output:  dist         matrix including the distances between configurations C and C' (ndarray of shape 'n1' x 'n2')
def dist(x1,x2):
    n1 = x1.shape[0]
    n2 = x2.shape[0]
    D = x1.shape[1]
    dist2 = np.zeros((n1,n2))
    for i in range(n1):
        for j in range(n2):
            dist2[i,j] = np.sum((x1[i,:]-x2[j,:])**2)
    dist = np.sqrt(dist2)
    return dist