##### muller_brown.py
##### Copyright: Olli-Pekka Koistinen, Aalto University, 13.7.2017
#####
##### This function gives the Muller-Brown potential and gradient in N_im
##### 2-dimensional locations.
##### 
##### Input:   R    ndarray (shape N_im x 2) including the coordinates of N_im locations
##### Output:  E_R  ndarray (shape N_im x 1) including the potential in the N_im locations
#####          G_R  ndarray (shape N_im x 2) including the gradient for the N_im locations

import numpy as np

def muller_brown(R):
    N_R = R.shape[0]
    E_R = np.zeros([N_R,1])
    G_R = np.zeros([N_R,2])
    c1 = -0.2
    c2 = -0.1
    c3 = -0.17
    c4 = 0.015
    F1 = np.array([[-2.0,0.0],[0.0,-20.0]])
    F2 = np.array([[-2.0,0.0],[0.0,-20.0]])
    F3 = np.array([[-13.0,11.0],[11.0,-13.0]])
    F4 = np.array([[7.0/5.0,3.0/5.0],[3.0/5.0,7.0/5.0]])
    q1 = np.array([[1.0],[0.0]])
    q2 = np.array([[0.0],[1.0/2.0]])
    q3 = np.array([[-1.0/2.0],[3.0/2.0]])
    q4 = np.array([[-1.0],[1.0]])
    for i in range(N_R):
        E1 = c1*np.exp(0.5*(R[i,:]-q1.T).dot(F1.dot((R[i,:][np.newaxis].T-q1))))[0]
        E2 = c2*np.exp(0.5*(R[i,:]-q2.T).dot(F2.dot((R[i,:][np.newaxis].T-q2))))[0]
        E3 = c3*np.exp(0.5*(R[i,:]-q3.T).dot(F3.dot((R[i,:][np.newaxis].T-q3))))[0]
        E4 = c4*np.exp(0.5*(R[i,:]-q4.T).dot(F4.dot((R[i,:][np.newaxis].T-q4))))[0]
        G1_1 = E1*np.array([F1[0,0],(F1[0,1]+F1[1,0])/2.0]).dot(R[i,:][np.newaxis].T-q1)[0]
        G1_2 = E1*np.array([(F1[0,1]+F1[1,0])/2.0,F1[1,1]]).dot(R[i,:][np.newaxis].T-q1)[0]
        G2_1 = E2*np.array([F2[0,0],(F2[0,1]+F2[1,0])/2.0]).dot(R[i,:][np.newaxis].T-q2)[0]
        G2_2 = E2*np.array([(F2[0,1]+F2[1,0])/2.0,F2[1,1]]).dot(R[i,:][np.newaxis].T-q2)[0]
        G3_1 = E3*np.array([F3[0,0],(F3[0,1]+F3[1,0])/2.0]).dot(R[i,:][np.newaxis].T-q3)[0]
        G3_2 = E3*np.array([(F3[0,1]+F3[1,0])/2.0,F3[1,1]]).dot(R[i,:][np.newaxis].T-q3)[0]
        G4_1 = E4*np.array([F4[0,0],(F4[0,1]+F4[1,0])/2.0]).dot(R[i,:][np.newaxis].T-q4)[0]
        G4_2 = E4*np.array([(F4[0,1]+F4[1,0])/2.0,F4[1,1]]).dot(R[i,:][np.newaxis].T-q4)[0]
        E_R[i,0] = E1 + E2 + E3 + E4
        G_R[i,0] = G1_1 + G2_1 + G3_1 + G4_1
        G_R[i,1] = G1_2 + G2_2 + G3_2 + G4_2
    return E_R, G_R
