##### demo_muller_brown_GP_NEB_AIE2.py
##### Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
#####
##### This script shows how to use 'GP_NEB_AIE2.py' in a Muller-Brown example.

import numpy as np
import matplotlib.pyplot as plt
import utils
import GP_NEB_AIE2
import muller_brown
import GPy
import paramz
import pdb

pot_general = muller_brown.muller_brown # define the potential energy function

min1 = np.array([[-0.5582,1.4417]]) # define the first minimum point
min2 = np.array([[0.6235,0.0280]]) # define the second minimum point

N_im = 10 # define the number of images on the path
R_init = utils.initialize_path_linear(min1,min2,N_im) # define the initial path
method_step = utils.step_QMVelocityVerlet # define the step method (e.g., "qmVV" or "simple")
param_step = 0.1 # define parameters for the step method (time step in case of qmVV)
method_force = utils.force_NEB2 # use regular NEB force
param_force = 1.0 # define parallel spring constant
#method_force = utils.force_sNEB2 # use sNEB force
#param_force = np.array([[1.0,1.0]]) # define parallel and perpendicular spring constants

# 'T_MEP' defines the final convergence threshold for 'maxmaxG_R_perp',
# which is the accurate maximum component of gradient perpendicular to
# the path tangent at any of the 'N_im'-2 intermediate images
# (i.e., the algorithm is stopped when the maximum component of perpendicular gradient is below 'T_MEP' for all images).
T_MEP = 0.001

# 'T_CI' defines an additional final convergence threshold for 'maxG_CI',
# which is the accurate maximum component of gradient at the climbing image,
# if the climbing image option is used.
T_CI = 0.001

# 'T_CIon_gp' defines a preliminary GP convergence threshold for turning
# the climbing image mode on during each relaxation phase:
# When the estimated 'maxmaxF_R', which is the maximum component of
# NEB force at any of the intermediate images, is below 'T_CIon_gp',
# the climbing image mode is turned on.
# If you don't want to use climbing image at all, set 'T_CIon_gp' to zero.
T_CIon_gp = 0.01

# If 'divisor_T_MEP_gp' is set to zero, the default GP convergence threshold for
# the estimated 'maxmaxF_R' is 1/10 of the lowest final threshold.
# To save inner iterations during the first relaxation phases, one can set
# a positive value for 'divisor_T_MEP_gp', so that the GP convergence threshold will be
# 1/'divisor_T_MEP_gp' of the smallest accurate 'maxG_R_perp' obtained so far on any of the
# 'N_im'-2 intermediate images, but not less than 1/10 of the lowest final threshold.
# If the estimation error is assumed to not decrease more than that during one outer iteration,
# there is no need for more accurate relaxation on an estimated surface.
divisor_T_MEP_gp = 0.0

# 'disp_max' defines the maximum displacement of image from the nearest observed data point
# relative to the length of the initial path.
# Thus, the last inner step is rejected and the relaxation phase stopped if, for any image, the distance
# to the nearest observed data point is larger than 'disp_max' times the length of the initial path.
disp_max = 0.5

# 'num_bigiter_initpath' defines the number of outer iterations started from the initial path 'R_init'
# - Until 'num_bigiter_initpath' is reached, each relaxation phase is started from the initial path 'R_init'.
#     (If climbing image is used, the CI phase is continued from the "preliminarily converged" evenly spaced path.)
# - After that, each relaxation phase is started from the latest converged path.
#     (If climbing image is used, each relaxation phase is started from the latest "preliminarily converged" evenly spaced path,
#      and the CI phase started from the latest converged CI-path if CI is unchanged
#      (otherwise continued from the current "preliminarily converged" evenly spaced path).)
# Starting each round from the initial path may improve stability (and decrease outer iterations),
# but starting from the latest path may decrease the amount of inner iterations during the relaxation phases.
num_bigiter_initpath = np.inf

# 'num_bigiter_initparam' defines the number of outer iterations where the hyperparameter
# optimization is started from values initialized based on the range of current data.
# After that, the optimization is started from the values of the previous round.
num_bigiter_initparam = np.inf

num_bigiter = 300 # define the maximum number of outer iterations (new sets of observations)
num_iter = 10000 # define the maximum number of inner iterations (steps during a relaxation phase)

# 'islarge_num_iter' indicates if 'num_iter' is assumed to be much larger than required
# for NEB convergence on accurate energy surface. If not (0), the next relaxation phase is
# continued from the current path in case 'num_iter' is reached.
islarge_num_iter = 1

# 'num_bigiter_hess' defines the number of outer iterations using the "virtual Hessian",
# i.e., additional observations around the minimum points. The "virtual Hessian"
# may slow down the GP computations especially in high-dimensional cases,
# but they may give useful information in the beginning.
# They usually don't bring gain after 4 outer iterations (but in some cases do).
# By setting 'num_bigiter_hess' to zero, the "virtual Hessian" is set off.
num_bigiter_hess = np.inf
eps_hess = 0.001 # defines the distance of the additional points from the minimum points

load_file = '' # start normally from beginning
save_file = '' # no saves after each outer iteration

quatern = 0 # no quaternion trick used

# 'visualize' indicates if the true energy along the path is visualized (1) after each relaxation phase or not (0).
# These visualizations require large amount of extra evaluations, so this option is not meant to be used in real applications.
visualize = 1

# Call the GP-NEB algorithm
R,E_R,G_R,i_CI,gp_model,R_all,E_all,G_all,Elevel,obs_at,E_R_acc,E_R_gp, \
maxG_R_perp_acc,maxF_R_gp,maxG_CI_acc,maxG_CI_gp,param_gp,figs = \
GP_NEB_AIE2.GP_NEB_AIE2(pot_general=pot_general,R_init=R_init,method_step=method_step,param_step=param_step, \
method_force=method_force,param_force=param_force,T_MEP=T_MEP,T_CI=T_CI,T_CIon_gp=T_CIon_gp, \
divisor_T_MEP_gp=divisor_T_MEP_gp,disp_max=disp_max,num_bigiter_initpath=num_bigiter_initpath, \
num_bigiter_initparam=num_bigiter_initparam,num_bigiter=num_bigiter,num_iter=num_iter, islarge_num_iter=islarge_num_iter, \
num_bigiter_hess=num_bigiter_hess,eps_hess=eps_hess,load_file=load_file,save_file=save_file,quatern=quatern,visualize=visualize)

# Plot the behaviour
fig = plt.figure()
sub1 = fig.add_subplot(121)
sub1.set_title('Maximum component of NEB force on one image (GP estimate)')
sub1.plot(range(1,maxF_R_gp.shape[1]+1),np.max(maxF_R_gp,0),label='Max',color='r')
sub1.plot(range(1,maxF_R_gp.shape[1]+1),np.min(maxF_R_gp,0),label='Min',color='b')
sub1.plot(range(1,maxG_CI_gp.shape[0]+1),maxG_CI_gp,label='CI',color='g')
sub1.plot(obs_at,np.max(maxG_R_perp_acc,0),'ro')
sub1.plot(obs_at,np.min(maxG_R_perp_acc,0),'bo')
sub1.plot(obs_at,maxG_CI_acc,'go')
sub1.set_xlabel('iteration')
sub1.legend()
sub2 = fig.add_subplot(122)
sub2.set_title('Maximum energy over the images (GP estimate)')
sub2.plot(range(1,E_R_gp.shape[1]+1),np.max(E_R_gp,0),color='b')
sub2.plot(obs_at,np.max(E_R_acc,0),'bo')
sub2.set_xlabel('iteration')
plt.show()
