##### demo_CuH2_atomic_GP_dimer.m
##### Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
#####
##### This script shows how to use 'atomic_GP_dimer.py' in a CuH2 example.

import numpy as np
import matplotlib.pyplot as plt
import potential_CuH2
import utils_atomic
import utils_dimer
import atomic_GP_dimer
import GPy
import paramz

# 'pot_general' gives the potential energy and its gradient vector as a function of the atomic configuration.
# Each row of 'R' represents one configuration including the coordinates of the moving atoms:
# [x_1,y_1,z_1,x_2,y_2,z_2,...]
conf = potential_CuH2.conf_CuH2()
pot_general = lambda R: potential_CuH2.ECuH2(R,conf)

R_init = np.array([[9.0,10.0,7.9,7.6,10.0,7.9]]) # define the initial middle point of the dimer
orient_init = np.array([[1.0,0.0,0.0,0.0,0.0,0.0]]) # define the initial orientation of the dimer (unit vector along the direction of the dimer)
method_rot = utils_dimer.rot_iter_lbfgsext # use the L-BFGS method for rotations (with extrapolation of G1)
method_trans = utils_dimer.trans_iter_lbfgs # use the L-BFGS method for translations
param_trans = np.array([[0.1, 0.1]]) # define a step length for convex regions and maximum step length

D = R_init.shape[1]
N_mov = D/3 # number of moving atoms
E_init = np.ndarray(shape=(0,1)) # energy at initial middle point not observed
G_init = np.ndarray(shape=(0,D)) # gradient at initial middle point not observed
R_all_init = np.ndarray(shape=(0,D)) # no initial data points
E_all_init = np.ndarray(shape=(0,1)) # no initial data
G_all_init = np.ndarray(shape=(0,D)) # no initial data
dimer_sep = 0.01 # define the dimer separation (distance from the middle point of the dimer to the two images)

# 'conf_info' is a dictionary including information about the configurations necessary for the GP model:
conf_info = {}
# 'conf_info['conf_fro']': coordinates of active frozen atoms (ndarray of shape 'N_fro' x 3)
# In the beginning, list none of the frozen atoms as active:
conf_info['conf_fro'] = np.empty((0,3))
# 'conf_info['atomtype_mov']': atomtype indices for moving atoms (ndarray of shape 'N_mov')
conf_info['atomtype_mov'] = np.zeros(N_mov,dtype='int64') # H atoms
# 'conf_info['atomtype_fro']': atomtype indices for active frozen atoms (ndarray of shape 'N_fro')
conf_info['atomtype_fro'] = np.empty(0,dtype='int64')
# The atomtypes must be indexed as 0,1,2,...,'n_at'-1.
n_at = 2 # number of atomtypes (including also the types of inactive frozen atoms)
# 'conf_info['pairtype']': pairtype indices for pairs of atomtypes (ndarray of shape 'n_at' x 'n_at')
# Active pairtypes are indexed as 0,1,2,...,'n_pt'-1. Inactive pairtypes are given index -1.
conf_info['pairtype'] = -np.ones((n_at,n_at),dtype='int64')
# 'conf_info['n_pt']': number of active pairtypes
conf_info['n_pt'] = 0
# Set pairtype indices for moving+moving atom pairs (and update number of active pairtypes):
conf_info['pairtype'],conf_info['n_pt'] = utils_atomic.set_pairtype_mov(conf_info['atomtype_mov'],conf_info['pairtype'],conf_info['n_pt'])

# 'conf_info_inactive' is a dictionary including information about inactive frozen atoms:
conf_info_inactive = {}
# 'conf_info_inactive['conf_ifro']': coordinates of inactive frozen atoms (ndarray of shape 'N_ifro' x 3)
# In the beginning, list all frozen atoms as inactive:
conf_info_inactive['conf_ifro'] = conf[0:216,0:3]
# 'conf_info_inactive['atomtype_ifro']': atomtype indices for inactive frozen atoms (ndarray of shape 'N_ifro')
conf_info_inactive['atomtype_ifro'] = np.ones(conf_info_inactive['conf_ifro'].shape[0],dtype='int64') # Cu atoms

# Only active frozen atoms are taken into account in the covariance function.
# A frozen atom is activated, when it is within the radius of 'actdist_fro'
# from some moving atom.
# Once a frozen atom is activated, it stays active from then on.
# If 'actdist_fro' is set to infinity, all frozen atoms are taken into account.
# When a frozen atom is activated, its coordinates and atomtype index are added
# to 'conf_info['conf_fro']' and 'conf_info['atomtype_fro']', respectively,
# and removed from 'conf_info_inactive['conf_fro']' and 'conf_info_inactive['atomtype_fro']'.
# If the new frozen atom activates new pairtypes, also 'conf_info['pairtype']'
# and 'conf_info['n_pt']' are updated.
actdist_fro = 5
# Activate frozen atoms within activation distance:
utils_atomic.update_active_fro(conf_info,conf_info_inactive,np.vstack((R_init,R_all_init)),actdist_fro)

# 'eval_image1' indicates if image 1 of the dimer is evaluted (1) or not (0)
# after each relaxation phase in addition to the middle point of the dimer:
eval_image1 = 0

# 'T_dimer' defines the final convergence threshold for 'maxF_R', which is
# the maximum component of the force acting on the middle point of the dimer (i.e., the
# algorithm is stopped when all components of the accurate force are below 'T_dimer'):
T_dimer = 0.01

# 'initrot_nogp' indicates if the initial rotations are performed without GP (1) or with GP (0):
initrot_nogp = 0

# 'T_anglerot_init' defines a convergence threshold for the rotation angle
# in the initial rotations performed in the beginning of the algorithm
# (the dimer is not rotated when the estimated rotation angle is less than this):
T_anglerot_init = 0.0873

# 'num_iter_initrot' defines the maximum number of initial rotations (0 if initial rotations skipped):
num_iter_initrot = D

# 'inittrans_nogp' is an indicator if an initial test translation step is taken without GP (1)
# or if GP is used right after initial rotations (0):
inittrans_nogp = 0

# 'T_anglerot_gp' defines a convergence threshold for the rotation angle
# during a relaxation phase (the dimer is not rotated when the estimated
# rotation angle is less than this):
T_anglerot_gp = 0.01

# 'num_iter_rot_gp' defines a maximum number of rotation iterations per
# translation during a relaxation phase:
num_iter_rot_gp = 10

# If 'divisor_T_dimer_gp' is set to zero, the default convergence threshold
# for each relaxation phase for the approximated 'maxF_R' on the
# approximated energy surface is 1/10 of the 'T_dimer'. To save inner
# iterations during the first relaxation phases, one can set a positive
# value for 'divisor_T_dimer_gp', so that the GP convergence threshold will
# be 1/'divisor_T_dimer_gp' of the smallest accurate 'maxF_R' obtained so
# far, but not less than 1/10 of the 'T_dimer'. If the approximation error
# is assumed to not decrease more than that during one outer iteration,
# there is no need for more accurate relaxation on an approximated surface:
divisor_T_dimer_gp = 10.0

# 'disp_max' defines the maximum displacement of the middle point of the
# dimer from the nearest observed data point. Thus, the last inner step is
# rejected and the relaxation phase stopped, if the distance to the nearest
# observed data point is larger than 'disp_max':
disp_max = 0.5

# 'ratio_at_limit' defines the limit for the ratio (< 1) of inter-atomic
# distances between image and its "nearest" observed data point.
# More precisely, the last inner step is rejected and the relaxation phase
# stopped if the following does not hold:
# There is an observed data point so that all inter-atomic distances of the
# current image are more than 'ratio_at_limit' (by default 2/3) but less
# than 1/'ratio_at_limit' (3/2) times the corresponding inter-atomic
# distance of the observed data point.
ratio_at_limit = 2.0/3.0

# 'num_bigiter_initloc' defines the number of outer iterations started from the initial location 'R_init'.
# After that, each relaxation phase is started from the latest converged dimer.
# Starting each round from the initial location may improve stability (and decrease outer iterations),
# but starting from the latest dimer may decrease the number of inner iterations during the relaxation phases:
num_bigiter_initloc = np.inf

# 'num_bigiter_initparam' defines the number of outer iterations where the hyperparameter
# optimization is started from values initialized based on the range of current data.
# After that, the optimization is started from the values of the previous round.
num_bigiter_initparam = np.inf

num_bigiter = 300 # define the maximum number of outer iterations (new sets of observations)
num_iter = 10000 # define the maximum number of inner iterations (steps during a relaxation phase)

# 'islarge_num_iter' indicates if 'num_iter' is assumed to be much larger than required
# for dimer convergence on accurate energy surface. If not (0), the next relaxation phase is
# continued from the current dimer in case 'num_iter' is reached:
islarge_num_iter = 1

load_file = '' # start normally from beginning
save_file = '' # no saves after each outer iteration

# Call the atomic GP-dimer function
R, orient, E_R, G_R, gp_model, R_all, E_all, G_all, obs_at, E_R_acc, E_R_gp, maxF_R_acc, maxF_R_gp, param_gp_initrot, \
param_gp, obs_initrot, obs_total, num_esmax, num_es1, num_es2 = \
atomic_GP_dimer.atomic_GP_dimer(pot_general=pot_general, conf_info=conf_info, conf_info_inactive=conf_info_inactive, \
actdist_fro=actdist_fro, R_init=R_init, orient_init=orient_init, method_rot=method_rot, \
method_trans=method_trans, param_trans=param_trans, E_init=E_init, G_init=G_init, R_all_init=R_all_init, \
E_all_init=E_all_init, G_all_init=G_all_init, dimer_sep=dimer_sep, eval_image1=eval_image1, T_dimer=T_dimer, \
initrot_nogp=initrot_nogp, T_anglerot_init=T_anglerot_init, num_iter_initrot=num_iter_initrot, \
inittrans_nogp=inittrans_nogp, T_anglerot_gp=T_anglerot_gp, num_iter_rot_gp=num_iter_rot_gp, \
divisor_T_dimer_gp=divisor_T_dimer_gp, disp_max=disp_max, ratio_at_limit=ratio_at_limit, \
num_bigiter_initloc=num_bigiter_initloc, num_bigiter_initparam=num_bigiter_initparam, num_bigiter=num_bigiter, \
num_iter=num_iter, islarge_num_iter=islarge_num_iter, load_file=load_file, save_file=save_file)

fig = plt.figure()

sub1 = fig.add_subplot(211)
sub1.set_title('Maximum component of force on the middle point')
sub1.plot(range(maxF_R_gp.shape[0]),maxF_R_gp)
sub1.plot(obs_at,maxF_R_acc,'o')
sub1.legend(('GP approximation','Accurate evaluation'))

sub2 = fig.add_subplot(212)
sub2.set_title('Energy at the middle point')
sub2.plot(range(E_R_gp.shape[0]),E_R_gp)
sub2.plot(obs_at,E_R_acc,'o')
sub2.set_xlabel('iteration')
sub1.legend(('GP approximation','Accurate evaluation'))

plt.show()