##### demo_CuH2_atomic_GP_NEB_OIE.py
##### Copyright: Olli-Pekka Koistinen, Aalto University, 23.12.2018
#####
##### This script shows how to use 'atomic_GP_NEB_OIE.py' in a CuH2 example.

import numpy as np
import matplotlib.pyplot as plt
import potential_CuH2
import utils
import utils_atomic
import atomic_GP_NEB_OIE
import GPy
import paramz
import pdb

# 'pot_general' gives the potential energy and its gradient vector as a function of the atomic configuration.
# Each row of 'R' represents one configuration including the coordinates of the moving atoms:
# [x_1,y_1,z_1,x_2,y_2,z_2,...]
conf = potential_CuH2.conf_CuH2()
pot_general = lambda R: potential_CuH2.ECuH2(R,conf)
min1 = np.array([[8.6822,9.9470,11.7330,7.9422,9.9470,11.7330]]) # define the first minimum point
min2 = np.array([[9.8914,9.9470,7.7599,6.7330,9.9470,7.7599]]) # define the second minimum point
N_mov = int(min1.shape[1]/3) # number of moving atoms

# 'conf_info' is a dictionary including information about the configurations necessary for the GP model:
conf_info = {}
# 'conf_info['conf_fro']': coordinates of active frozen atoms (ndarray of shape 'N_fro' x 3)
# In the beginning, list none of the frozen atoms as active:
conf_info['conf_fro'] = np.empty((0,3))
# 'conf_info['atomtype_mov']': atomtype indices for moving atoms (ndarray of shape 'N_mov')
conf_info['atomtype_mov'] = np.zeros(N_mov,dtype='int64') # H atoms
# 'conf_info['atomtype_fro']': atomtype indices for active frozen atoms (ndarray of shape 'N_fro')
conf_info['atomtype_fro'] = np.empty(0,dtype='int64')
# The atomtypes must be indexed as 0,1,2,...,'n_at'-1.
n_at = 2 # number of atomtypes (including also the types of inactive frozen atoms)
# 'conf_info['pairtype']': pairtype indices for pairs of atomtypes (ndarray of shape 'n_at' x 'n_at')
# Active pairtypes are indexed as 0,1,2,...,'n_pt'-1. Inactive pairtypes are given index -1.
conf_info['pairtype'] = -np.ones((n_at,n_at),dtype='int64')
# 'conf_info['n_pt']': number of active pairtypes
conf_info['n_pt'] = 0
# Set pairtype indices for moving+moving atom pairs (and update number of active pairtypes):
conf_info['pairtype'],conf_info['n_pt'] = utils_atomic.set_pairtype_mov(conf_info['atomtype_mov'],conf_info['pairtype'],conf_info['n_pt'])

# 'conf_info_inactive' is a dictionary including information about inactive frozen atoms:
conf_info_inactive = {}
# 'conf_info_inactive['conf_ifro']': coordinates of inactive frozen atoms (ndarray of shape 'N_ifro' x 3)
# In the beginning, list all frozen atoms as inactive:
conf_info_inactive['conf_ifro'] = conf[0:216,0:3]
# 'conf_info_inactive['atomtype_ifro']': atomtype indices for inactive frozen atoms (ndarray of shape 'N_ifro')
conf_info_inactive['atomtype_ifro'] = np.ones(conf_info_inactive['conf_ifro'].shape[0],dtype='int64') # Cu atoms

# Only active frozen atoms are taken into account in the covariance function.
# A frozen atom is activated, when it is within the radius of 'actdist_fro'
# from some moving atom in some configuration on the path.
# Once a frozen atom is activated, it stays active from then on.
# If 'actdist_fro' is set to infinity, all frozen atoms are taken into account.
# When a frozen atom is activated, its coordinates and atomtype index are added
# to 'conf_info['conf_fro']' and 'conf_info['atomtype_fro']', respectively,
# and removed from 'conf_info_inactive['conf_fro']' and 'conf_info_inactive['atomtype_fro']'.
# If the new frozen atom activates new pairtypes, also 'conf_info['pairtype']'
# and 'conf_info['n_pt']' are updated.
actdist_fro = 5
# Activate frozen atoms within activation distance:
utils_atomic.update_active_fro(conf_info,conf_info_inactive,np.vstack((min1,min2)),actdist_fro)

N_im = 10 # define the number of images on the path
R_init = utils.initialize_path_linear(min1,min2,N_im) # define the initial path
method_step = utils.step_QMVelocityVerlet # define the step method (e.g., "qmVV" or "simple")
param_step = 0.1 # define parameters for the step method (time step in case of qmVV)
k_par = 1.0 # define the parallel spring constant

# 'T_MEP' defines the final convergence threshold for the 'maxnormF', which is
# the maximum of the accurate norms of the NEB forces acting on the 'N_im'-2 intermediate
# images (i.e., the algorithm is stopped when the accurate NEB force is below 'T_MEP' for all images).
T_MEP = 0.3

# 'T_CI' defines an additional final convergence threshold for the
# climbing image, if the climbing image option is used.
# If you don't want to use a tighter convergence threshold for the climbing
# image, set 'T_CI' equal to 'T_MEP' (or larger, because
# the general threshold 'T_MEP' concerns also the climbing image).
T_CI = 0.01

# 'T_CIon_gp' defines a preliminary convergence threshold for each relaxation phase:
# When the approximated 'maxnormF' is below 'T_CIon_gp', the climbing
# image mode is turned on.
# If you don't want to use climbing image at all, set 'T_CIon_gp' to zero.
T_CIon_gp = 1.0

# If 'divisor_T_MEP_gp' is set to zero, the default convergence threshold for each relaxation
# phase for the approximated 'maxnormF' on the approximated energy surface is 1/10 of the lowest final threshold.
# To save inner iterations during the first relaxation phases, one can set
# a positive value for 'divisor_T_MEP_gp', so that the GP convergence threshold will be
# 1/'divisor_T_MEP_gp' of the smallest accurate norm of NEB force obtained so far
# on any of the 'N_im'-2 intermediate images, but not less than 1/10 of the lowest final threshold.
# If the approximation error is assumed to not decrease more than that during one outer iteration,
# there is no need for more accurate relaxation on an approximated surface.
divisor_T_MEP_gp = 10.0

# 'disp_max' defines the maximum displacement of image from the nearest evaluated image
# relative to the length of the initial path.
# Thus, the last inner step is rejected and the relaxation phase stopped if, for any image, the distance
# to the nearest evaluated image is larger than 'disp_max' times the length of the initial path.
disp_max = 0.5

# 'ratio_at_limit' defines the limit for the ratio (< 1) of inter-atomic distances between image
# and its "nearest" observed data point.
# More precisely, the last inner step is rejected and the relaxation phase stopped if the following
# does not hold for some of the current images:
# There is an observed data point so that all inter-atomic distances of the current image are more
# than 'ratio_at_limit' (by default 2/3) but less than 1/'ratio_at_limit' (3/2) times the corresponding
# inter-atomic distance of the observed data point.
ratio_at_limit = 2.0/3.0

# 'num_bigiter_init' defines the number of outer iterations started from the initial path 'R_init'
# - Until 'num_bigiter_init' is reached, each relaxation phase is started from the initial path 'R_init'.
#     (If climbing image is used, the CI phase is continued from the "preliminarily converged" equally spaced path.)
# - After that, each relaxation phase is started from the latest converged path.
#     (If climbing image is used, each relaxation phase is started from the latest "preliminarily converged" equally spaced path,
#      and the CI phase started from the latest converged CI-path if CI is unchanged (otherwise from the current
#      "preliminarily converged" evenly spaced path).)
# Starting each round from the initial path may improve stability (and decrease outer iterations),
# but starting from the latest path may decrease the amount of inner iterations during the relaxation phases.
num_bigiter_init = np.inf

num_bigiter = 300 # define the maximum number of outer iterations (new sets of observations)
num_iter = 10000 # define the maximum number of inner iterations (steps during a relaxation phase)

# 'num_bigiter_hess' defines the number of outer iterations using the "virtual Hessian",
# i.e., additional observations around the minimum points. The "virtual Hessian"
# may slow down the GP computations especially in high-dimensional cases,
# but they may give useful information in the beginning.
# They usually don't bring gain after 4 outer iterations (but in some cases do).
# By setting 'num_bigiter_hess' to zero, the "virtual Hessian" is set off.
num_bigiter_hess = np.inf
eps_hess = 0.001 # defines the distance of the additional points from the minimum points

# 'visualize' indicates if the true energy along the path is visualized (1) after each relaxation phase or not (0).
# These visualizations require large amount of extra evaluations, so this option is not meant to be used in real applications.
visualize = 1

# Call the GP-NEB algorithm
R,E_R,G_R,i_CI,gp_model,R_all,E_all,G_all,obs_at,E_R_ae,E_R_gp,normF_R_ae,normF_R_gp,normFCI_ae,normFCI_gp,param_gp,figs = \
atomic_GP_NEB_OIE.atomic_GP_NEB_OIE(pot_general=pot_general,conf_info=conf_info, \
conf_info_inactive=conf_info_inactive,actdist_fro=actdist_fro,R_init=R_init,method_step=method_step, \
param_step=param_step,k_par=k_par,T_MEP=T_MEP,T_CI=T_CI,T_CIon_gp=T_CIon_gp, \
divisor_T_MEP_gp=divisor_T_MEP_gp,disp_max=disp_max,ratio_at_limit=ratio_at_limit, \
num_bigiter_init=num_bigiter_init,num_bigiter=num_bigiter,num_iter=num_iter, \
num_bigiter_hess=num_bigiter_hess,eps_hess=eps_hess,visualize=visualize)

# Plot the behaviour
fig = plt.figure()
sub1 = fig.add_subplot(121)
sub1.set_title('Magnitude of the NEB force on one image (GP approximation)')
sub1.plot(range(1,normF_R_gp.shape[1]+1),np.max(normF_R_gp,0),label='Max',color='r')
sub1.plot(range(1,normF_R_gp.shape[1]+1),np.mean(normF_R_gp,0),label='Mean',color='b')
sub1.plot(range(1,normF_R_gp.shape[1]+1),normFCI_gp,label='CI',color='g')
sub1.plot(obs_at,np.max(normF_R_ae,0),'ro')
sub1.plot(obs_at,np.mean(normF_R_ae,0),'bo')
sub1.plot(obs_at,normFCI_ae,'go')
sub1.set_xlabel('iteration')
sub1.legend()
sub2 = fig.add_subplot(122)
sub2.set_title('Mean energy over the images (GP approximation)')
sub2.plot(range(1,E_R_gp.shape[1]+1),np.mean(E_R_gp,0),color='b')
sub2.plot(obs_at,np.mean(E_R_ae,0),'bo')
sub2.set_xlabel('iteration')
plt.show()

