%%%%% update_active_fro.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 24.1.2018
%%%%%

%%%%% This function activates inactive frozen atoms within the radius of
%%%%% 'actdist_fro' from some moving atom in configurations 'R_new'.
%%%%% These frozen atoms are then taken into account in the covariance function.
%%%%% 
%%%%% When a frozen atoms is activated, its coordinates and atomtype index are added
%%%%% to 'conf_info.conf_fro' and 'conf_info.atomtype_fro', respectively,
%%%%% and removed from 'conf_info_inactive.conf_fro' and 'conf_info_inactive.atomtype_fro'.
%%%%% If the new frozen atom activates new pairtypes, also 'conf_info.pairtype'
%%%%% and 'conf_info.n_pt' are updated. 
%%%%%
%%%%% Input:
%%%%%   conf_info           structure array including information about the configurations necessary for the GP model
%%%%%                       - conf_info.conf_fro: coordinates of active frozen atoms (N_fro x 3)
%%%%%                       - conf_info.atomtype_mov: atomtype indices for moving atoms (1 x N_mov)
%%%%%                       - conf_info.atomtype_fro: pairtype indices for active frozen atoms (1 x N_fro)
%%%%%                       - conf_info.pairtype: pairtype indices for pairs of atomtypes (n_at x n_at)
%%%%%                       - conf_info.n_pt: number of active pairtypes
%%%%%   conf_info_inactive  structure array including information about inactive frozen atoms
%%%%%                       - conf_info_inactive.conf_ifro: coordinates of inactive frozen atoms (N_ifro x 3)
%%%%%                       - conf_info_inactive.atomtype_ifro: atomtype indices for inactive frozen atoms (1 x N_ifro)
%%%%%   R_new               coordinates of moving atoms in the new observations (N_obs x 3*N_mov)
%%%%%   actdist_fro         activation distance for moving+frozen atom pairs (inf when all active)
%%%%%
%%%%% Output:
%%%%%   conf_info           structure array including information about the configurations necessary for the GP model (updated)
%%%%%   conf_info_inactive  structure array including information about inactive frozen atoms (updated)
%%%%%   new_act             1 if new active frozen atoms included, 0 if not

function [conf_info,conf_info_inactive,new_act] = update_active_fro(conf_info,conf_info_inactive,R_new,actdist_fro)
    
    new_act = 0; % indicator of occurrence of new active frozen atoms
    atomtype_active_fro_new = []; % atomtypes of new active frozen atoms
    
    if size(conf_info_inactive.conf_ifro,1) > 0
        
        if actdist_fro < inf
            % Activate inactive frozen atoms within the radius of 'actdist_fro' from some moving atom:
            for i = 1:size(R_new,1)
                dist_ifro_i = sqrt(bsxfun(@minus,conf_info_inactive.conf_ifro(:,1),R_new(i,1:3:end)).^2+bsxfun(@minus,conf_info_inactive.conf_ifro(:,2),R_new(i,2:3:end)).^2+bsxfun(@minus,conf_info_inactive.conf_ifro(:,3),R_new(i,3:3:end)).^2);
                active_fro_new_i = find(any(dist_ifro_i<=actdist_fro,2));
                if not(isempty(active_fro_new_i))
                    new_act = 1;
                    conf_info.conf_fro = [conf_info.conf_fro;conf_info_inactive.conf_ifro(active_fro_new_i,:)];
                    conf_info.atomtype_fro = [conf_info.atomtype_fro,conf_info_inactive.atomtype_ifro(active_fro_new_i)];
                    atomtype_active_fro_new = unique([atomtype_active_fro_new,conf_info_inactive.atomtype_ifro(active_fro_new_i)]);
                    conf_info_inactive.conf_ifro(active_fro_new_i,:) = [];
                    conf_info_inactive.atomtype_ifro(active_fro_new_i) = [];				
                end
            end
        else
            % Activate all inactive frozen atoms:
            new_act = 1;
            conf_info.conf_fro = [conf_info.conf_fro;conf_info_inactive.conf_ifro];
            conf_info.atomtype_fro = [conf_info.atomtype_fro,conf_info_inactive.atomtype_ifro];
            atomtype_active_fro_new = unique(conf_info_inactive.atomtype_ifro);
            conf_info_inactive.conf_ifro = [];
            conf_info_inactive.atomtype_ifro = [];
        end
        
        % Activate new pairtypes if necessary:
        if new_act > 0
            for at_i = unique(conf_info.atomtype_mov)
                for at_j = atomtype_active_fro_new
                    if conf_info.pairtype(at_i,at_j) == 0
                        conf_info.n_pt = conf_info.n_pt + 1;
                        conf_info.pairtype(at_i,at_j) = conf_info.n_pt;
                        conf_info.pairtype(at_j,at_i) = conf_info.n_pt;
                    end
                end
            end
        end
        
    end
    
end

