%%%%% trans_iter_qmvv.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 23.3.2019
%%%%%
%%%%% This is an auxiliary function for the dimer method ('dimer.m').
%%%%% The dimer is translated one step towards saddle point according to the
%%%%% quick-min optimizer implemented using the velocity Verlet algorithm.
%%%%%
%%%%% Input:
%%%%%   R                  coordinates of the middle point of the dimer (1 x D)
%%%%%   orient             unit vector along the direction of the dimer (1 x D)
%%%%%   F_R                force at the middle point of the dimer (1 x D)
%%%%%   Curv               curvature of energy along the direction of the dimer
%%%%%   param_trans        [time step dt, maximum step length]
%%%%%   transinfo          structure array including necessary input information for the translation method
%%%%%                      - transinfo.F_trans_old: translational force of the previous translation iteration (1 x D)
%%%%%                      - transinfo.V_old: velocity of the dimer in the previous translation iteration (1 x D)
%%%%%                      - transinfo.zeroV: indicator if zero velocity used (for the first iteration)
%%%%%
%%%%% Output:
%%%%%   R_new              coordinates of the new middle point of the dimer (1 x D)
%%%%%   R_obs              coordinates of new observed locations [empty for this translation method]
%%%%%   E_obs              energy at new observed locations [empty for this translation method]
%%%%%   G_obs              gradient at new observed locations [empty for this translation method]
%%%%%   transinfo          structure array including necessary input information for the translation method
%%%%%                      - transinfo.F_trans_old: translational force of this translation iteration (1 x D)
%%%%%                      - transinfo.V_old: velocity of the dimer in this translation iteration (1 x D)
%%%%%                      - transinfo.zeroV: indicator if zero velocity used (for the first iteration)

function [R_new,R_obs,E_obs,G_obs,transinfo] = trans_iter_qmvv(R,orient,F_R,Curv,param_trans,transinfo)
    F_trans_old = transinfo.F_trans_old;
    V_old = transinfo.V_old;
    zeroV = transinfo.zeroV;
    dt = param_trans(1,1);
    max_steplength = param_trans(1,2);
    D = size(R,2);
    if Curv < 0
        F_trans = F_R - 2*(F_R*orient')*orient;
    else
        F_trans = -(F_R*orient')*orient;
    end
    if zeroV > 0
        V = zeros(1,D);
    else
        V = V_old + dt/2*(F_trans_old + F_trans);
    end
    normF_trans = sqrt(sum(F_trans.^2,2));
    P = V*F_trans'/normF_trans;
    V = P*F_trans/normF_trans;
    if P < 0
        V = 0;
    end
    step = dt*V+(dt^2)/2*F_trans;
    steplength = sqrt(sum(step.^2,2));
    if steplength > max_steplength
        R_new = R + max_steplength/steplength*step;
        F_trans = 0;
        V = 0;
    else
        R_new = R + step;
    end
    R_obs = [];
    E_obs = [];
    G_obs = [];
    transinfo.F_trans_old = F_trans;
    transinfo.V_old = V;
    transinfo.zeroV = 0;
end
