%%%%% trans_iter_lbfgs.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
%%%%%
%%%%% This is an auxiliary function for the dimer method ('dimer.m').
%%%%% The dimer is translated one step towards saddle point according to the
%%%%% L-BFGS method.
%%%%%
%%%%% Input:
%%%%%   R                 coordinates of the middle point of the dimer (1 x D)
%%%%%   orient            unit vector along the direction of the dimer (1 x D)
%%%%%   F_R               force at the middle point of the dimer (1 x D)
%%%%%   Curv              curvature of energy along the direction of the dimer
%%%%%   param_trans       [predefined step length for convex regions, maximum step length]     
%%%%%   transinfo         structure array including necessary input information for the translation method
%%%%%                     - transinfo.F_trans_old: translational force of the previous translation iteration (1 x D)
%%%%%                     - transinfo.deltaR_mem: change of location in m previous translation iterations (m x D)
%%%%%                     - transinfo.deltaF_mem: change of translational force in m previous translation iterations excluding the last one ((m-1) x D)
%%%%%                     - transinfo.num_lbfgsiter_trans: maximum number of previous translation iterations kept in memory
%%%%%
%%%%% Output:
%%%%%   R_new             coordinates of the new middle point of the dimer (1 x D)
%%%%%   R_obs             coordinates of the new observed location [empty for this translation method]
%%%%%   E_obs             energy at the new observed location [empty for this translation method]
%%%%%   G_obs             gradient at the new observed location [empty for this translation method]
%%%%%   transinfo         structure array including necessary input information for the translation method (updated)
%%%%%                     - transinfo.F_trans_old: translational force of this translation iteration (1 x D)
%%%%%                     - transinfo.deltaR_mem: change of location in m previous translation iterations including this one ((m+1) x D)
%%%%%                     - transinfo.deltaF_mem: change of translational force in m previous translation iterations (m x D)
%%%%%                     - transinfo.num_lbfgsiter_trans: maximum number of previous translation iterations kept in memory


function [R_new,R_obs,E_obs,G_obs,transinfo] = trans_iter_lbfgs(R,orient,F_R,Curv,param_trans,transinfo)
    D = size(R,2);
    F_trans_old = transinfo.F_trans_old;
    deltaR_mem = transinfo.deltaR_mem;
    deltaF_mem = transinfo.deltaF_mem;
    num_lbfgsiter_trans = transinfo.num_lbfgsiter_trans;
    steplength_convex = param_trans(1,1);
    max_steplength = param_trans(1,2);
    m = size(deltaR_mem,1);
    if Curv < 0
        F_trans = F_R - 2*(F_R*orient')*orient;
        if m > 0
            deltaF_mem = [deltaF_mem;F_trans-F_trans_old];
        end
        q = -F_trans';
        a_mem = zeros(m,1);            
        for k = 0:(m-1)
            s = deltaR_mem(m-k,:)';
            y = -deltaF_mem(m-k,:)';
            rho = 1/(y'*s);
            a = rho*s'*q;
            a_mem(m-k,1) = a;
            q = q - a*y;
        end
        if m > 0
            s = deltaR_mem(m,:)';
            y = -deltaF_mem(m,:)';
            scaling = (s'*y)/(y'*y);
        else
            scaling = 0.01;
        end
        r = scaling*eye(D)*q;
        for k = 1:m
            s = deltaR_mem(k,:)';
            y = -deltaF_mem(k,:)';
            rho = 1/(y'*s);
            b = rho*y'*r;
            r = r + s*(a_mem(k,1)-b);
        end
        steplength = sqrt(sum(r.^2));
        if steplength > max_steplength
            r = max_steplength/steplength*r;
            transinfo.F_trans_old = 0;
            transinfo.deltaR_mem = [];
            transinfo.deltaF_mem = [];
        else
            deltaR_mem = [deltaR_mem;-r'];
            if m >= num_lbfgsiter_trans
                deltaR_mem(1,:) = [];
                deltaF_mem(1,:) = [];
            end
            transinfo.F_trans_old = F_trans;
            transinfo.deltaR_mem = deltaR_mem;
            transinfo.deltaF_mem = deltaF_mem;
        end
        R_new = R - r';
    else
        F_trans = -(F_R*orient')*orient;
        orient_search = F_trans/sqrt(sum(F_trans.^2,2));
        R_new = R + steplength_convex*orient_search;
        transinfo.F_trans_old = 0;
        transinfo.deltaR_mem = [];
        transinfo.deltaF_mem = [];
    end    
    R_obs = [];
    E_obs = [];
    G_obs = [];
end
