%%%%% trans_iter_cg.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 23.3.2019
%%%%%
%%%%% This is an auxiliary function for the dimer method ('dimer.m').
%%%%% The dimer is translated one step towards saddle point according to the
%%%%% conjugate gradient method.
%%%%%
%%%%% Input:
%%%%%   R                 coordinates of the middle point of the dimer (1 x D)
%%%%%   orient            unit vector along the direction of the dimer (1 x D)
%%%%%   F_R               force at the middle point of the dimer (1 x D)
%%%%%   Curv              curvature of energy along the direction of the dimer
%%%%%   param_trans       [step length for a small test displacement, predefined step length for convex regions, maximum step length]     
%%%%%   transinfo         structure array including necessary input information for the translation method
%%%%%                     - transinfo.potential: potential and gradient function
%%%%%                     - transinfo.F_trans_old: translational force of the previous translation iteration (1 x D)
%%%%%                     - transinfo.F_modtrans_old: modified translational force of the previous translation iteration (1 x D)
%%%%%                     - transinfo.cgiter_trans: number of conjugated transition iterations
%%%%%                     - transinfo.num_cgiter_trans: maximum number of conjugated transition iterations before resetting the conjugate directions
%%%%%
%%%%% Output:
%%%%%   R_new             coordinates of the new middle point of the dimer (1 x D)
%%%%%   R_obs             coordinates of the new observed location (1 x D)
%%%%%   E_obs             energy at the new observed location
%%%%%   G_obs             gradient at the new observe location (1 x D)
%%%%%   transinfo         structure array including necessary input information for the translation method (updated)
%%%%%                     - transinfo.potential: potential and gradient function
%%%%%                     - transinfo.F_trans_old: translational force of this translation iteration (1 x D)
%%%%%                     - transinfo.F_modtrans_old: modified translational force of this translation iteration (1 x D)
%%%%%                     - transinfo.cgiter_trans: number of conjugated transition iterations
%%%%%                     - transinfo.num_cgiter_trans: maximum number of conjugated transition iterations before resetting the conjugate directions


function [R_new,R_obs,E_obs,G_obs,transinfo] = trans_iter_cg(R,orient,F_R,Curv,param_trans,transinfo)
    potential = transinfo.potential;
    F_trans_old = transinfo.F_trans_old;
    F_modtrans_old = transinfo.F_modtrans_old;
    cgiter_trans = transinfo.cgiter_trans;
    num_cgiter_trans = transinfo.num_cgiter_trans;
    steplength_convex = param_trans(1,1);
    max_steplength = param_trans(1,2);
    if cgiter_trans >= num_cgiter_trans
        F_trans_old = 0;
        F_modtrans_old = 0;
        cgiter_trans = 1;
    else
        cgiter_trans = cgiter_trans + 1;
    end
    if Curv < 0
        F_trans = F_R - 2*(F_R*orient')*orient;
        F_trans_old2 = F_trans_old*F_trans_old';
        if F_trans_old2 == 0
            gamma = 0;
        else
            gamma = ((F_trans-F_trans_old)*F_trans')/F_trans_old2;
            if gamma < 0 || sum((gamma*F_modtrans_old).^2,2) > sum(F_trans.^2,2)
                gamma = 0;
                cgiter_trans = 1;
            end
        end
        F_modtrans = F_trans + gamma*F_modtrans_old;
        orient_search = F_modtrans/sqrt(sum(F_modtrans.^2,2));
        F_0 = F_trans*orient_search';
        dr = 0.5*F_0/abs(Curv);
        R_dr = R + dr*orient_search;
        [E_R_dr,G_R_dr] = potential(R_dr);
        F_trans_dr = -G_R_dr + 2*(G_R_dr*orient')*orient;     
        F_dr = F_trans_dr*orient_search';
        steplength = -F_0*dr/(F_dr-F_0);
        if steplength < 0
            steplength = steplength_convex; 
            F_trans = 0;
            F_modtrans = 0;
            cgiter_trans = 0;
        elseif steplength > max_steplength
            steplength = max_steplength;
            F_trans = 0;
            F_modtrans = 0;
            cgiter_trans = 0;
        end
        R_obs = R_dr;
        E_obs = E_R_dr;
        G_obs = G_R_dr;
    else
        F_trans = -(F_R*orient')*orient;
        orient_search = F_trans/sqrt(sum(F_trans.^2,2));
        steplength = steplength_convex;
        F_trans = 0;
        F_modtrans = 0;
        cgiter_trans = 0;
        R_obs = [];
        E_obs = [];
        G_obs = [];
    end    
    R_new = R + steplength*orient_search;
    transinfo.F_trans_old = F_trans;
    transinfo.F_modtrans_old = F_modtrans;
    transinfo.cgiter_trans = cgiter_trans;
end
