%%%%% rot_iter_lbfgsext.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
%%%%%
%%%%% This is an auxiliary function for the dimer method ('dimer.m').
%%%%% The dimer is rotated one step towards its minimum energy orientation
%%%%% according to the modified Newton method on a rotation plane chosen
%%%%% based on the L-BFGS method (this version gives the estimated gradient
%%%%% at image 1 after the rotation as an output to be used in the next
%%%%% iteration instead of evaluating it).
%%%%%
%%%%% Input:
%%%%%   R                     coordinates of the middle point of the dimer (1 x D)
%%%%%   orient                unit vector along the direction of the dimer (before rotation) (1 x D)
%%%%%   G01                   gradient at the middle point and image 1 of the dimer (before rotation) (2 x D)
%%%%%   potential             potential and gradient function
%%%%%   dimer_sep             dimer separation (distance from the middle point of the dimer to the two images)
%%%%%   T_anglerot            convergence threshold for the rotation angle
%%%%%   estim_Curv            if 1, an estimate for the curvature along the direction of the dimer after the rotation is calculated
%%%%%   rotinfo               structure array including necessary input information for the rotation method
%%%%%                         - rotinfo.F_rot_old: rotational force of the previous rotation iteration (1 x D)
%%%%%                         - rotinfo.deltaR_mem: change of orientation in m previous rotation iterations (m x D)
%%%%%                         - rotinfo.deltaF_mem: change of rotational force in m previous rotation iterations excluding the last one ((m-1) x D)
%%%%%                         - rotinfo.num_lbfgsiter_rot: maximum number of previous rotation iterations kept in memory
%%%%%   
%%%%% Output:
%%%%%   orient_new            unit vector along the direction of the dimer after optimal rotation (1 x D)
%%%%%   Curv                  estimate for the curvature along the direction of the dimer after the rotation (empty if estim_Curv = 0 or no rotation)
%%%%%   R_obs                 coordinates of the new observed location (1 x D)
%%%%%   E_obs                 energy at the new observed location
%%%%%   G_obs                 gradient at the new observed location (1 x D)
%%%%%   rotinfo               structure array including necessary input information for the rotation method (updated)
%%%%%                         - rotinfo.F_rot_old: rotational force of this rotation iteration (1 x D)
%%%%%                         - rotinfo.deltaR_mem: change of orientation in m previous rotation iterations including this one ((m+1) x D)
%%%%%                         - rotinfo.deltaF_mem: change of rotational force in m previous rotation iterations (m x D)
%%%%%                         - rotinfo.num_lbfgsiter_rot: maximum number of previous rotation iterations kept in memory
%%%%%                         - rotinfo.G1: estimated gradient at image 1 after the rotation (1 x D)


function [orient_new,Curv,R_obs,E_obs,G_obs,rotinfo] = rot_iter_lbfgsext(R,orient,G01,potential,dimer_sep,T_anglerot,estim_Curv,rotinfo)
    D = size(R,2);
    F_rot_old = rotinfo.F_rot_old;
    deltaR_mem = rotinfo.deltaR_mem;
    deltaF_mem = rotinfo.deltaF_mem;
    num_lbfgsiter_rot = rotinfo.num_lbfgsiter_rot;
    m = size(deltaR_mem,1);
    F_rot = force_rot(G01,orient,dimer_sep);
    if m > 0
        deltaF_mem = [deltaF_mem;F_rot-F_rot_old];
    end
    q = -F_rot';
    a_mem = zeros(m,1);            
    for k = 0:(m-1)
        s = deltaR_mem(m-k,:)';
        y = -deltaF_mem(m-k,:)';
        rho = 1/(y'*s);
        a = rho*s'*q;
        a_mem(m-k,1) = a;
        q = q - a*y;
    end
    if m > 0
        s = deltaR_mem(m,:)';
        y = -deltaF_mem(m,:)';
        scaling = (s'*y)/(y'*y);
    else
        scaling = 0.01;
    end
    r = scaling*eye(D)*q;
    for k = 1:m
        s = deltaR_mem(k,:)';
        y = -deltaF_mem(k,:)';
        rho = 1/(y'*s);
        b = rho*y'*r;
        r = r + s*(a_mem(k,1)-b);
    end
    orient_rot = r'-(orient*r)*orient;
    orient_rot = orient_rot/sqrt(sum(orient_rot.^2));
    F_rot_oriented = (F_rot*orient_rot')*orient_rot;
    [orient_new,~,Curv,G1,R_obs,E_obs,G_obs] = rotate_dimer(R,orient,G01,F_rot_oriented,potential,dimer_sep,T_anglerot,estim_Curv,1);
    if isempty(R_obs)
        F_rot = 0;
        deltaR_mem = [];
        deltaF_mem = [];
    else
        deltaR_mem = [deltaR_mem;orient_new-orient];
        if m >= num_lbfgsiter_rot
            deltaR_mem(1,:) = [];
            deltaF_mem(1,:) = [];
        end
    end
    rotinfo.F_rot_old = F_rot;
    rotinfo.deltaR_mem = deltaR_mem;
    rotinfo.deltaF_mem = deltaF_mem;
    rotinfo.G1 = G1;
end
