%%%%% rot_iter_cg.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
%%%%%
%%%%% This is an auxiliary function for the dimer method ('dimer.m').
%%%%% The dimer is rotated one step towards its minimum energy orientation
%%%%% according to the modified Newton method on a rotation plane chosen
%%%%% based on a conjugate gradient method.
%%%%%
%%%%% Input:
%%%%%   R                     coordinates of the middle point of the dimer (1 x D)
%%%%%   orient                unit vector along the direction of the dimer (before rotation) (1 x D)
%%%%%   G01                   gradient at the middle point and image 1 of the dimer (before rotation) (2 x D)
%%%%%   potential             potential and gradient function
%%%%%   dimer_sep             dimer separation (distance from the middle point of the dimer to the two images)
%%%%%   T_anglerot            convergence threshold for the rotation angle
%%%%%   estim_Curv            if 1, an estimate for the curvature along the direction of the dimer after the rotation is calculated
%%%%%   rotinfo               structure array including necessary input information for the rotation method
%%%%%                         - rotinfo.F_rot_old: rotational force of the previous rotation iteration (1 x D)
%%%%%                         - rotinfo.F_modrot_old: modified rotational force of the previous rotation iteration (1 x D)
%%%%%                         - rotinfo.orient_rot_oldplane: unit vector perpendicular to 'orient' within the rotation plane of the previous rotation iteration (1 x D)
%%%%%                         - rotinfo.cgiter_rot: number of conjugated rotation iterations
%%%%%                         - rotinfo.num_cgiter_rot: maximum number of conjugated rotation iterations before resetting the conjugate directions
%%%%%   
%%%%% Output:
%%%%%   orient_new            unit vector along the direction of the dimer after optimal rotation (1 x D)
%%%%%   Curv                  estimate for the curvature along the direction of the dimer after the rotation (empty if estim_Curv = 0 or no rotation)
%%%%%   R_obs                 coordinates of the new observed location (1 x D)
%%%%%   E_obs                 energy at the new observed location
%%%%%   G_obs                 gradient at the new observed location (1 x D)
%%%%%   rotinfo               structure array including necessary input information for the rotation method (updated)
%%%%%                         - rotinfo.F_rot_old: rotational force of this rotation iteration (1 x D)
%%%%%                         - rotinfo.F_modrot_old: modified rotational force of this rotation iteration (1 x D)
%%%%%                         - rotinfo.orient_rot_oldplane: unit vector perpendicular to 'orient_new' within the rotation plane of this rotation iteration (1 x D)
%%%%%                         - rotinfo.cgiter_rot: number of conjugated rotation iterations
%%%%%                         - rotinfo.num_cgiter_rot: maximum number of conjugated rotation iterations before resetting the conjugate directions


function [orient_new,Curv,R_obs,E_obs,G_obs,rotinfo] = rot_iter_cg(R,orient,G01,potential,dimer_sep,T_anglerot,estim_Curv,rotinfo)
    F_rot_old = rotinfo.F_rot_old;
    F_modrot_old = rotinfo.F_modrot_old;
    orient_rot_oldplane = rotinfo.orient_rot_oldplane;
    cgiter_rot = rotinfo.cgiter_rot;
    num_cgiter_rot = rotinfo.num_cgiter_rot;
    if cgiter_rot >= num_cgiter_rot
        F_rot_old = 0;
        F_modrot_old = 0;
        orient_rot_oldplane = 0;
        cgiter_rot = 1;
    else
        cgiter_rot = cgiter_rot + 1;
    end
    F_rot = force_rot(G01,orient,dimer_sep);
    F_rot_old2 = F_rot_old*F_rot_old';
    if F_rot_old2 == 0
        F_modrot = F_rot;
    else
        gamma = ((F_rot-F_rot_old)*F_rot')/F_rot_old2;
        if gamma < 0 || sum((gamma*F_modrot_old).^2,2) > sum(F_rot.^2,2)
            F_modrot = F_rot;
            cgiter_rot = 1;
        else
            F_modrot = F_rot + gamma*sqrt(sum(rotinfo.F_modrot_old.^2,2))*orient_rot_oldplane;
        end
    end
    orient_rot = F_modrot-(orient*F_modrot')*orient;
    orient_rot = orient_rot/sqrt(sum(orient_rot.^2));
    F_rot_oriented = (F_rot*orient_rot')*orient_rot;
	[orient_new,orient_rot_new,Curv,~,R_obs,E_obs,G_obs] = rotate_dimer(R,orient,G01,F_rot_oriented,potential,dimer_sep,T_anglerot,estim_Curv,0);
    if isempty(R_obs)
        F_rot = 0;
        F_modrot = 0;
        cgiter_rot = 0;
    end
    rotinfo.F_rot_old = F_rot;
    rotinfo.F_modrot_old = F_modrot;
    rotinfo.orient_rot_oldplane = orient_rot_new;
    rotinfo.cgiter_rot = cgiter_rot;
end
