%%%%% dist_maxrel_atomwise3.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 11.7.2018
%%%%% 
%%%%% This function gives the maximum of the following ratio over the moving atoms:
%%%%% length of the change of the position of the moving atom between configurations C and C',
%%%%% relative to the distance from the atom to its nearest neighbour atom in C or C'
%%%%% (depending which is smaller):
%%%%%
%%%%% dist(C,C') = MAX_i{sqrt((x_i-x_i')^2+(y_i-y_i')^2+(z_i-z_i')^2)/MIN{mindist_interatomic_i(C),mindist_interatomic_i(C')}},
%%%%% where (x_i,y_i,z_i) and (x_i',y_i',z_i') are the coordinates of atoms i
%%%%% in configurations C and C', respectively.
%%%%%
%%%%% The input vectors x1 and x2 are assumed to be row vectors including the
%%%%% coordinates of the moving atoms: [x_1,y_1,z_1,x_2,y_2,z_2,...].
%%%%%
%%%%% Input:   x1           coordinates of the moving atoms in configurations C (n1 x 3*N_mov)
%%%%%          x2           coordinates of the moving atoms in configurations C' (n2 x 3*N_mov)
%%%%%          conf_info    structure array including information about the configurations necessary for the GP model
%%%%%                       - conf_info.conf_fro: coordinates of active frozen atoms (N_fro x 3)
%%%%%                       - conf_info.atomtype_mov: atomtype indices for moving atoms (1 x N_mov)
%%%%%                       - conf_info.atomtype_fro: pairtype indices for active frozen atoms (1 x N_fro)
%%%%%                       - conf_info.pairtype: pairtype indices for pairs of atomtypes (n_at x n_at)
%%%%%                       - conf_info.n_pt: number of active pairtypes
%%%%%
%%%%% Output:  dist         matrix including the "distances" from configurations C to configurations C' (n1 x n2)

function dist = dist_maxrel_atomwise3(x1,x2,conf_info)
    
    n1 = size(x1,1);
    n2 = size(x2,1);
    dist = zeros(n1,n2);
    mindist_interatomic_x1 = mindist_interatomic(x1,conf_info);
    mindist_interatomic_x2 = mindist_interatomic(x2,conf_info);
    for ind_x1 = 1:n1
        for ind_x2 = 1:n2
            dist_atomwise = sqrt(bsxfun(@minus,x1(ind_x1,1:3:end),x2(ind_x2,1:3:end)).^2+bsxfun(@minus,x1(ind_x1,2:3:end),x2(ind_x2,2:3:end)).^2+bsxfun(@minus,x1(ind_x1,3:3:end),x2(ind_x2,3:3:end)).^2);
            dist_rel_atomwise = dist_atomwise./min(mindist_interatomic_x1(ind_x1,:),mindist_interatomic_x2(ind_x2,:));
            dist(ind_x1,ind_x2) = max(dist_rel_atomwise,[],2);
        end
    end
    
end
