%%%%% demo_CuH2_atomic_GP_dimer.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
%%%%%
%%%%% This script shows how to use 'atomic_GP_dimer.m' in a CuH2 example.

load('orients_CuH2.mat')
load('MEP_CuH2.mat')
load('CuH2_idpp8_villi.mat')
i_dist = 3;
i_run = 5;

% 'pot_general' gives the potential energy and its gradient vector as a function of the atomic configuration.
% Each row of 'R' represents one configuration including the coordinates of the moving atoms:
% [x_1,y_1,z_1,x_2,y_2,z_2,...]
pot_general = @(R) ECuH2(R,conf);

dist_sp = [0.02,0.05,0.1,0.2,0.3,0.4,0.6,1.0,2.0,3.0];
R_all_init = []; % no initial data points
E_all_init = []; % no initial data
G_all_init = []; % no initial data
R_init = R_sp + dist_sp(i_dist)*orient_start(i_run,:); % define the initial middle point of the dimer
E_init = []; % initial middle point not observed
G_init = []; % initial middle point not observed
orient_init = orient_init(i_run,:); % define the initial orientation of the dimer (unit vector along the direction of the dimer)
D = size(R_init,2); % number of dimensions
N_mov = D/3; % number of moving atoms

% 'conf_info' is a structure array including information about the configurations necessary for the GP model:
conf_info = struct;
% 'conf_info.conf_fro': coordinates of active frozen atoms (N_fro x 3)
% In the beginning, list none of the frozen atoms as active:
conf_info.conf_fro = [];
% 'conf_info.atomtype_mov': atomtype indices for moving atoms (1 x N_mov)
conf_info.atomtype_mov = ones(1,N_mov); % H atoms
% 'conf_info.atomtype_fro': atomtype indices for active frozen atoms (1 x N_fro)
conf_info.atomtype_fro = [];
% The atomtypes must be indexed as 1,2,...,n_at.
n_at = 2; % number of atomtypes (including also the types of inactive frozen atoms)
% 'conf_info.pairtype': pairtype indices for pairs of atomtypes (n_at x n_at)
% Active pairtypes are indexed as 1,2,...,n_pt. Inactive pairtypes are given index 0.
conf_info.pairtype = zeros(n_at);
% 'conf_info.n_pt': number of active pairtypes
conf_info.n_pt = 0;
% Set pairtype indices for moving+moving atom pairs (and update number of active pairtypes):
[conf_info.pairtype,conf_info.n_pt] = set_pairtype_mov(conf_info.atomtype_mov,conf_info.pairtype,conf_info.n_pt);

% 'conf_info_inactive' is a structure array including information about inactive frozen atoms:
conf_info_inactive = struct;
% 'conf_info_inactive.conf_ifro': coordinates of inactive frozen atoms (N_ifro x 3)
% In the beginning, list all frozen atoms as inactive:
conf_info_inactive.conf_ifro = conf(1:216,1:3);
% 'conf_info_inactive.atomtype_ifro': atomtype indices for inactive frozen atoms (1 x N_ifro)
conf_info_inactive.atomtype_ifro = 2*ones(1,size(conf_info_inactive.conf_ifro,1)); % Cu atoms

% Only active frozen atoms are taken into account in the covariance function.
% A frozen atom is activated, when it is within the radius of 'actdist_fro'
% from some moving atom.
% Once a frozen atom is activated, it stays active from then on.
% If 'actdist_fro' is set to infinity, all frozen atoms are taken into account.
% When a frozen atom is activated, its coordinates and atomtype index are added
% to 'conf_info.conf_fro' and 'conf_info.atomtype_fro', respectively,
% and removed from 'conf_info_inactive.conf_fro' and 'conf_info_inactive.atomtype_fro'.
% If the new frozen atom activates new pairtypes, also 'conf_info.pairtype'
% and 'conf_info.n_pt' are updated.
actdist_fro = 5;
% Activate frozen atoms within activation distance:
[conf_info,conf_info_inactive,~] = update_active_fro(conf_info,conf_info_inactive,[R_init;R_all_init],actdist_fro);

dimer_sep = 0.01; % define the dimer separation (distance from the middle point of the dimer to the two images)
method_rot = @rot_iter_lbfgs; % use the conjugate gradient method for rotations
method_trans = @trans_iter_lbfgs; % use the conjugate gradient method for translations
param_trans = [0.1, 0.1]; % define a step length for convex regions and maximum step length

% 'eval_image1' indicates if image 1 of the dimer is evaluted (1) or not (0)
% after each relaxation phase in addition to the middle point of the dimer
eval_image1 = 0;

% 'T_dimer' defines the final convergence threshold for 'maxF_R', which is
% the maximum component of the force acting on the middle point of the dimer (i.e., the
% algorithm is stopped when all components of the accurate force are below 'T_dimer').
T_dimer = 0.01;

% 'initrot_nogp' indicates if the initial rotations are performed without GP (1) or with GP (0).
initrot_nogp = 0;

% 'T_anglerot_init' defines a convergence threshold for the rotation angle
% in the initial rotations performed in the beginning of the algorithm
% (the dimer is not rotated when the estimated rotation angle is less than this).
T_anglerot_init = 0.0873;

% 'num_iter_initrot' defines the maximum number of initial rotations (0 if initial rotations skipped).
num_iter_initrot = D;

% 'inittrans_nogp' is an indicator if an initial test translation step is taken without GP (1)
% or if GP is used right after initial rotations (0).
inittrans_nogp = 0;

% 'T_anglerot_gp' defines a convergence threshold for the rotation angle
% during a relaxation phase (the dimer is not rotated when the estimated
% rotation angle is less than this).
T_anglerot_gp = 0.01;

% 'num_iter_rot_gp' defines a maximum number of rotation iterations per
% translation during a relaxation phase.
num_iter_rot_gp = 10;

% If 'divisor_T_dimer_gp' is set to zero, the default convergence threshold
% for each relaxation phase for the approximated 'maxF_R' on the
% approximated energy surface is 1/10 of the 'T_dimer'. To save inner
% iterations during the first relaxation phases, one can set a positive
% value for 'divisor_T_dimer_gp', so that the GP convergence threshold will
% be 1/'divisor_T_dimer_gp' of the smallest accurate 'maxF_R' obtained so
% far, but not less than 1/10 of the 'T_dimer'. If the approximation error
% is assumed to not decrease more than that during one outer iteration,
% there is no need for more accurate relaxation on an approximated surface.
divisor_T_dimer_gp = 10;

% 'disp_max' defines the maximum displacement of the middle point of the
% dimer from the nearest observed data point. Thus, the last inner step is
% rejected and the relaxation phase stopped, if the distance to the nearest
% observed data point is larger than 'disp_max'.
disp_max = 0.5;

% 'ratio_at_limit' defines the limit for the ratio (< 1) of inter-atomic
% distances between image and its "nearest" observed data point.
% More precisely, the last inner step is rejected and the relaxation phase
% stopped if the following does not hold:
% There is an observed data point so that all inter-atomic distances of the
% current image are more than 'ratio_at_limit' (by default 2/3) but less
% than 1/'ratio_at_limit' (3/2) times the corresponding inter-atomic
% distance of the observed data point.
ratio_at_limit = 2/3;

% 'num_bigiter_initloc' defines the number of outer iterations started from the initial location 'R_init'.
% After that, each relaxation phase is started from the latest converged dimer.
% Starting each round from the initial location may improve stability (and decrease outer iterations),
% but starting from the latest dimer may decrease the number of inner iterations during the relaxation phases.
num_bigiter_initloc = inf;

% 'num_bigiter_initparam' defines the number of outer iterations where the hyperparameter
% optimization is started from values initialized based on the range of current data.
% After that, the optimization is started from the values of the previous round.
num_bigiter_initparam = inf;

num_bigiter = 300; % define the maximum number of outer iterations (new sets of observations)
num_iter = 10000; % define the maximum number of inner iterations (steps during a relaxation phase)

% 'islarge_num_iter' indicates if 'num_iter' is assumed to be much larger than required
% for dimer convergence on accurate energy surface. If not (0), the next relaxation phase is
% continued from the current dimer in case 'num_iter' is reached.
islarge_num_iter = 1;

load_file = []; % start normally from beginning
save_file = []; % no saves after each outer iteration

% Call the atomic GP-dimer algorithm
[R,orient,E_R,G_R,gp,R_all,E_all,G_all,obs_at,E_R_acc,E_R_gp,maxF_R_acc,maxF_R_gp,param_gp_initrot,param_gp,obs_initrot,obs_total,num_esmax,num_es1,num_es2] = ...
             atomic_GP_dimer(pot_general,conf_info,conf_info_inactive,actdist_fro,R_all_init,E_all_init,G_all_init,R_init,E_init,G_init,orient_init,dimer_sep,method_rot,method_trans, ...
             param_trans,eval_image1,T_dimer,initrot_nogp,T_anglerot_init,num_iter_initrot,inittrans_nogp,T_anglerot_gp,num_iter_rot_gp,divisor_T_dimer_gp, ...
             disp_max,ratio_at_limit,num_bigiter_initloc,num_bigiter_initparam,num_bigiter,num_iter,islarge_num_iter,load_file,save_file);      
         
%{
% Plot the behaviour
figure
subplot(2,1,1)
title('Maximum component of force on the middle point (GP approximation)')
xlabel('iteration')
hold
plot(0:(size(maxF_R_gp,2)-1),maxF_R_gp)
plot(obs_at,maxF_R_acc,'o')
subplot(2,1,2)
title('Energy at the middle point (GP approximation)')
xlabel('iteration')
hold
plot(0:(size(E_R_gp,2)-1),E_R_gp)
plot(obs_at,E_R_acc,'o')
%}

