%%%%% demo_heptamer_atomic_GP_NEB_AIE.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 1.8.2018
%%%%%
%%%%% This script shows how to use 'atomic_GP_NEB_AIE.m' in a heptamer example.

addpath(genpath('/users/home/via9/GPRNEB/GPR_CODE/data_lennardjones/'))
addpath(genpath('/users/home/via9/GPRNEB/GPR_CODE/GP_NEB_SCRIPTS/'))
addpath(genpath('/users/home/via9/GPRNEB/gpstuff-develop/'))


conf1 = readxyz('react.xyz'); 
conf2 = readxyz('prod.xyz');
constr = zeros(length(conf1),1);
indices = linspace(1,length(conf1), length(conf1))';
conf1 = [conf1 constr indices];
conf2 = [conf2 constr indices];
conf = conf1;

% 'pot_general' gives the potential energy and its gradient vector as a function of the atomic configuration.
% Each row of 'R' represents one configuration including the coordinates of the moving atoms:
% [x_1,y_1,z_1,x_2,y_2,z_2,...]
pot_general = @(R) EvalLJ(R, conf);
min1 = conf2vec(conf); % define the first minimum point
min2 = conf2vec(conf2); % define the second minimum point
N_mov = size(min1,2)/3; % number of moving atoms

fprintf('Testing potential energy minimum 1')
[E1, G1] = pot_general(min1)
fprintf('Testing potential energy minimum 2')
[E2, G2] = pot_general(min2)

fprintf('Starting GPR-NEB run')
% 
% 'conf_info' is a structure array including information about the configurations necessary for the GP model:
conf_info = struct;
% 'conf_info.conf_fro': coordinates of active frozen atoms (N_fro x 3)
% In the beginning, list none of the frozen atoms as active:
conf_info.conf_fro = [];
% 'conf_info.atomtype_mov': atomtype indices for moving atoms (1 x N_mov)
conf_info.atomtype_mov = ones(1,sum(conf(:,4)==0));
% 'conf_info.atomtype_fro': atomtype indices for active frozen atoms (1 x N_fro)
conf_info.atomtype_fro = [];
% The atomtypes must be indexed as 1,2,...,n_at.
n_at = 1; % number of atomtypes (including also the types of inactive frozen atoms)
% 'conf_info.pairtype': pairtype indices for pairs of atomtypes (n_at x n_at)
% Active pairtypes are indexed as 1,2,...,n_pt. Inactive pairtypes are given index 0.
conf_info.pairtype = zeros(n_at);
% 'conf_info.n_pt': number of active pairtypes
conf_info.n_pt = 0;
% Set pairtype indices for moving+moving atom pairs (and update number of active pairtypes):
[conf_info.pairtype,conf_info.n_pt] = set_pairtype_mov(conf_info.atomtype_mov,conf_info.pairtype,conf_info.n_pt);

% 'conf_info_inactive' is a structure array including information about inactive frozen atoms:
conf_info_inactive = struct;
% 'conf_info_inactive.conf_ifro': coordinates of inactive frozen atoms (N_ifro x 3)
% In the beginning, list all frozen atoms as inactive:
conf_info_inactive.conf_ifro = conf(conf(:,4)==1,1:3);
% 'conf_info_inactive.atomtype_ifro': atomtype indices for inactive frozen atoms (1 x N_ifro)
conf_info_inactive.atomtype_ifro = ones(1,size(conf_info_inactive.conf_ifro,1));

% Only active frozen atoms are taken into account in the covariance function.
% A frozen atom is activated, when it is within the radius of 'actdist_fro'
% from some moving atom in some configuration on the path.
% Once a frozen atom is activated, it stays active from then on.
% If 'actdist_fro' is set to infinity, all frozen atoms are taken into account.
% When a frozen atom is activated, its coordinates and atomtype index are added
% to 'conf_info.conf_fro' and 'conf_info.atomtype_fro', respectively,
% and removed from 'conf_info_inactive.conf_fro' and 'conf_info_inactive.atomtype_fro'.
% If the new frozen atom activates new pairtypes, also 'conf_info.pairtype'
% and 'conf_info.n_pt' are updated.
actdist_fro = 5;
% Activate frozen atoms within activation distance:
[conf_info,conf_info_inactive,~] = update_active_fro(conf_info,conf_info_inactive,[min1;min2],actdist_fro);

N_im = 7; % define the number of images on the path
R_init = initialize_path_linear(min1,min2,N_im); % define the initial path
method_step = @(R,F_R,param_step,F_R_old,V_old,zeroV) step_QMVelocityVerlet(R,F_R,param_step,F_R_old,V_old,zeroV); % use quick-min Velocity Verlet to define the steps during path relaxation
param_step = 0.01; % define the time step for the quick-min Velocity Verlet algorithm
k_par = 1; % define the parallel spring constant

% 'T_MEP' defines the final convergence threshold for the 'maxnormF', which is
% the maximum of the accurate norms of the NEB forces acting on the 'num_im'-2 intermediate
% images (i.e., the algorithm is stopped when the accurate NEB force is below 'T_MEP' for all images).
T_MEP = 0.3;

% 'T_CI' defines an additional final convergence threshold for the
% climbing image, if the climbing image option is used.
% If you don't want to use a tighter convergence threshold for the climbing
% image, set 'T_CI' equal to 'T_MEP' (or larger, because
% the general threshold 'T_MEP' concerns also the climbing image).
T_CI = 0.01;

% 'T_CIon_gp' defines a preliminary convergence threshold for each relaxation phase:
% When the approximated 'maxnormF' is below 'T_CIon_gp', the climbing
% image mode is turned on.
% If you don't want to use climbing image at all, set 'T_CIon_gp' to zero.
T_CIon_gp = 1;

% If 'divisor_T_MEP_gp' is set to zero, the default convergence threshold for each relaxation
% phase for the approximated 'maxnormF' on the approximated energy surface is 1/10 of the lowest final threshold.
% To save inner iterations during the first relaxation phases, one can set
% a positive value for 'divisor_T_MEP_gp', so that the GP convergence threshold will be
% 1/'divisor_T_MEP_gp' of the smallest accurate norm of NEB force obtained so far
% on any of the 'N_im'-2 intermediate images, but not less than 1/10 of the lowest final threshold.
% If the approximation error is assumed to not decrease more than that during one outer iteration,
% there is no need for more accurate relaxation on an approximated surface.
divisor_T_MEP_gp = 0;

% 'disp_max' defines the maximum displacement of image from the nearest observed data point
% relative to the length of the initial path.
% Thus, the last inner step is rejected and the relaxation phase stopped if, for any image, the distance
% to the nearest observed data point is larger than 'disp_max' times the length of the initial path.
disp_max = 0.5;

% 'ratio_at_limit' defines the limit for the ratio (< 1) of inter-atomic distances between image
% and its "nearest" observed data point.
% More precisely, the last inner step is rejected and the relaxation phase stopped if the following
% does not hold for some of the current images:
% There is an observed data point so that all inter-atomic distances of the current image are more
% than 'ratio_at_limit' (by default 2/3) but less than 1/'ratio_at_limit' (3/2) times the corresponding
% inter-atomic distance of the observed data point.
ratio_at_limit = 2/3;

% 'num_bigiter_init' defines the number of outer iterations started from the initial path 'R_init'
% - Until 'num_bigiter_init' is reached, each relaxation phase is started from the initial path 'R_init'.
%     (If climbing image is used, the CI phase is continued from the "preliminarily converged" evenly spaced path.)
% - After that, each relaxation phase is started from the latest converged path.
%     (If climbing image is used, each relaxation phase is started from the latest "preliminarily converged" evenly spaced path,
%      and the CI phase started from the latest converged CI-path if CI is unchanged
%      (otherwise continued from the current "preliminarily converged" evenly spaced path).)
% Starting each round from the initial path may improve stability (and decrease outer iterations),
% but starting from the latest path may decrease the amount of inner iterations during the relaxation phases.
num_bigiter_init = inf;

num_bigiter = 300; % define the maximum number of outer iterations (new sets of observations)
num_iter = 10000; % define the maximum number of inner iterations (steps during a relaxation phase)

% 'num_bigiter_hess' defines the number of outer iterations using the "virtual Hessian",
% i.e., additional observations around the minimum points. The "virtual Hessian"
% may slow down the GP computations especially in high-dimensional cases,
% but they may give useful information in the beginning.
% They usually don't bring gain after 4 outer iterations (but in some cases do).
% By setting 'num_bigiter_hess' to zero, the "virtual Hessian" is set off.
num_bigiter_hess = inf;
eps_hess = 0.001; % defines the distance of the additional points from the minimum points

% Call the atomic GP-NEB algorithm
[R,E_R,G_R,i_CI, bigtimes_iter, gp,R_all,E_all,G_all,obs_at,E_R_acc,E_R_gp,normF_R_acc,normF_R_gp,normFCI_acc,normFCI_gp,param_gp] = ...
             atomic_GP_NEB_AIE(pot_general,conf_info,conf_info_inactive,actdist_fro,R_init,method_step,param_step,k_par,T_MEP,T_CI, ...
             T_CIon_gp,divisor_T_MEP_gp,disp_max,ratio_at_limit,num_bigiter_init,num_bigiter,num_iter,num_bigiter_hess,eps_hess);

fprintf('%d %f' \n', bigtimes_iter, max(E_R))
fprintf('Done\n')