%%%%% GP_sNEB_AIE.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 15.9.2017
%%%%%
%%%%% This is the simpler all-images-evaluated (AIE) version of the
%%%%% GP-sNEB algorithm for finding a minimum energy path and a saddle
%%%%% point between two minimum points.
%%%%% The relaxation of the path on the approximated energy surface
%%%%% is done according to the stabilized nudged elastic band (sNEB) method with
%%%%% a climbing image option for the highest-energy image.
%%%%% After each relaxation phase, the energy and gradient are acquired
%%%%% for all the intermediate images, and the GP hyperparameters are
%%%%% reoptimized.
%%%%%
%%%%% Using the function requires that GPstuff is installed and added to the path in Matlab.
%%%%%
%%%%% Input:
%%%%%   pot_general        accurate potential and gradient function
%%%%%   R_init             coordinates for the images on the initial path (N_im x D)
%%%%%   method_step        a function defining the following step during path relaxation
%%%%%   param_step         parameters of the path relaxation method (shape depends on 'method_step')
%%%%%   k_par              parallel spring constant
%%%%%   k_perp             perpendicular spring constant
%%%%%   T_MEP              final convergence threshold (the algorithm is stopped when the accurate norm of
%%%%%                        NEB force is less than this for all images)
%%%%%   T_CI               additional final convergence threshold for the climbing image
%%%%%   T_CIon_gp          premilinary GP convergence threshold after which the climbing image
%%%%%                        mode is turned on during relaxation phase (use 0 if CI not used at all)
%%%%%   divisor_T_MEP_gp   if this option is set on (> 0), the convergence threshold for a relaxation phase
%%%%%                        is 1/'divisor_T_MEP_gp' of the smallest accurate norm of NEB force obtained so far
%%%%%                        on any of the intermediate images, but not less than 1/10 of the lowest final threshold
%%%%%                        (otherwise the GP convergence threshold is always 1/10 of the lowest final threshold)
%%%%%   disp_max           maximum displacement of image from the nearest observed data point
%%%%%                        relative to the length of the initial path
%%%%%                        (the relaxation phase is stopped if 'disp_max' is reached for any image)
%%%%%   num_bigiter_init   number of outer iterations started from the initial path 'R_init'
%%%%%                      - Until 'num_bigiter_init' is reached, each relaxation phase is started from the initial path 'R_init'
%%%%%                          (if climbing image is used, the CI phase is continued from the "preliminarily converged" evenly spaced path)
%%%%%                      - After that, each relaxation phase is started from the latest converged path
%%%%%                          (if climbing image is used, each relaxation phase is started
%%%%%                           from the latest "preliminarily converged" evenly spaced path,
%%%%%                           and the CI phase is started from the latest converged CI-path if CI is unchanged
%%%%%                           (otherwise continued from the current "preliminarily converged" evenly spaced path))
%%%%%   num_bigiter        maximum number of outer iterations (new sets of evaluations)
%%%%%   num_iter           maximum number of inner iterations (steps during a relaxation phase)
%%%%%   num_bigiter_hess   number of outer iterations using the "virtual Hessian" around the minimum points
%%%%%   eps_hess           epsilon for the "virtual Hessian"
%%%%%   gp                 the GP model defined using 'gp_set' in GPstuff
%%%%%   opt                optimization settings for the GP hyperparameters defined using 'optimset'
%%%%%   optimf             optimization function for the GP hyperparameters
%%%%%
%%%%% Output:
%%%%%   R                  coordinates for the images on the final path (N_im x D)
%%%%%   E_R                energy at the images on the final path (N_im x 1)
%%%%%   G_R                gradient at the images on the final path (N_im x D)
%%%%%   i_CI               index of the climbing image among the intermediate images of the final path
%%%%%   gp                 the final GP model
%%%%%   R_all              coordinates for all image observations
%%%%%   E_all              energy for all image observations
%%%%%   G_all              gradient for all image observations
%%%%%   obs_at             total numbers of inner iterations before new observations were taken
%%%%%   E_R_acc            accurate energies of the images for each outer iteration
%%%%%   E_R_gp             approximated energies of the images for each inner iteration
%%%%%   normF_R_acc        accurate norm of NEB force acting on each intermediate image for each outer iteration
%%%%%   normF_R_gp         approximated norm of NEB force acting on each intermediate image for each inner iteration
%%%%%   normFCI_acc        accurate norm of NEB force acting on the climbing image for each outer iteration (0 if CI is off)
%%%%%   normFCI_gp         approximated norm of NEB force acting on the climbing image for each inner iteration (0 if CI is off)
%%%%%   param_gp           optimized GP hyperparameters for each outer iteration (in logarithmic scale)

function [R,E_R,G_R,i_CI,gp,R_all,E_all,G_all,obs_at,E_R_acc,E_R_gp,normF_R_acc,normF_R_gp,normFCI_acc,normFCI_gp,param_gp] = ...
             GP_sNEB_AIE(pot_general,R_init,method_step,param_step,k_par,k_perp,T_MEP,T_CI, ...
             T_CIon_gp,divisor_T_MEP_gp,disp_max,num_bigiter_init,num_bigiter,num_iter,num_bigiter_hess,eps_hess,gp,opt,optimf)

         
    %%%     
    %%% THIS INFORMATION IS ASSUMED TO BE KNOWN BEFORE BEGINNING
    %%%
    
    N_im = size(R_init,1); % number of images on the path
    D = size(R_init,2); % dimension of the space
    min1 = R_init(1,:); % minimum point 1
    min2 = R_init(end,:); % minimum point 2
    scale = 0;
    for im = 1:N_im-1
        scale = scale + sqrt(sum((R_init(im+1,:)-R_init(im,:)).^2)); % length of the initial path
    end
    [E_min1,G_min1] = pot_general(min1); % energy and gradient at minimum point 1
    [E_min2,G_min2] = pot_general(min2); % energy and gradient at minimum point 2
    % Elevel = min([E_min1,E_min2]); % zero level of energy is set to the lower minimum
    Elevel = E_min1; % zero level of energy is set to minimum point 1
    E_min1 = E_min1 - Elevel;
    E_min2 = E_min2 - Elevel;
    if num_bigiter_hess > 0
        R_h = get_hessian_points(R_init,eps_hess); % define the "virtual Hessian" points if used
        [E_h,G_h] = pot_general(R_h); % energy and gradient at the "virtual Hessian" points
        E_h = E_h - Elevel;
    else
        R_h = [];
        E_h = [];
        G_h = [];
    end
    R_all = [R_h;min1;min2]; % coordinates of all observation points
    E_all = [E_h;E_min1;E_min2]; % energy for all observation points
    G_all = [G_h;G_min1;G_min2]; % gradient for all observation points
       
    
    %%%
    %%% THE ALGORITHM BEGINS HERE
    %%%
    
    R = R_init; % coordinates of the images
    R_latest_equal = []; % latest evenly spaced path (no climbing image)
    if T_CIon_gp > 0
        i_CI_latest = 0; % latest climbing image index among the intermediate images
    end
    
    % in case of 2D space, define a range for visualization and plot the initial path
    if D == 2
        scale1 = abs(min2(1)-min1(1))/4;
        scale2 = abs(min2(2)-min1(2))/4;
        [X1,X2] = meshgrid(min(min1(1),min2(1))-2*scale1:scale1/20:max(min1(1),min2(1))+scale1,min(min1(2),min2(2))-scale2:scale2/20:max(min1(2),min2(2))+scale2);
        R_all2 = [repmat(R_all,D+1,1),reshape(repmat(0:D,size(R_all,1),1),[],1)];
        gp = gp_optim(gp,R_all2,[E_all;G_all(:)],'opt',opt,'optimf',optimf);
        param_gp_init = gp_pak(gp) % optimized GP hyperparameters for the initial data (in logarithmic scale)
        figure()
        [Ef,Varf] = gp_pred(gp, R_all2, [E_all;G_all(:)], [X1(:),X2(:),zeros(size(X1(:)))]);
        pcolor(X1,X2,reshape(Ef,size(X1,1),size(X1,2))),shading flat;
        colorbar;
        hold on;
        axis equal tight;
        plot(R(:,1),R(:,2),'yo','markerFaceColor','y')  
        plot(R_all(:,1),R_all(:,2),'r+')
        title('Approximated energy surface in the beginning, initial path');
    end
    
    E_R_acc = []; % matrix gathering accurate energies of the images for each outer iteration
    E_R_gp = []; % matrix gathering approximated energies of the images for each inner iteration
    normF_R_acc = []; % matrix gathering accurate norm of the NEB force acting on each intermediate image for each outer iteration
    normFCI_acc = []; % vector gathering accurate norm of the NEB force acting on the climbing image for each outer iteration (0 if CI is off)
    normF_R_gp = []; % matrix gathering approximated norm of the NEB force acting on each intermediate image for each inner iteration
    normFCI_gp = []; % vector gathering approximated norm of the NEB force acting on the climbing image for each inner iteration (0 if CI is off)
    obs_at = []; % vector gathering the total numbers of inner iterations before new observations were taken    
    param_gp = []; % optimized GP hyperparameters for each relaxation phase (in logarithmic scale)

    for bigiter = 0:num_bigiter % OUTER ITERATION LOOP
        
        % acquire the accurate energy and gradient on the relaxed path and add them to the data
        E_R = [E_min1;zeros(N_im-2,1);E_min2]; % energy at the images
        G_R = [G_min1;zeros(N_im-2,D);G_min2]; % gradient at the images
        R_all = [R_all;R(2:(N_im-1),:)];
        [E_R(2:(N_im-1),:),G_R(2:(N_im-1),:)] = pot_general(R(2:(N_im-1),:));
        E_R(2:(N_im-1),:) = E_R(2:(N_im-1),:) - Elevel;
        [F_R,normFCI,i_CI] = force_sNEB(R,E_R,G_R,k_par,k_perp,T_CIon_gp);
        E_all = [E_all;E_R(2:(N_im-1),:)];
        G_all = [G_all;G_R(2:(N_im-1),:)];
        E_R_acc = [E_R_acc,E_R];
        normF_R_acc = [normF_R_acc,sqrt(sum(F_R.^2,2))];
        normFCI_acc = [normFCI_acc,normFCI];
        obs_at = [obs_at; size(E_R_gp,2)];
        if T_CIon_gp > 0
            fprintf('Accurate values: meanE_R = %1.3g, maxnormF_R = %1.3g, minnormF_R = %1.3g, normFCI = %1.3g (image %g) \n\n',mean(E_R_acc(:,end)),max(normF_R_acc(:,end)),min(normF_R_acc(:,end)),normFCI_acc(end),i_CI+1);
        else
            fprintf('Accurate values: meanE_R = %1.3g, maxnormF_R = %1.3g, minnormF_R = %1.3g \n\n',mean(E_R_acc(:,end)),max(normF_R_acc(:,end)),min(normF_R_acc(:,end))); 
        end

        % stop the algorithm if final convergence is obtained
        if max(normF_R_acc(:,end)) < T_MEP && normFCI_acc(end) < T_CI
            fprintf('Final convergence obtained after %g relaxation phases (%g image evaluations).\n', bigiter, (N_im-2)*(bigiter+1));
            break;
        end

        % stop the algorithm if maximum number of outer iterations is reached
        if bigiter == num_bigiter
            fprintf('Stopped the algorithm: Maximum number of outer iterations (%g) reached.\n', bigiter);
            break;
        end

        % remove the "virtual Hessian" observations if needed
        if num_bigiter_hess > 0 && bigiter == num_bigiter_hess
            R_all(1:2*D,:) = [];
            E_all(1:2*D,:) = [];
            G_all(1:2*D,:) = [];
        end

        % visualize the true energy along the spline interpolation of the path
        figure(10)
        hold on
        visualize_path_true
        hold off
        
        % optimize the GP hyperparameters and calculate some variables unchanged during the relaxation
        R_all2 = [repmat(R_all,D+1,1),reshape(repmat(0:D,size(R_all,1),1),[],1)];
        gp = gp_optim(gp,R_all2,[E_all;G_all(:)],'opt',opt,'optimf',optimf);
        param_gp = [param_gp;gp_pak(gp)];
        [~, C] = gp_trcov(gp, R_all2);
        L = chol(C,'lower');
        a = L'\(L\[E_all;G_all(:)]);
        
        % define the convergence threshold for the relaxation phase
        if divisor_T_MEP_gp > 0
            % if this option is set on, the GP convergence threshold is 1/'divisor_T_MEP_gp'
            % of the smallest accurate norm of NEB force obtained so far on any of the intermediate images,
            % but not less than 1/10 of the lowest final threshold
            T_MEP_gp = max([min(min(normF_R_acc))/divisor_T_MEP_gp,min([T_MEP/10,T_CI/10])]);
        else
            % otherwise the GP convergence threshold is always 1/10 of the lowest final threshold
            T_MEP_gp = min([T_MEP,T_CI])/10;
        end
        
        % define the start path for the relaxation phase
        if bigiter >= num_bigiter_init && ~isempty(R_latest_equal)
            R = R_latest_equal;
            if T_CIon_gp > 0
                fprintf('Started relaxation phase %g from the latest "preliminarily converged" evenly spaced path (no climbing image).\n', bigiter+1);
            else
                fprintf('Started relaxation phase %g from the latest converged path.\n', bigiter+1);
            end
        else
            R = R_init;
            fprintf('Started relaxation phase %g from the initial path.\n', bigiter+1);
        end
        
        iters = 0;
        CI_on = 0; % set climbing image mode off in the beginning
        not_relaxed = 0; % indicator of early stopping
        V_old = zeros(N_im-2,D); % velocities of the intermediate images (given as an output of the previous step)
        F_R_old = 0; % NEB forces on the intermediate images of the previous path
        zeroV = 1; % indicator if zero velocity used (for the first iteration)
        
        for iter = 0:num_iter % INNER ITERATION LOOP
            
            % calculate approximated energy and gradient on the path
            KK = gp_cov(gp,R_all2,[repmat(R,D+1,1),reshape(repmat(0:D,N_im,1),[],1)]);
            EG_R = KK'*a;
            E_R = EG_R(1:N_im,1);
            G_R = reshape(EG_R(N_im+1:end,1),N_im,D);
            [F_R,normFCI,i_CI] = force_sNEB(R,E_R,G_R,k_par,k_perp,CI_on);
            normF_R = sqrt(sum(F_R.^2,2));
            
            % turn climbing image mode on and correct the NEB force accordingly if sufficiently relaxed
            if CI_on <= 0 && max(normF_R) < T_CIon_gp
                R_latest_equal = R;
                CI_on = 1;
                [~,i_CI_test] = max(E_R(2:(N_im-1),:));
                fprintf('Climbing image (image %g) turned on after %g inner iterations. \n', i_CI_test+1, iters);
                if bigiter+1 > num_bigiter_init && i_CI_test == i_CI_latest
                    KK_test2 = gp_cov(gp,R_all2,[repmat(R_latest_climb,D+1,1),reshape(repmat(0:D,N_im,1),[],1)]);
                    EG_R_test2 = KK_test2'*a;
                    E_R_test2 = EG_R_test2(1:N_im,1);
                    [~,i_CI_test2] = max(E_R_test2(2:(N_im-1),:));
                    if i_CI_test2 == i_CI_latest
                        R = R_latest_climb;
                        E_R = E_R_test2;
                        G_R = reshape(EG_R_test2(N_im+1:end,1),N_im,D);
                        fprintf('CI unchanged: continued from the latest converged CI-path.\n');
                    end
                end
                [F_R,normFCI,i_CI] = force_sNEB(R,E_R,G_R,k_par,k_perp,CI_on);
                normF_R = sqrt(sum(F_R.^2,2));
                zeroV = 1;
            end

            E_R_gp = [E_R_gp,E_R];
            normF_R_gp = [normF_R_gp,normF_R];
            normFCI_gp = [normFCI_gp,normFCI];
            
            % stop the relaxation phase if converged
            if ( T_CIon_gp <= 0 || CI_on > 0 ) && max(normF_R) < T_MEP_gp && iters > 0
                if CI_on > 0
                    R_latest_climb = R;
                    i_CI_latest = i_CI;
                    fprintf('Stopped relaxation phase %g: converged after %g inner iterations (CI: image %g).\n', bigiter+1, iters, i_CI+1);
                else
                    R_latest_equal = R;
                    fprintf('Stopped relaxation phase %g: converged after %g inner iterations.\n', bigiter+1, iters);
                end
                break;
            end

            % stop the relaxation phase if maximum number of inner iterations reached
            if iters == num_iter
                fprintf('Stopped the relaxation phase: maximum number of inner iterations (%g) reached.\n', iters)
                not_relaxed = 1;
                break;
            end
            
            % move the path one step along the NEB force according to the chosen method
            [R_new,V_old] = method_step(R,F_R,param_step,F_R_old,V_old,zeroV);
            zeroV = 0;
            
            % reject the step and stop the relaxation phase if the distance from any current image to the
            % nearest observed data point is larger than 'disp_max' times the length of the initial path
            if iters > 0
                disp_nearest = zeros(N_im-2,1);
                for im = 2:N_im-1
                    disp_nearest(im-1,1) = sqrt(min(sum((repmat(R_new(im,:),size(R_all,1),1)-R_all).^2,2)));
                end
                [maxdisp_nearest,i_maxdisp_nearest] = max(disp_nearest);
                if maxdisp_nearest > disp_max*scale
                    fprintf('Stopped the relaxation phase after %g inner iterations: image %g too far from the nearest observed data point.\n', iters, i_maxdisp_nearest+1)
                    not_relaxed = 1;
                    break;
                end
            end
            
            % otherwise accept the step and continue the relaxation
            iters = iters + 1;
            R = R_new;
            F_R_old = F_R;
            
        end % END OF INNER ITERATION LOOP

        % in case of 2D space, plot the relaxed path
        if D == 2
            figure()
            [Ef,Varf] = gp_pred(gp, R_all2, [E_all;G_all(:)], [X1(:),X2(:),zeros(size(X1(:)))]);
            pcolor(X1,X2,reshape(Ef,size(X1,1),size(X1,2))),shading flat;
            colorbar;
            hold on;
            axis equal tight;
            plot(R(:,1),R(:,2),'yo','markerFaceColor','y') 
            plot(R_all(:,1),R_all(:,2),'r+') 
            if not_relaxed > 0
                title(['Approximated energy surface on round ',num2str(bigiter+1),', path relaxation stopped early']);
            else
                title(['Approximated energy surface on round ',num2str(bigiter+1),', relaxed path']);
            end
        end
        
    end % END OF OUTER ITERATION LOOP
    
    % visualize the true energy along the spline interpolation of the final path
    figure(10)
    hold on
    N_im = size(R,1);
    R_spline = spline((0:N_im-1)/(N_im-1),R',(0:0.1:N_im-1)/(N_im-1))';
    [E_spline,G_spline] = pot_general(R_spline);
    E_spline = E_spline - Elevel;
    plot(1:0.1:N_im,E_spline,'b','LineWidth',2);
    plot(1:N_im,E_R,'o','MarkerEdgeColor','b','MarkerFaceColor','b')
    hold off
    
end


