%%%%% trans_iter.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 9.7.2020
%%%%%
%%%%% This is an auxiliary function for the dimer method ('dimer.m').
%%%%% The dimer is translated one step towards saddle point according to the
%%%%% Newton method.
%%%%%
%%%%% Input:
%%%%%   R                  coordinates of the middle point of the dimer (1 x D)
%%%%%   orient             unit vector along the direction of the dimer (1 x D)
%%%%%   F_R                force at the middle point of the dimer (1 x D)
%%%%%   Curv               curvature of energy along the direction of the dimer
%%%%%   param_trans        [predefined step length for convex regions, maximum step length]
%%%%%   transinfo          structure array including necessary input information for the translation method
%%%%%                      - transinfo.potential: potential and gradient function
%%%%%
%%%%% Output:
%%%%%   R_new              coordinates of the new middle point of the dimer (1 x D)
%%%%%   R_obs              coordinates of the new observed location (1 x D)
%%%%%   E_obs              energy at the new observed location
%%%%%   G_obs              gradient at the new observe location (1 x D)
%%%%%   transinfo          structure array including necessary input information for the translation method
%%%%%                      - transinfo.potential: potential and gradient function


function [R_new,R_obs,E_obs,G_obs,transinfo] = trans_iter(R,orient,F_R,Curv,param_trans,transinfo)
    potential = transinfo.potential;
    steplength_convex = param_trans(1,1);
    max_steplength = param_trans(1,2);
    if Curv < 0
        F_trans = F_R - 2*(F_R*orient')*orient;
        F_0 = sqrt(sum(F_trans.^2,2));
        orient_search = F_trans/F_0;
        dr = 0.5*F_0/abs(Curv);
        R_dr = R + dr*orient_search;
        [E_R_dr,G_R_dr] = potential(R_dr);
        F_trans_dr = -G_R_dr + 2*(G_R_dr*orient')*orient;
        F_dr = F_trans_dr*orient_search';
        steplength = -F_0*dr/(F_dr-F_0);
        if steplength < 0
            steplength = steplength_convex;
        elseif steplength > max_steplength
            steplength = max_steplength;
        end
        R_obs = R_dr;
        E_obs = E_R_dr;
        G_obs = G_R_dr;
    else
        F_trans = -(F_R*orient')*orient;
        orient_search = F_trans/sqrt(sum(F_trans.^2,2));
        steplength = steplength_convex;
        R_obs = [];
        E_obs = [];
        G_obs = [];
    end 
    R_new = R + steplength*orient_search;   
end
