%%%%% step_QMVelocityVerlet.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 1.8.2018
%%%%%
%%%%% This function moves the path one step along the NEB force according
%%%%% to the quick-min optimizer implemented using the velocity Verlet algorithm.
%%%%%
%%%%% Input:
%%%%%   R           coordinates for the images on the current path (N_im x D)
%%%%%   F_R         NEB force on the intermediate images of the current path
%%%%%   param_step  time step dt (scalar)
%%%%%   F_R_old     NEB force on the intermediate images of the previous path
%%%%%   V_old       velocity of the intermediate images (given as output of the previous step)
%%%%%   zeroV       indicator if zero velocity used (for the first iteration)
%%%%%
%%%%% Output:
%%%%%   R_new       coordinates for the images on the moved path
%%%%%   V           velocity of the intermediate images

function [R_new,V] = step_QMVelocityVerlet(R,F_R,param_step,F_R_old,V_old,zeroV)
    dt = param_step;
    [N_im,D] = size(R);    
    if zeroV > 0
        V = zeros(N_im-2,D);      
    else
        V = V_old + dt/2*(F_R_old + F_R);
    end
    for im = 1:N_im-2
        normF_R = sqrt(sum(F_R.^2,2));
        P = V(im,:)*F_R(im,:)'/normF_R(im,:);
        V(im,:) = P*F_R(im,:)/normF_R(im,:);
        if P < 0
            V(im,:) = 0;
        end
    end
    R_new = R;
    R_new(2:(N_im-1),:) = R(2:(N_im-1),:) + dt*V + (dt^2)/2*F_R;
end
