%%%%% rotate_dimer.m
%%%%% Copyright: Olli-Pekka Koistinen, Aalto University, 12.5.2019
%%%%%
%%%%% This is an auxiliary function for the dimer method ('dimer.m').
%%%%% Given a rotation plane spanned by 'orient' and 'F_rot',
%%%%% the dimer is rotated one step towards its minimum energy orientation.
%%%%% The rotation angle is optimized according to the modified Newton method
%%%%% based on a finite difference estimation of the scalar rotational force
%%%%% along the direction of the rotation.
%%%%%
%%%%% Input:
%%%%%   R                     coordinates of the middle point of the dimer (1 x D)
%%%%%   orient                unit vector along the direction of the dimer (before rotation) (1 x D)
%%%%%   G01                   gradient at the middle point and image 1 of the dimer (before rotation) (2 x D)
%%%%%   F_rot                 rotational force acting on image 1 of the dimer (1 x D)
%%%%%   potential             potential and gradient function
%%%%%   dimer_sep             dimer separation (distance from the middle point of the dimer to the two images)
%%%%%   T_anglerot            convergence threshold for the rotation angle
%%%%%   estim_Curv            if 1, an estimate for the curvature along the direction of the dimer after the rotation is calculated
%%%%%   estim_G1              if 1, an estimate for the gradient at image 1 after the rotation is calculated
%%%%%   
%%%%% Output:
%%%%%   orient_new            unit vector along the direction of the dimer after optimal rotation (1 x D)
%%%%%   orient_rot_new        unit vector perpendicular to 'orient_new' within the rotation plane of this iteration (1 x D)
%%%%%   Curv                  estimate for the curvature along the direction of the dimer after the rotation (empty if estim_Curv = 0)
%%%%%   G1                    estimate for the gradient at image 1 after the rotation (empty if estim_G1 = 0)
%%%%%   R1_dtheta             coordinates of image 1 of the dimer after a small test rotation (1 x D)
%%%%%   E1_dtheta             energy at image 1 of the dimer after a small test rotation
%%%%%   G1_dtheta             gradient at image 1 of the dimer after a small test rotation (1 x D)

function [orient_new,orient_rot_new,Curv,G1,R1_dtheta,E1_dtheta,G1_dtheta] = rotate_dimer(R,orient,G01,F_rot,potential,dimer_sep,T_anglerot,estim_Curv,estim_G1)
    F_0 = sqrt(sum(F_rot.^2,2));
    C_0 = (-G01(1,:)+G01(2,:))*orient'/dimer_sep;
    dtheta = 0.5*atan(0.5*F_0/abs(C_0));
    if dtheta < T_anglerot
        orient_new = orient;
        orient_rot_new = 0;
        Curv = [];
        G1 = [];
        R1_dtheta = [];
        E1_dtheta = [];
        G1_dtheta = [];
    else
        orient_rot = F_rot/F_0;
        orient_dtheta = cos(dtheta)*orient + sin(dtheta)*orient_rot;
        orient_rot_dtheta = -sin(dtheta)*orient + cos(dtheta)*orient_rot;
	    R1_dtheta = R + dimer_sep*orient_dtheta;
        [E1_dtheta,G1_dtheta] = potential(R1_dtheta);
        F_rot_dtheta = force_rot([G01(1,:);G1_dtheta],orient_dtheta,dimer_sep);
        F_dtheta = F_rot_dtheta*orient_rot_dtheta';
        a1 = (F_dtheta-F_0*cos(2*dtheta))/(2*sin(2*dtheta));
        b1 = -0.5*F_0;
        angle_rot = 0.5*atan(b1/a1);
        if angle_rot < 0
            angle_rot = pi/2 + angle_rot;
        end
        orient_new = cos(angle_rot)*orient + sin(angle_rot)*orient_rot;
        orient_new = orient_new/sqrt(sum(orient_new.^2,2));
	    orient_rot_new = -sin(angle_rot)*orient + cos(angle_rot)*orient_rot;
        orient_rot_new = orient_rot_new/sqrt(sum(orient_rot_new.^2,2));
        if estim_Curv > 0
            Curv = C_0 + a1*(cos(2*angle_rot)-1) + b1*sin(2*angle_rot);
        else
            Curv = [];
        end
        if estim_G1 > 0
            G1 = sin(dtheta-angle_rot)/sin(dtheta)*G01(2,:)+sin(angle_rot)/sin(dtheta)*G1_dtheta+(1-cos(angle_rot)-sin(angle_rot)*tan(dtheta/2))*G01(1,:);
        else
            G1 = [];
        end
    end
end